/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb;

import java.io.Serializable;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;

/**
 * Service.
 * <p/>
 * Simple immutable compound value object associating a Service "Category" with a
 * Service "Name".
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class Service implements Serializable{

    /**
     * 
     */
    private static final long serialVersionUID = 1L;
    private String category;
    private String name;

    public Service(String category, String name) {
        AssertArgument.isNotNullAndNotEmpty(category, "category");
        AssertArgument.isNotNullAndNotEmpty(name, "name");
        this.category = category.trim();
        this.name = name.trim();
    }
    
    public boolean equals(Service service) {
        if (category.equals(service.getCategory()) && name.equals(service.getName())) {
            return true;
        } else {
            return false;
        }
    }

    public String getCategory() {
        return category;
    }

    public String getName() {
        return name;
    }

    public boolean equals(Object obj) {
        if(obj instanceof Service) {
            Service service = (Service) obj;
            if(service.category.equals(category) && service.name.equals(name)) {
                return true;
            }
        }

        return false;
    }

    public int hashCode() {
        return (category + name).hashCode();
    }

    public String toString() {
        return category + ":" + name;
    }

    /**
     * Construct a {@link Service} instance corresponding to the target Service
     * for the supplied Gateway configuration.
     *
     * @param config Gateway configuration.
     * @return Service instance corresponding to the target Service.
     * @throws ConfigurationException One of the Target Service config attributes was not
     * specified on the supplied configuration.
     */
    public static Service getGatewayTargetService(ConfigTree config) throws ConfigurationException {
        String category = config.getRequiredAttribute(ListenerTagNames.TARGET_SERVICE_CATEGORY_TAG);
        String name = config.getRequiredAttribute(ListenerTagNames.TARGET_SERVICE_NAME_TAG);

        return new Service(category, name);
    }
    
    public static Service getService(String category, String name) {
    	return (category != null && category.trim().length() > 0 && name != null && name.trim().length() > 0) ? new Service(category, name) : null;
    }
}
