/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.lifecycle;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.Map.Entry;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;


import org.apache.log4j.Logger;
import org.jboss.soa.esb.helpers.NamingContext;

/**
 * Lifecycle resource manager.
 * 
 * @author kevin
 */
public class LifecycleResourceManager
{
    /**
     * The logger for this class.
     */
    private static Logger logger = Logger.getLogger(LifecycleResourceManager.class);
    
    /**
     * Cleanup hook for cleaning non-esb lifecycle resources.
     */
    private static volatile Thread cleanupHook = new Thread() {
        public void run()
        {
            LifecycleResourceManager.getSingleton().cleanupAllResources() ;
        }
    } ;
    
    static
    {
        // Initialise the shutdown hook in case we are not in an ESB.
        Runtime.getRuntime().addShutdownHook(cleanupHook) ;
    }
    
    /**
     * The singleton instance.
     */
    private static final LifecycleResourceManager SINGLETON = new LifecycleResourceManager() ;
    
    /**
     * The default identity used for unassociated requests.
     */
    public static final String DEFAULT_IDENTITY = "ID-DEFAULT" ;
    
    /**
     * The mapping of associated deployments.
     */
    private final Map<ClassLoader, Set<String>> associatedDeployments = new HashMap<ClassLoader, Set<String>>() ;
    /**
     * The classloader identities.
     */
    private final Map<ClassLoader, String> identities = new HashMap<ClassLoader, String>() ;
    /**
     * The deployment lock.
     */
    private ReadWriteLock deploymentLock = new ReentrantReadWriteLock() ;

    /**
     * The class loader resource map.
     */
    private final Map<ClassLoader, Map<Integer, Set<LifecycleResource<?>>>> classLoaderResourceMap = new HashMap<ClassLoader, Map<Integer, Set<LifecycleResource<?>>>>() ;
    /**
     * The resource map.
     */
    private final Map<Integer, Set<LifecycleResource<?>>> resourceMap = new TreeMap<Integer, Set<LifecycleResource<?>>>() ;

    /**
     * The resource lock.
     */
    private Lock resourceLock = new ReentrantLock() ;
    
    /**
     * The identity counter.
     */
    private long identityCounter ;

    /**
     * Associate the current thread with a specified deployment.
     * @param deploymentName The deployment name.
     */
    public void associateDeployment(final String deploymentName)
    {
        final ClassLoader classLoader = Thread.currentThread().getContextClassLoader() ;
        String identity ;
        final Lock writeLock = deploymentLock.writeLock() ;
        writeLock.lock() ;
        try
        {
            identity = identities.get(classLoader) ;
            if (identity != null)
            {
                final Set<String> currentAssociations = associatedDeployments.get(classLoader) ;
                currentAssociations.add(deploymentName) ;
            }
            else
            {
                final Set<String> associations = new HashSet<String>() ;
                associations.add(deploymentName) ;
                associatedDeployments.put(classLoader, associations) ;
                
                identity = "ID-" + Long.toString(identityCounter++) ;
                identities.put(classLoader, identity) ;
            }
        }
        finally
        {
            writeLock.unlock() ;
        }
        
        if (logger.isDebugEnabled())
        {
            logger.debug("Associating deploymentName " + deploymentName + " with identity: " + identity) ;
        }
    }
    
    /**
     * Get the list of deployments associated with the context classloader.
     * @return The list of associated deployments or null if no association exists.
     */
    public String[] getAssociatedDeployments()
    {
        final ClassLoader classLoader = Thread.currentThread().getContextClassLoader() ;
        final Lock readLock = deploymentLock.readLock() ;
        readLock.lock() ;
        try
        {
            final Set<String> currentAssociations = associatedDeployments.get(classLoader) ;
            if (currentAssociations != null)
            {
                return (String[])currentAssociations.toArray(new String[currentAssociations.size()]) ;
            }
            else
            {
                return null ;
            }
        }
        finally
        {
            readLock.unlock() ;
        }
    }

    /**
     * Disassociate the current thread with a specified deployment.
     * @param deploymentName The deployment name.
     */
    public void disassociateDeployment(final String deploymentName)
    {
        final ClassLoader classLoader = Thread.currentThread().getContextClassLoader() ;
        final String identity = identities.get(classLoader) ;
        
        final boolean cleanContext ;
        final Lock writeLock = deploymentLock.writeLock() ;
        writeLock.lock() ;
        try
        {
            final Set<String> currentAssociations = associatedDeployments.get(classLoader) ;
            if (currentAssociations != null)
            {
                currentAssociations.remove(deploymentName) ;
                cleanContext = (currentAssociations.size() == 0) ;
                if (cleanContext)
                {
                    identities.remove(classLoader) ;
                    associatedDeployments.remove(classLoader) ;
                }
            }
            else
            {
                cleanContext = false ;
            }
        }
        finally
        {
            writeLock.unlock() ;
        }
        
        if (logger.isDebugEnabled())
        {
            logger.debug("Disassociating deploymentName " + deploymentName + " from identity: " + identity) ;
        }
        
        if (cleanContext)
        {
            cleanContextResources(identity) ;
        }
    }

    /**
     * The the current classloader identity.
     * @return The identity or null if disassociated.
     */
    public String getIdentity()
    {
        final String identity ;
        final Lock readLock = deploymentLock.readLock() ;
        readLock.lock() ;
        try
        {
            ClassLoader classLoader = Thread.currentThread().getContextClassLoader() ;
            String currentIdentity = null ;
            while(classLoader != null)
            {
                currentIdentity = identities.get(classLoader) ;
                if (currentIdentity != null)
                {
                    break ;
                }
                classLoader = classLoader.getParent() ;
            }
            identity = currentIdentity ;
        }
        finally
        {
            readLock.unlock() ;
        }
        return (null == identity ? DEFAULT_IDENTITY : identity) ;
    }

    /**
     * Register a lifecycle resource with the manager.
     * @param lifecycleResource The lifecycle resource.
     * @param classLoader The classloader associated with the factory.
     * @param priority The priority for cleaning.
     * 
     * The priority dictates the order in which the resources will be asked to cleanup,
     * lower priorities being asked to cleanup before higher priorities.
     */
    void registerResource(final LifecycleResource<?> lifecycleResource,
        final ClassLoader classLoader, final int priority)
    {
        resourceLock.lock() ;
        try
        {
            final Map<Integer, Set<LifecycleResource<?>>> classLoaderMap = classLoaderResourceMap.get(classLoader) ;
            if (classLoaderMap != null)
            {
                insertInto(classLoaderMap, priority, lifecycleResource) ;
            }
            else
            {
                final Map<Integer, Set<LifecycleResource<?>>> newMap = new TreeMap<Integer, Set<LifecycleResource<?>>>() ;
                classLoaderResourceMap.put(classLoader, newMap) ;
                insertInto(newMap, priority, lifecycleResource) ;
            }
            insertInto(resourceMap, priority, lifecycleResource) ;
        }
        finally
        {
            resourceLock.unlock() ;
        }
    }
    
    /**
     * Cleanup resources for the specific identity.
     * @param identity The identity being cleaned.
     */
    public void cleanContextResources(final String identity)
    {
        resourceLock.lock();
        try
        {
            // cleanup resources for this identity.
            final Iterator<Set<LifecycleResource<?>>> resourceSetIter = resourceMap.values().iterator() ;
            while(resourceSetIter.hasNext())
            {
                final Set<LifecycleResource<?>> resourceSet = resourceSetIter.next() ;
                final Iterator<LifecycleResource<?>> resourceIter = resourceSet.iterator() ;
                while(resourceIter.hasNext())
                {
                    final LifecycleResource<?> resource = resourceIter.next() ;
                    resource.cleanupResource(identity) ;
                }
            }
        }
        finally
        {
            resourceLock.unlock() ;
        }
        // NamingContext has been replaced by NamingContextPool and is now deprecated
        NamingContext.closeAllContexts() ;
    }
    
    /**
     * Destroy the class loader resources.
     */
    public void destroyResources()
    {
        resourceLock.lock();
        try
        {
            // destroy all resources associated with the class loader.
            final ClassLoader classLoader = Thread.currentThread().getContextClassLoader() ;
            final Map<Integer, Set<LifecycleResource<?>>> classLoaderMap = classLoaderResourceMap.remove(classLoader) ;
            if (classLoaderMap != null)
            {
                final Iterator<Entry<Integer, Set<LifecycleResource<?>>>> entryIter = classLoaderMap.entrySet().iterator() ;
                while(entryIter.hasNext())
                {
                    final Map.Entry<Integer, Set<LifecycleResource<?>>> entry = entryIter.next() ;
                    final Integer priority = entry.getKey() ;
                    final Set<LifecycleResource<?>> classLoaderResourceMapSet = entry.getValue() ;
                    
                    final Set<LifecycleResource<?>> resourceMapSet = resourceMap.get(priority) ;
                    final Iterator<LifecycleResource<?>> resourceIter = classLoaderResourceMapSet.iterator() ;
                    while(resourceIter.hasNext())
                    {
                        final LifecycleResource<?> resource = resourceIter.next() ;
                        resource.destroyResources() ;
                        resourceMapSet.remove(resource) ;
                    }
                    
                    if (resourceMapSet.size() == 0)
                    {
                        resourceMap.remove(priority) ;
                    }
                }
            }
        }
        finally
        {
            resourceLock.unlock() ;
        }
    }

    /**
     * Cleanup all remaining resources.
     */
    public void cleanupAllResources()
    {
        resourceLock.lock();
        try
        {
            // destroy resources.
            final Iterator<Set<LifecycleResource<?>>> resourceSetIter = resourceMap.values().iterator() ;
            while(resourceSetIter.hasNext())
            {
                final Set<LifecycleResource<?>> resourceSet = resourceSetIter.next() ;
                final Iterator<LifecycleResource<?>> resourceIter = resourceSet.iterator() ;
                while(resourceIter.hasNext())
                {
                    final LifecycleResource<?> resource = resourceIter.next() ;
                    
                    try
                    {
                	resource.destroyResources() ;
                    }
                    catch (final Throwable ex)
                    {
                	// log it, but continue on to other resources
                	
                	logger.warn("Caught exception "+ex+" during destroyResources.");
                    }
                }
            }
            
            resourceMap.clear() ;
            classLoaderResourceMap.clear();
        }
        finally
        {
            resourceLock.unlock() ;
        }
    }

    /**
     * Register a lifecycle resource using the lowest priority.
     * @param lifecycleResource The lifecycle resource.
     * @param classLoader The classloader associated with the factory.
     */
    void registerResource(final LifecycleResource<?> lifecycleResource,
        final ClassLoader classLoader)
    {
        registerResource(lifecycleResource, classLoader, Integer.MIN_VALUE) ;
    }
    
    /**
     * Get the singleton instance.
     * @return The singleton instance.
     */
    public static LifecycleResourceManager getSingleton()
    {
        return SINGLETON ;
    }
    
    /**
     * Deactivate the cleanup hook.
     */
    public static void deactivateHook()
    {
        final Thread hook = cleanupHook ;
        if (hook != null)
        {
            cleanupHook = null ;
            Runtime.getRuntime().removeShutdownHook(hook) ;
        }
    }
    
    /**
     * Insert the lifecycle resource into the priority map.
     * @param priorityMap The priority map.
     * @param priority The resource priority.
     * @param lifecycleResource The lifecycle resource.
     */
    private static void insertInto(final Map<Integer, Set<LifecycleResource<?>>> priorityMap,
        final int priority, final LifecycleResource<?> lifecycleResource)
    {
        final Integer key = Integer.valueOf(priority) ;
        final Set<LifecycleResource<?>> currentResourceSet = priorityMap.get(key) ;
        if (currentResourceSet != null)
        {
            currentResourceSet.add(lifecycleResource) ;
        }
        else
        {
            final Set<LifecycleResource<?>> newResourceSet = new HashSet<LifecycleResource<?>>() ;
            priorityMap.put(key, newResourceSet) ;
            newResourceSet.add(lifecycleResource) ;
        }
    }
}
