/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.listeners.config.model;

import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;

import org.apache.xmlbeans.XmlException;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.listeners.config.ServiceContract;
import org.jboss.soa.esb.listeners.config.ServicePublisher;
import org.jboss.soa.esb.listeners.config.WebserviceInfo;
import org.jboss.soa.esb.listeners.config.mappers.XMLBeansModel;
import org.jboss.soa.esb.listeners.config.xbeanmodel101.JbossesbDocument.Factory;
import org.w3c.dom.Document;

/**
 * Schema parser for jbossesb-1.0.1.xsd schema.
 * 
 * <a href='mailto:Kevin.Conner@jboss.com>Kevin Conner</a>
 */
class Model101SchemaParser implements SchemaParser
{
    /**
     * The namespace associated with this parser.
     */
    static final String NAMESPACE = "http://anonsvn.labs.jboss.com/labs/jbossesb/trunk/product/etc/schemas/xml/jbossesb-1.0.1.xsd" ;
    
    /**
     * The schema resource for validation.
     */
    static final String SCHEMA = "/jbossesb-1.0.1.xsd";

    
    /**
     * Obtain a model adapter from the specified stream.
     * @param reader The XML stream reader.
     * @return The model adapter
     * @throws XMLStreamException for errors during parsing.
     */
    public ModelAdapter parse(final XMLStreamReader reader)
        throws ModelException
    {
        try
        {
           return new Model101Adapter(new XMLBeansModel(Factory.parse(reader).getJbossesb())) ;
        }
        catch (final XmlException xmle)
        {
            throw new ModelException("Error during schema parsing", xmle) ;
        }
    }
    
    /**
     * Model adapter for the jbossesb-1.0.1.xsd schema.
     * 
     * <a href='mailto:Kevin.Conner@jboss.com>Kevin Conner</a>
     *
     */
    private static final class Model101Adapter implements ModelAdapter
    {
        /**
         * The model for jbossesb-1.0.1.xsd.
         */
        private final XMLBeansModel model ;
        
        /**
         * Create the jbossesb-1.0.1 model adapter.
         * @param model The jbossesb-1.0.1 model.
         */
        Model101Adapter(final XMLBeansModel model)
        {
            this.model = model ;
        }
        
        /**
         * Verify the schedule provider configuration.
         * @throws ConfigurationException
         */
        public void verifyScheduleProviderConfig()
            throws ConfigurationException
        {
            model.verifyScheduleProviderConfig() ;
        }

        /**
         * Get a map of service publishers provided by each service.
         * 
         * @return The map of service publishers, keyed by service.
         */
        public Map<Service, List<ServicePublisher>> getServicePublishers()
        {
            return model.getServicePublishers() ;
        }
        

        /**
         * Generate the ESB Aware configuration document.
         * @return The ESB aware configuration.
         * @throws ConfigurationException Error creating configuration. 
         */
        public Document generateESBAwareConfig()
            throws ConfigurationException
        {
            return model.generateESBAwareConfig() ;
        }

        /**
         * Generate the gateway configuration document.
         * @return The gateway configuration.
         * @throws ConfigurationException Error creating configuration. 
         */
        public Document generateGatewayConfig()
            throws ConfigurationException
        {
            return model.generateGatewayConfig() ;
        }
        
        /**
         * Get the list of actions in this deployment.
         * @return a list of actions or null if none present.
         */
        public Set<String> getActions()
        {
            return model.getActions() ;
        }
        
        /**
         * Get the list of services which require a webservice endpoint.
         * @return The list of services.
         */
        public List<WebserviceInfo> getWebserviceServices()
        {
            return model.getWebserviceServices() ;
        }
        
        /**
         * Get the list of service contracts.
         * @return The list of service contracts.
         */
        public List<ServiceContract> getServiceContracts()
        {
            return model.getServiceContracts() ;
        }

        /**
         * This feature in not implemented in the 1.0.1 version of the schema.
         * @return null Since this feature is not implemented.
         */
        public String getAuthDomain()
        {
            return null;
        }

        /**
         * This feature in not implemented in the 1.0.1 version of the schema.
         * @return null Since this feature is not implemented.
         */
        public String getAuthMethod()
        {
            return null;
        }
    }
}
