/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.notification;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.util.ClassUtil;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

/**
 * Abstract class to define expected behaviour of all NotificationTargets and
 * provide some common functionality to all of them
 * <p>
 * Description:
 * </p>
 * <p>
 * Heuristica - Buenos Aires - Argentina
 * </p>
 * 
 * @version 1.0
 */
public abstract class NotificationTarget
{
	/**
	 * Derived classes must implement this method to do what has to be done to
	 * trigger that specific type of notification event
	 * 
	 * @param message
	 *            Object - The toString() method of this object will be the
	 *            actual notification content
	 * @throws NotificationException -
	 *             invoke Exception.getMessage() at runtime for this object
	 * @see ConfigTree
	 */
	public abstract void sendNotification (Message message)
			throws NotificationException;

	private static final String NOTIF_PFX = NotificationTarget.class
			.getPackage().getName();

	public static final String PRM_NOTIF_CLASS = "class";

	/**
	 * Common object to hold details of object's information - Each derived
	 * class will hold it's own needs
	 */
	protected ConfigTree m_oParms;

	/**
	 * Instantiate an empty NotificationTarget object
	 */
	protected NotificationTarget ()
	{
	}

	/**
	 * Instantiate a NotificationTarget object with the information contained in
	 * &lt;arg 1&gt;
	 * 
	 * @param p_oP
	 *            ConfigTree - Holds details to instantiate this object
	 */
	protected NotificationTarget (ConfigTree p_oP)
	{
		m_oParms = p_oP;
	} // __________________________________

	/**
	 * A typical Factory.getInstance() method
	 * 
	 * @param p_oP
	 *            ConfigTree - Contents of this argument will determine the type
	 *            (derived class) of NotificationTarget returned
	 * @throws ConfigurationException -
	 *             Arg 1 does not contain a valid structure for currently
	 *             implemented NotificationTarget subclasses - invoke
	 *             Exception.getMessage() at runtime for details
	 * @return NotificationTarget - An object that instantiates the
	 *         NotificationTarget abstract class
	 */
	public static NotificationTarget fromParams (ConfigTree p_oP)
			throws ConfigurationException
	{
		String sClass = p_oP.getAttribute(PRM_NOTIF_CLASS);
		if (null == sClass)
			throw new IllegalArgumentException(
					"Missing '" + PRM_NOTIF_CLASS + "' attribute in parameters");
		Class oCls = null;
		try
		{
			oCls = ClassUtil.forName(NOTIF_PFX + "." + sClass, NotificationTarget.class);
		}
		catch (Exception e)
		{
			try
			{
				oCls = ClassUtil.forName(sClass, NotificationTarget.class);
			}
			catch (Exception e1)
			{/* oCls will be null */
			}
		}
		if (null == oCls)
			throw new ConfigurationException(
					"Invalid class <" + sClass + ">, or missing library");

		Constructor oCons = null;
		try
		{
			oCons = oCls.getConstructor(new Class[] { ConfigTree.class });
		}
		catch (Exception e)
		{
		}
		if (null == oCons)
			throw new ConfigurationException(
					"No valid constructor " + sClass + "(ConfigTree)");

		Object oRet = null;
		try
		{
			oRet = oCons.newInstance(new Object[] { p_oP });
        } catch (InvocationTargetException e) {
            Throwable cause = e.getCause();
            throw new ConfigurationException("Failed to create an instance of NotificationTarget class '" + sClass + "'.", (cause != null?cause:e));
        } catch (Exception e) {
            throw new ConfigurationException("Failed to create an instance of NotificationTarget class '" + sClass + "'.", e);
        }
        if (!(oRet instanceof NotificationTarget)) {
            throw new ConfigurationException(sClass + "  does not extend NotificationTarget");
        }

		return (NotificationTarget) oRet;
	} // __________________________________
} // ____________________________________________________________________________
