/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions.converters;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.TestBean;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.junit.Test;

/**
 * XStreamToObject unit tests.
 * @author daniel Marchant
 * @author Daniel Bevenius
 * @since Version 4.0
 */
public class XStreamToObjectUnitTest {
	
	@SuppressWarnings("unused")
	private Logger log = Logger.getLogger( XStreamToObjectUnitTest.class );
	
	@Test
    public void test_default() throws ActionProcessingException {
        XStreamConfigBuilder builder = new XStreamConfigBuilder();
        builder.incomingType(TestBean.class.getName());
        
        XStreamToObject xstreamToObject = new XStreamToObject(builder.build());

        Message oMsg = MessageFactory.getInstance().getMessage();

        String msg = "<" + TestBean.class.getSimpleName() + "> <name>Tom</name><phone>1234</phone></" + TestBean.class.getSimpleName() + ">";
        
        oMsg.getBody().add(msg);
        oMsg = xstreamToObject.process(oMsg);
       
        TestBean bean = (TestBean)oMsg.getBody().get();
  
        assertEquals("Tom", bean.getName());
        assertEquals("1234", bean.getPhone());
         
    }

	@Test
    public void test_with_package() throws ActionProcessingException {
        XStreamConfigBuilder builder = new XStreamConfigBuilder();
        builder.excludePackage(false);
        builder.incomingType(TestBean.class.getName());
        XStreamToObject xstreamToObject = new XStreamToObject(builder.build());

        Message oMsg = MessageFactory.getInstance().getMessage();
        
        String msg = "<" + TestBean.class.getName() + "> <name>Tom</name><phone>1234</phone></"+ TestBean.class.getName() +">";
        
        oMsg.getBody().add(msg);
        oMsg = xstreamToObject.process(oMsg);
       
        TestBean bean = (TestBean)oMsg.getBody().get();
  
        assertEquals("Tom", bean.getName());
        assertEquals("1234", bean.getPhone());
        
    }

	@Test
    public void test_with_alias() throws ActionProcessingException {
        XStreamConfigBuilder builder = new XStreamConfigBuilder();
        builder.classAlias("TomsClass");
        builder.excludePackage(false);
        builder.incomingType(TestBean.class.getName());
        
        XStreamToObject xstreamToObject = new XStreamToObject(builder.build());

        Message oMsg = MessageFactory.getInstance().getMessage();
        
        String msg = "<TomsClass><name>Tom</name><phone>1234</phone></TomsClass>";
        
        oMsg.getBody().add(msg);
        oMsg = xstreamToObject.process(oMsg);
       
        TestBean bean = (TestBean)oMsg.getBody().get();
  
        assertEquals("Tom", bean.getName());
        assertEquals("1234", bean.getPhone());
    }
	
	@Test
    public void fieldAliases() throws ActionProcessingException {
        XStreamConfigBuilder builder = new XStreamConfigBuilder();
        builder.classAlias("TomsClass");
        builder.excludePackage(false);
        builder.incomingType(TestBean.class.getName());
        builder.fieldAlias("firstName", TestBean.class, "name");
        builder.fieldAlias("phoneNumber", TestBean.class, "phone");
        
        XStreamToObject xstreamToObject = new XStreamToObject(builder.build());

        Message oMsg = MessageFactory.getInstance().getMessage();
        String msg = "<TomsClass><firstName>Tom</firstName><phoneNumber>1234</phoneNumber></TomsClass>";
        
        oMsg.getBody().add(msg);
        oMsg = xstreamToObject.process(oMsg);
       
        TestBean bean = (TestBean) oMsg.getBody().get();
  
        assertEquals("Tom", bean.getName());
        assertEquals("1234", bean.getPhone());
    }
	
	@Test
	public void implicitCollections() throws ActionProcessingException
	{
        XStreamConfigBuilder builder = new XStreamConfigBuilder();
        builder.classAlias("TomsClass");
        builder.incomingType(TestBean.class.getName());
        builder.alias("address", String.class.getName());
        builder.implicitCollection(List.class.getName(), "addresses", "address", String.class.getName());
        
        XStreamToObject xstreamToObject = new XStreamToObject(builder.build());
        
        Message msg = MessageFactory.getInstance().getMessage();
        String xml = "<TomsClass><name>Tom</name><phone>1234</phone><addresses><address>street1</address><address>street2</address></addresses></TomsClass>";
        msg.getBody().add(xml);
        
        msg = xstreamToObject.process(msg);
        
        TestBean bean = (TestBean) msg.getBody().get();
        System.out.println(bean);
        List<String> addresses = bean.getAddresses();
        assertEquals(2, addresses.size());
        assertEquals("street1", addresses.get(0));
        assertEquals("street2", addresses.get(1));
	}
	
	
	@Test
    public void getAliases()
    {
        XStreamConfigBuilder builder = new XStreamConfigBuilder();
        builder.classAlias("TomsClass");
        builder.excludePackage(false);
        builder.incomingType(TestBean.class.getName());
        builder.alias("aliasName1", TestBean.class.getName());
        builder.alias("aliasName2", TestBean.class.getName());
        
    	ConfigTree configTree = builder.build();
        XStreamToObject xstreamToObject = new XStreamToObject( configTree );
        
        Map<String, String> aliases = xstreamToObject.getAliases( configTree, "alias" );
        assertAliases( aliases );
    }
    
	@Test
    public void test_with_multiple_aliases() throws ActionProcessingException {
    	
        XStreamToObject xstreamToObject = createXStreamObject( "TomsClass", false, TestBean.class, null );

        Message oMsg = MessageFactory.getInstance().getMessage();
        
        String msg = "<TomsClass><name>Tom</name><phone>1234</phone></TomsClass>";
        
        oMsg.getBody().add(msg);
        oMsg = xstreamToObject.process(oMsg);
       
        TestBean bean = (TestBean)oMsg.getBody().get();
  
        assertEquals("Tom", bean.getName());
        assertEquals("1234", bean.getPhone());
    }
    
	@Test
    public void test_setRootNodeNegative() throws ActionProcessingException
    {
        XStreamToObject xstreamToObject = createXStreamObject( "TomsClass", false, TestBean.class, null );
        assertEquals ( null , xstreamToObject.getRootNodeName() );
    }
    
	@Test
    public void fromXmlToObject_with_rootNode() throws ActionProcessingException
    {
    	String rootNodeName = "/someNode/TomsClass";
        XStreamToObject xstreamToObject = createXStreamObject( "TomsClass", false, TestBean.class, rootNodeName );
        
        String xml = "<someNode><TomsClass><name>Tom</name><phone>1234</phone></TomsClass></someNode>";
        Object obj = new TestBean();
        
        obj = xstreamToObject.fromXmlToObject( xml, obj );
        
        assertTrue( obj instanceof TestBean );
        assertEquals("1234", ((TestBean)obj).getPhone());
    }
    
	@Test
    public void fromXmlToObject_with_rootNode_as_second_child_node() throws ActionProcessingException
    {
    	String rootNodeName = "/someNode/TomsClass2";
        XStreamToObject xstreamToObject = createXStreamObject( "TomsClass2", false,TestBean2.class, rootNodeName );
        
        String xml = "<someNode><TomsClass><name>Tom</name><phone>1234</phone></TomsClass><TomsClass2><name>Daniel</name></TomsClass2></someNode>";
        Object obj = new TestBean2();
        
        obj = xstreamToObject.fromXmlToObject( xml, obj );
        
        assertTrue( obj instanceof TestBean2 );
        assertEquals("Daniel", ((TestBean2)obj).getName());
    }
	
	@Test
    public void fromXmlToObject_with_rootNode_as_second_child_node_and_sibling() throws ActionProcessingException
    {
    	String rootNodeName = "/someNode/TomsClass2[2]";
        XStreamToObject xstreamToObject = createXStreamObject( "TomsClass2", false,TestBean2.class, rootNodeName );
        
        String xml = "<someNode><TomsClass><name>Tom</name><phone>1234</phone></TomsClass><TomsClass2><name>Daniel</name></TomsClass2><TomsClass2><name>Bevenius</name></TomsClass2></someNode>";
        Object obj = new TestBean2();
        
        obj = xstreamToObject.fromXmlToObject( xml, obj );
        
        assertTrue( obj instanceof TestBean2 );
        assertEquals("Bevenius", ((TestBean2)obj).getName());
    }
	
	@Test
    public void fromXmlToObject_with_collection() throws ActionProcessingException
    {
    	String rootNodeName = "/someNode/list";
    	
        XStreamConfigBuilder builder = new XStreamConfigBuilder();
        builder.classAlias("list");
        builder.excludePackage(true);
        builder.incomingType(ArrayList.class.getName());
        builder.rootNodeName(rootNodeName);
        builder.alias("TomsClass", TestBean.class.getName());
    	
        XStreamToObject xstreamToObject = new XStreamToObject( builder.build() );
        
        String xml = "<someNode><list><TomsClass><name>Tom</name></TomsClass><TomsClass><name>Daniel</name></TomsClass></list></someNode>";
        
        Object obj = new ArrayList<TestBean>();
        obj = xstreamToObject.fromXmlToObject( xml, obj );
        
        assertTrue( obj instanceof ArrayList );
        ArrayList<TestBean> nodes = (ArrayList<TestBean>) obj;
        assertEquals( 2, nodes.size() );
        
    }
	
	@Test
    public void fromXmlToObject_with_attributeAliases() throws ActionProcessingException
    {
    	String rootNodeName = "/someNode/list";
    	
        XStreamConfigBuilder builder = new XStreamConfigBuilder();
        builder.classAlias("list");
        builder.excludePackage(true);
        builder.incomingType(ArrayList.class.getName());
        builder.rootNodeName(rootNodeName);
        builder.alias("TomsClass", TestBean.class.getName());
        builder.attributeAlias("initials", String.class.getName());
        
        XStreamToObject xstreamToObject = new XStreamToObject( builder.build() );
        
        String xml = "<someNode><list><TomsClass initials='TF'><name>Tom</name></TomsClass><TomsClass initials='DB'><name>Daniel</name></TomsClass></list></someNode>";
        
        Object obj = new ArrayList<TestBean>();
        obj = xstreamToObject.fromXmlToObject( xml, obj );
        
        assertTrue( obj instanceof ArrayList );
        ArrayList<TestBean> nodes = (ArrayList<TestBean>) obj;
        assertEquals( 2, nodes.size() );
        
        assertEquals( "TF", nodes.get(0).getInitials() );
        assertEquals( "DB", nodes.get(1).getInitials() );
    }
	
	@Test
    public void fromXmlToObject_with_enum() throws ActionProcessingException
    {
    	String rootNodeName = "/someNode/list";
    	
        XStreamConfigBuilder builder = new XStreamConfigBuilder();
        builder.classAlias("list");
        builder.excludePackage(true);
        builder.incomingType(ArrayList.class.getName());
        builder.rootNodeName(rootNodeName);
        builder.alias("TomsClass", TestBean.class.getName());
        builder.alias("gender", TestBean.GENDER.class.getName());
        
        XStreamToObject xstreamToObject = new XStreamToObject( builder.build() );
        
        String xml = "<someNode><list><TomsClass><gender>MALE</gender></TomsClass><TomsClass><gender>FEMALE</gender></TomsClass></list></someNode>";
        
        Object obj = new ArrayList<TestBean>();
        obj = xstreamToObject.fromXmlToObject( xml, obj );
        
        assertTrue( obj instanceof ArrayList );
        ArrayList<TestBean> nodes = (ArrayList<TestBean>) obj;
        assertEquals( 2, nodes.size() );
        
        TestBean person1 = nodes.get( 0 );
        assertEquals( TestBean.GENDER.MALE, person1.getGender() );
        
        TestBean person2 = nodes.get( 1 );
        assertEquals( TestBean.GENDER.FEMALE, person2.getGender() );
    }
	
	@Test
    public void fromXmlToObject_with_collections() throws ActionProcessingException
    {
    	String rootNodeName = "/someNode/list";
        XStreamConfigBuilder builder = new XStreamConfigBuilder();
        builder.classAlias("list");
        builder.excludePackage(true);
        builder.incomingType(ArrayList.class.getName());
        builder.rootNodeName(rootNodeName);
        builder.alias("Persons", ArrayList.class.getName());
        builder.alias("TomsClass", TestBean.class.getName());
    	
        XStreamToObject xstreamToObject = new XStreamToObject( builder.build() );
        
        String xml = "<someNode><list><Persons><TomsClass><list></list></TomsClass><TomsClass></TomsClass></Persons></list></someNode>";
        
        Object obj = new ArrayList<TestBean>();
        obj = xstreamToObject.fromXmlToObject( xml, obj );
        
        assertTrue( obj instanceof ArrayList );
        ArrayList list = (ArrayList) obj;
        ArrayList subList = (ArrayList)list.get( 0 );
        assertEquals( 2, subList.size() );
    }
	
	@Test
    public void fromXmlToObject_with_converter() throws ActionProcessingException
    {
        XStreamConfigBuilder builder = new XStreamConfigBuilder().classAlias("comment").excludePackage(true).incomingType(Comment.class.getName());
    	ConfigTree configTree = builder.build();
    	
        ConfigTree converter = new ConfigTree( "converter", configTree);
    	converter.setAttribute( "class", CommentConverter.class.getName());
    	
        XStreamToObject xstreamToObject = new XStreamToObject( configTree );
        
        final String expectedComment = "Just a comment";
        String xml = "<comment>" + expectedComment + "</comment>";
        
        Object obj = new Comment();
        obj = xstreamToObject.fromXmlToObject( xml, obj );
        assertTrue( obj instanceof Comment );
        assertEquals( expectedComment, ((Comment)obj).getComment());
    }
    
    private XStreamToObject createXStreamObject( String classAlias, boolean exludePackage, Class<?> incomingType, String rootNodeName)
    {
        XStreamConfigBuilder builder = new XStreamConfigBuilder().classAlias(classAlias).excludePackage(exludePackage).incomingType(incomingType.getName()).rootNodeName(rootNodeName);
    	ConfigTree configTree = builder.build();
        return new XStreamToObject( configTree );
    }
    
    private void assertAliases( Map<String, String> aliases )
    {
    	assertNotNull( "Aliases map should not have been null!",  aliases );
        assertEquals( TestBean.class.getName(), aliases.get( "aliasName1" ) );
        assertEquals( TestBean.class.getName(), aliases.get( "aliasName2" ) );
    }
    
    /**
     * An abstract class for testing
     */
    private static abstract class AbstractTestBean {     }

    /**
     * Just a bean for testing
     */
	private static class TestBean2 extends AbstractTestBean {
		private String name;

		public String getName()
		{
			return name;
		}

		public void setName( String name )
		{
			this.name = name;
		}
	}
	
	/**
	 * Just here to get Ant to find annotated test.
	 */
	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter( XStreamToObjectUnitTest.class);
	}
	
}	
	