/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.listeners.config;

import junit.framework.TestCase;
import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.testutils.FileUtil;

import java.io.File;
import java.io.IOException;

/**
 * Generator unit tests.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class GeneratorUnitTest extends TestCase {

	private Logger log = Logger.getLogger( GeneratorUnitTest.class );

	private File outdir = new File("test-outdir");
	private File gatewayConfig;
	private File awareConfig;

	public void test_constructor() throws ConfigurationException, IOException {
		try {
			new Generator(null);
			fail("Expected IllegalArgumentException.");
		} catch (IllegalArgumentException e) {
			// OK
		}
	}

	public void test_generate_badargs() throws ConfigurationException, IOException {
		test_generate_badargs(null, "null 'outdir' arg in method call", "jbossesb_config_01.xml");
		test_generate_badargs(new File("build.xml"), "'outdir' exists, but is not a directory", "jbossesb_config_01.xml");
	}

    public void test_generate_badargs_110() throws ConfigurationException, IOException {
        test_generate_badargs(null, "null 'outdir' arg in method call", "jbossesb_config_01.110.xml");
        test_generate_badargs(new File("build.xml"), "'outdir' exists, but is not a directory", "jbossesb_config_01.110.xml");
    }

    public void test_generate_badargs_120() throws ConfigurationException, IOException {
        test_generate_badargs(null, "null 'outdir' arg in method call", "jbossesb_config_01.120.xml");
        test_generate_badargs(new File("build.xml"), "'outdir' exists, but is not a directory", "jbossesb_config_01.120.xml");
    }

	public void test_generate_files() throws Exception {
        System.setProperty("jboss.esb.invm.scope.default", "NONE");
		execute_test_generate_files("jbossesb_config_01.xml");
		execute_test_generate_files("jbossesb_config_02.xml");
	}

    public void test_generate_files110() throws Exception {
        System.setProperty("jboss.esb.invm.scope.default", "NONE");
        execute_test_generate_files110("jbossesb_config_01.110.xml");
        execute_test_generate_files110("jbossesb_config_02.110.xml");
    }

    public void test_generate_files120() throws Exception {
        System.setProperty("jboss.esb.invm.scope.default", "NONE");
        execute_test_generate_files120("jbossesb_config_01.120.xml");
        execute_test_generate_files120("jbossesb_config_02.120.xml");
    }
    
	private void execute_test_generate_files(String filename) throws Exception {
	    execute_test_generate_files(filename, "jbossesb_config_01_gateways.xml", "jbossesb_config_01_esbaware.xml");
	}
	
	private void execute_test_generate_files110(String filename) throws Exception {
	    execute_test_generate_files(filename, "jbossesb_config_110_gateways.xml", "jbossesb_config_110_esbaware.xml");
	}
	
	private void execute_test_generate_files120(String filename) throws Exception {
	    execute_test_generate_files(filename, "jbossesb_config_120_gateways.xml", "jbossesb_config_120_esbaware.xml");
	}

	private void execute_test_generate_files(String filename, String esbGatewayConfig, String esbAwareConfig) throws Exception {
		Generator generator = new Generator(getClass().getResourceAsStream(filename));
		String expectedGatewayConfig = new String(StreamUtils.readStream(getClass().getResourceAsStream(esbGatewayConfig))).trim();
		String expectedESBAwareConfig = new String(StreamUtils.readStream(getClass().getResourceAsStream(esbAwareConfig))).trim();
		String actualGatewayConfig;
		String actualESBAwareConfig;

		expectedGatewayConfig = removeEOL(expectedGatewayConfig);
		expectedESBAwareConfig = removeEOL(expectedESBAwareConfig);

		generator.generate(outdir);

		gatewayConfig = new File(outdir, Generator.ESB_CONFIG_GATEWAY_XML_FILE);
		assertTrue(gatewayConfig.exists());
		actualGatewayConfig = FileUtil.readTextFile(gatewayConfig);
		log.debug("actual  [" + removeEOL(actualGatewayConfig) + "]");
        log.debug("expected[" + expectedGatewayConfig + "]");
		assertTrue("Gateway Configuration", XMLHelper.compareXMLContent(expectedGatewayConfig, actualGatewayConfig));

		awareConfig = new File(outdir, Generator.ESB_CONFIG_XML_FILE);
		assertTrue(awareConfig.exists());
		actualESBAwareConfig = FileUtil.readTextFile(awareConfig);
		log.debug("[" + actualESBAwareConfig + "]");
		actualESBAwareConfig = removeEOL(actualESBAwareConfig);
		boolean isIndentical = expectedESBAwareConfig.equals(actualESBAwareConfig);
		log.debug("Strings identical=" + isIndentical);
		assertTrue("ESBAware Configuration", XMLHelper.compareXMLContent(expectedESBAwareConfig, actualESBAwareConfig));
	}
	

	public void test_bad_base_listener_busref() throws ConfigurationException, IOException {
		Generator generator = new Generator(getClass().getResourceAsStream("jbossesb_config_03.xml"));

		try {
			generator.generate(outdir);
			fail("Expected ConfigurationException");
		} catch(ConfigurationException e) {
			if(!e.getMessage().equals("The base Listener config [JMS-ESBListener] must reference a base Bus config type (<bus>).")) {
				fail("Wrong exception");
			}
		}
	}

    public void test_bad_base_listener_busref_110() throws ConfigurationException, IOException {
        Generator generator = new Generator(getClass().getResourceAsStream("jbossesb_config_03.110.xml"));

        try {
            generator.generate(outdir);
            fail("Expected ConfigurationException");
        } catch(ConfigurationException e) {
            if(!e.getMessage().equals("The base Listener config [JMS-ESBListener] must reference a base Bus config type (<bus>).")) {
                fail("Wrong exception");
            }
        }
    }

    public void test_bad_base_listener_busref_120() throws ConfigurationException, IOException {
        Generator generator = new Generator(getClass().getResourceAsStream("jbossesb_config_03.120.xml"));

        try {
            generator.generate(outdir);
            fail("Expected ConfigurationException");
        } catch(ConfigurationException e) {
            if(!e.getMessage().equals("The base Listener config [JMS-ESBListener] must reference a base Bus config type (<bus>).")) {
                fail("Wrong exception");
            }
        }
    }

	public static String removeEOL(String string) {
		StringBuffer stringBuf = new StringBuffer(string.length());

		for(int i = 0; i < string.length(); i++) {
			char character = string.charAt(i);

			if(character != '\r' && character != '\n') {
				stringBuf.append(character);
			}
		}

		return stringBuf.toString();
	}

	private void test_generate_badargs(File outdir, String expectedMessage, final String config) throws ConfigurationException, IOException {
		Generator generator = new Generator(getClass().getResourceAsStream(config));

		try {
			generator.generate(outdir);
			fail("Expected IllegalArgumentException.");
		} catch(IllegalArgumentException e) {
			// Expected
			if(!e.getMessage().startsWith(expectedMessage)) {
				fail("Unexpected exception message.  Expected message to start with [" + expectedMessage + "].  Message was [" + e.getMessage() + "].");
			}
		}
	}
	
	/* (non-Javadoc)
	 * @see junit.framework.TestCase#tearDown()
	 */
	@Override
	protected void tearDown() throws Exception {
		if(awareConfig != null && awareConfig.exists()) {
			awareConfig.delete();
		}
		if(gatewayConfig != null && gatewayConfig.exists()) {
			gatewayConfig.delete();
		}
		if(outdir.exists()) {
			outdir.delete();
		}
	}
}
