/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.jbpm.integration.job;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.soa.esb.services.jbpm.integration.msg.RetryExecutor;
import org.jbpm.JbpmContext;
import org.jbpm.command.Command;
import org.jbpm.job.Timer;

/**
 * Command for processing timer executions, based on the jBPM code.
 */
public class ExecuteTimerCommand implements Command {

    /**
     * The timer id.
     */
    private final long timerId;
    /**
     * The redelivered flag.
     */
    private final boolean redelivered ;

    private static final long serialVersionUID = 1L;

    /**
     * Create the timer command with a specified timer id.
     * @param timerId The associated timer id.
     * @param redelivered The redelivered flag.
     */
    public ExecuteTimerCommand(long timerId, final boolean redelivered) {
      this.timerId = timerId;
      this.redelivered = redelivered ;
    }

    /**
     * Execute the command.
     * @param jbpmContext The jBPM context associated with the execution.
     * @throws Exception for errors during execution.
     */
    public Object execute(JbpmContext jbpmContext) throws Exception {
      Timer timer = jbpmContext.getJobSession().loadTimer(timerId);
      if (JobUtil.isDeleted(timer)) {
        if (log.isDebugEnabled()) {
          log.debug("timer " + timerId + " was deleted");
        }
        return null;
      }
      if (timer.isSuspended()) {
        if (log.isDebugEnabled()) {
          log.debug(timer + " is suspended");
        }
        RetryExecutor.handleSuspendedTimer(timer) ;
        return null ;
      }
      if (redelivered) {
          if (timer.getRetries() > 0) {
              timer.setRetries(timer.getRetries() - 1);
              if (log.isDebugEnabled()) {
                log.debug("Rescheduling timer " + timer.getId() + ", " + timer.getDueDate());
              }
              jbpmContext.getServices().getSchedulerService().createTimer(timer);
          } else {
              log.error("Timer retry count exceeded for timer " + timer.getId());
          }
          return null ;
      }
      timer.setLockOwner(getClass().getName()); // prevent others from removing timer
      log.debug("executing " + timer);
      try {
        if (timer.execute(jbpmContext)) {
          jbpmContext.getServices().getSchedulerService().deleteTimer(timer);
        } else if (timer.getRepeat() != null) {
            if (log.isDebugEnabled()) {
                log.debug("scheduling timer for repeat on " + timer.getDueDate());
            }
            jbpmContext.getServices().getSchedulerService().createTimer(timer);
        }
      }
      catch (Exception e) {
        if (log.isDebugEnabled()) {
          log.debug("exception while executing " + timer, e);
        }
        jbpmContext.setRollbackOnly();
      }
      return null;
    }

    /**
     * The log associated with this class.
     */
    private static final Log log = LogFactory.getLog(ExecuteTimerCommand.class);
}
