/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.graph.node;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.net.URI;
import java.net.URL;
import java.util.regex.Pattern;

import org.hibernate.LockMode;

import org.jbpm.JbpmConfiguration;
import org.jbpm.JbpmException;
import org.jbpm.db.AbstractDbTestCase;
import org.jbpm.graph.def.ProcessDefinition;

/**
 * Verifies the join node can be persisted correctly.
 * 
 * @author Alejandro Guizar
 */
public class JoinDbTest extends AbstractDbTestCase {

  protected JbpmConfiguration getJbpmConfiguration() {
    if (jbpmConfiguration == null) {
      URL cfgResource = getClass().getClassLoader().getResource("hibernate.cfg.xml");
      try {
        OutputStream outStream = new FileOutputStream(new File(URI.create(cfgResource.toString())
            .resolve("hibernate.join.cfg.xml")));
        sed("Join\\.hbm\\.xml", "Join.lock.hbm.xml", cfgResource.openStream(), outStream);
        outStream.close();
      }
      catch (IOException e) {
        throw new JbpmException("could not edit hibernate configuration", e);
      }
      jbpmConfiguration = JbpmConfiguration.parseXmlString("<jbpm-configuration>"
          + "  <string name='resource.hibernate.cfg.xml' value='hibernate.join.cfg.xml' />"
          + "</jbpm-configuration>");
    }
    return jbpmConfiguration;
  }

  protected void tearDown() throws Exception {
    super.tearDown();
    jbpmConfiguration.close();
  }

  private static void sed(String regex, String replacement, InputStream inStream,
      OutputStream outStream) throws IOException {
    BufferedReader reader = new BufferedReader(new InputStreamReader(inStream));
    PrintWriter writer = new PrintWriter(new OutputStreamWriter(outStream));
    Pattern pattern = Pattern.compile(regex);
    for (String inLine; (inLine = reader.readLine()) != null;) {
      String outLine = pattern.matcher(inLine).replaceAll(replacement);
      writer.println(outLine);
    }
    reader.close();
    writer.flush();
  }

  public void testParentLockMode() {
    ProcessDefinition processDefinition = ProcessDefinition.parseXmlString("<process-definition name='"
        + getName()
        + "'>"
        + "  <join name='read' lock='READ' />"
        + "  <join name='nowait' lock='UPGRADE_NOWAIT' />"
        + "  <join name='upgrade' lock='pessimistic' />"
        + "</process-definition>");
    jbpmContext.deployProcessDefinition(processDefinition);

    newTransaction();
    try {
      processDefinition = graphSession.findLatestProcessDefinition(getName());
      Join join = (Join) processDefinition.getNode("read");
      assertEquals(LockMode.READ.toString(), join.getParentLockMode());
      join = (Join) processDefinition.getNode("nowait");
      assertEquals(LockMode.UPGRADE_NOWAIT.toString(), join.getParentLockMode());
      join = (Join) processDefinition.getNode("upgrade");
      assertEquals(LockMode.UPGRADE.toString(), join.getParentLockMode());
    }
    finally {
      graphSession.deleteProcessDefinition(processDefinition.getId());
    }
  }
}
