package org.jbpm.jbpm1755;

import org.hibernate.LockMode;

import org.jbpm.db.AbstractDbTestCase;
import org.jbpm.graph.def.Event;
import org.jbpm.graph.def.EventCallback;
import org.jbpm.graph.def.ProcessDefinition;
import org.jbpm.graph.exe.ProcessInstance;
import org.jbpm.graph.node.Join;

/**
 * Allow process author to set the parent token lock mode in the join token.
 * https://jira.jboss.org/jira/browse/JBPM-1755
 * 
 * @author Alejandro Guizar
 */
public class JBPM1755Test extends AbstractDbTestCase {

  private ProcessDefinition processDefinition;

  private static final int processInstanceCount = 5;

  protected void setUp() throws Exception {
    super.setUp();
    processDefinition = ProcessDefinition.parseXmlResource("org/jbpm/jbpm1755/parallelprocess.xml");
  }

  protected void tearDown() throws Exception {
    graphSession.deleteProcessDefinition(processDefinition.getId());
    EventCallback.clear();
    super.tearDown();
  }

  public void testReadLock() {
    launchProcessInstances(LockMode.READ);
  }

  public void testUpgradeLock() {
    launchProcessInstances(LockMode.UPGRADE);
  }

  public void testForceLock() {
    launchProcessInstances(LockMode.FORCE);
  }

  private void launchProcessInstances(LockMode lockMode) {
    Join join = (Join) processDefinition.getNode("join1");
    join.setParentLockMode(lockMode.toString());
    jbpmContext.deployProcessDefinition(processDefinition);

    long[] processInstanceIds = new long[processInstanceCount];
    for (int i = 0; i < processInstanceCount; i++) {
      newTransaction();
      ProcessInstance processInstance = new ProcessInstance(processDefinition);
      processInstanceIds[i] = processInstance.getId();
      processInstance.getContextInstance().setVariable("eventCallback", new EventCallback());
      processInstance.signal();
      jbpmContext.save(processInstance);
    }

    commitAndCloseSession();
    try {
      startJobExecutor();
      EventCallback.waitForEvent(processInstanceCount, Event.EVENTTYPE_PROCESS_END);
    }
    finally {
      stopJobExecutor();
      beginSessionTransaction();
    }

    for (int i = 0; i < processInstanceCount; i++) {
      long processInstanceId = processInstanceIds[i];
      assertTrue("expected process instance " + processInstanceId + " to have ended",
          jbpmContext.loadProcessInstance(processInstanceId).hasEnded());
    }
  }

}
