/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.perf;

import java.text.NumberFormat;

import org.jbpm.db.AbstractDbTestCase;
import org.jbpm.graph.def.ProcessDefinition;
import org.jbpm.graph.exe.ProcessInstance;

/**
 * This test creates a number of process instances. Every instance has a call to an ActionHandler.
 * 
 * @see <a href="https://jira.jboss.org/jira/browse/JBPM-2043">JBPM-2043</a>
 * @author mvecera@redhat.com
 * @author pmacik@redhat.com
 * @author aguizar@redhat.com
 * @since 18-Feb-2009
 */
public class SimplePerformanceTest extends AbstractDbTestCase {

  private static final int WARMUP_INSTANCES = 100;
  private static final int MEASURED_INSTANCES = 1000;
  private static final long TIMEOUT = 2 * 60 * 1000;

  private ProcessDefinition processDefinition;

  protected void setUp() throws Exception {
    super.setUp();

    processDefinition = ProcessDefinition.parseXmlString("<process-definition name='perf'>"
        + "  <start-state name='start'>"
        + "    <transition to='end'/>"
        + "  </start-state>"
        + "  <end-state async='true' name='end'/>"
        + "</process-definition>");
    jbpmContext.deployProcessDefinition(processDefinition);
  }

  protected void tearDown() throws Exception {
    jbpmContext.getGraphSession().deleteProcessDefinition(processDefinition.getId());
    super.tearDown();
  }

  public void testAsyncCall() {
    long firstTime = System.currentTimeMillis();
    launchProcessInstances(WARMUP_INSTANCES);
    processJobs(TIMEOUT);

    long secondTime = System.currentTimeMillis();
    launchProcessInstances(MEASURED_INSTANCES);
    processJobs((secondTime - firstTime) * (MEASURED_INSTANCES / WARMUP_INSTANCES + 1));

    long duration = System.currentTimeMillis() - secondTime;
    NumberFormat format = NumberFormat.getNumberInstance();
    format.setMaximumFractionDigits(1);
    System.out.println("==> Processed " +
        format.format(1000D * MEASURED_INSTANCES / duration) +
        " instances per second <==");
  }

  private void launchProcessInstances(int count) {
    for (int i = 0; i < count; i++) {
      newTransaction();
      ProcessInstance processInstance = new ProcessInstance(processDefinition);
      processInstance.signal();
      jbpmContext.save(processInstance);
    }
  }
}
