package org.jboss.internal.soa.esb.message.format.serialized;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.message.Context;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

/**
 * This class is not thread safe and will be duplicated during InVM operations.
 */
public class ContextImpl implements Context, java.io.Serializable
{
    /**
     * The context variables.
     */
    private Map<String, Serializable> context = new HashMap<String, Serializable>() ;
    
    /**
     * Default constructor for the context.
     */ 
    public ContextImpl()
    {
    }
    
    /**
     * Create a copy of the context.
     * @param copy The context to copy.
     */
    ContextImpl(final ContextImpl copy)
    {
        context.putAll(copy.context) ;
    }
    
	private static final long serialVersionUID = 0x0;
	
	public String toString ()
	{
		return "context: " + context ;
	}
	
    /**
     * Set the context value, replacing existing value if present.
     * @param key The context key.
     * @param value The context value.
     * @return the previous value of the context or null if not set.
     */
    public Object setContext(final String key, final Object value)
    {
        AssertArgument.isNotNull(key, "key");
        AssertArgument.isNotNull(value, "value");
        
        if (value instanceof Serializable)
        {
            return context.put(key, (Serializable)value) ;
        }
        else
        {
            throw new IllegalArgumentException("value must be serializable");
        }
    }
    
    /**
     * Retrieves the context value.
     * @param key The context key.
     * @return The value or null if not present.
     */
    public Object getContext(final String key)
    {
        return context.get(key) ;
    }
    
    /**
     * Remove the context value.
     * @param key The context key.
     * @return The value of the context, or null if not present.
     */
    public Object removeContext(final String key)
    {
        return context.remove(key) ;
    }
    
    /**
     * Get the keys in the context.
     * @return the context keys.
     */
    public Set<String> getContextKeys()
    {
        return Collections.unmodifiableSet(context.keySet()) ;
    }
    
    /**
     * Clear the context.
     */
    public void clearContext()
    {
        context = new HashMap<String, Serializable>() ;
    }
    
    /**
     * Deserialise the object, checking for old versions.
     * @param ois The object input stream.
     * @throws IOException For IO exceptions during object deserialisation.
     * @throws ClassNotFoundException If dependent classes cannot be found.
     */
    private void readObject(final ObjectInputStream ois)
        throws IOException, ClassNotFoundException
    {
        ois.defaultReadObject() ;
        if (context == null)
        {
            context = new HashMap<String, Serializable>() ;
        }
    }
}
