/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.internal.soa.esb.services.registry;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import javax.xml.registry.BulkResponse;
import javax.xml.registry.BusinessLifeCycleManager;
import javax.xml.registry.BusinessQueryManager;
import javax.xml.registry.Connection;
import javax.xml.registry.FindQualifier;
import javax.xml.registry.JAXRException;
import javax.xml.registry.JAXRResponse;
import javax.xml.registry.RegistryService;
import javax.xml.registry.infomodel.Classification;
import javax.xml.registry.infomodel.ClassificationScheme;
import javax.xml.registry.infomodel.Concept;
import javax.xml.registry.infomodel.EmailAddress;
import javax.xml.registry.infomodel.Key;
import javax.xml.registry.infomodel.Organization;
import javax.xml.registry.infomodel.PersonName;
import javax.xml.registry.infomodel.PostalAddress;
import javax.xml.registry.infomodel.RegistryObject;
import javax.xml.registry.infomodel.Service;
import javax.xml.registry.infomodel.ServiceBinding;
import javax.xml.registry.infomodel.SpecificationLink;
import javax.xml.registry.infomodel.TelephoneNumber;
import javax.xml.registry.infomodel.User;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.ws.scout.registry.infomodel.SpecificationLinkImpl;
import org.jboss.internal.soa.esb.addressing.helpers.EPRHelper;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.MarshalException;
import org.jboss.soa.esb.UnmarshalException;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.services.registry.Registry;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.ServiceNotFoundException;
/**
 * Utility class for the Registry.
 * If need be we can extract the interface from here, add a factory and have JAXR as a plugin, allowing
 * for other RegistryAPIs.
 *
 * @author Kurt Stam
 * @author Tom Cunningham
 */
public class JAXRRegistryImpl implements Registry
{
	private static Logger logger = Logger.getLogger(JAXRRegistryImpl.class);
    private JAXRConnectionFactory jaxrConnectionFactory;
	private ClassificationScheme jbossESBScheme;
	private static Concept jbossTModelXXX;
	public static final String JBOSS_ESB_CATEGORY = "org.jboss.soa.esb.:category";
    private static final String REGISTERED_BY_JBOSSESB = "Registered by JBoss ESB";
    public String orgCategory;
    private static final boolean isV3;
	
    public JAXRRegistryImpl() throws ConfigurationException {
        jaxrConnectionFactory = new JAXRConnectionFactory();
        orgCategory = ModulePropertyManager.getPropertyManager(ModulePropertyManager.REGISTRY_MODULE).getProperty(Environment.REGISTRY_ORG_CATEGORY, JBOSS_ESB_CATEGORY);
    }
    
    /**
	 * Publish an EPR to the Registry
	 */
	protected Service registerService(String category, String serviceName, String serviceDescription) throws JAXRException 
	{
		Service service =null;
		Organization organization = getJBossESBOrganization();
		Connection connection = jaxrConnectionFactory.getConnection();
		try {
			RegistryService rs = connection.getRegistryService();
            BusinessQueryManager bqm = rs.getBusinessQueryManager();
			BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
		    //Adding the category as prefix for the name
			service = blm.createService(blm.createInternationalString(serviceName));
			service.setDescription(blm.createInternationalString(serviceDescription));
			ClassificationScheme cScheme = getClassificationScheme(bqm, blm);
            Classification classification = blm.createClassification(cScheme, "category", category);
			service.addClassification(classification);
			organization.addService(service);
			saveRegistryObject(service, jaxrConnectionFactory);
		} finally {
			jaxrConnectionFactory.closeConnection(connection);
		}
		return service;
	}
	/** 
	 * Remove an EPR from the Registry
	 */
    public void unRegisterService(String category, String serviceName) throws RegistryException, ServiceNotFoundException{
        //    first find the ServiceBindings for this service
        Connection connection = jaxrConnectionFactory.getConnection();
        Service service = null;
        try {
            service = findService(category, serviceName);
            if (service==null) {
                throw new ServiceNotFoundException("No such EPR found for service with name = " + serviceName);
            }
            RegistryService rs = connection.getRegistryService();
            BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
            Collection<Key> serviceKeys = new ArrayList<Key>();
            serviceKeys.add(service.getKey());
            blm.deleteServices(serviceKeys);
        } catch (JAXRException je) {
            throw new RegistryException(je.getLocalizedMessage(), je);
        } finally {
            jaxrConnectionFactory.closeConnection(connection);
        }
	}
	/** 
	 * Publish an EPR to the Registry
	 */
	public void registerEPR(String category, String serviceName, String serviceDescription, EPR epr, String eprDescription) 
		throws RegistryException
	{
		if (category == null)
		{
			throw new RegistryException("Null Category argument specified") ;
		}
		if (serviceName == null)
		{
			throw new RegistryException("Null Service Name argument specified") ;
		}
		if (epr == null)
		{
			throw new RegistryException("Null EPR argument specified") ;
		}
		Connection connection = jaxrConnectionFactory.getConnection();
        try {
			final Concept jbossTModel = getJBossESBTModel(connection);
		
            //Find the service
            Service service = findService(category,serviceName);
            if (service==null) {
                logger.log(Level.DEBUG, "Service " + serviceName + " does not yet exist, creating now..");
                service = registerService(category, serviceName, serviceDescription);
            }

            RegistryService rs = connection.getRegistryService();
            BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
            BusinessQueryManager bqm = rs.getBusinessQueryManager();
            ServiceBinding serviceBinding = blm.createServiceBinding();
            String xml = EPRHelper.toXMLString(epr);
            serviceBinding.setAccessURI(xml);
            
            ArrayList<ServiceBinding> serviceBindings = new ArrayList<ServiceBinding>();
            SpecificationLink sl = new SpecificationLinkImpl(blm);
            sl.setSpecificationObject(jbossTModel);
            serviceBinding.getSpecificationLinks().add(sl);
            if (isV3) {
               if ((eprDescription == null) || ("".equals(eprDescription))) {
                   serviceBinding.setDescription(blm.createInternationalString(REGISTERED_BY_JBOSSESB));
               }
            } else {
               serviceBinding.setDescription(blm.createInternationalString(eprDescription));
            }
            serviceBindings.add(serviceBinding);
            
            service.addServiceBindings(serviceBindings);
            ClassificationScheme cScheme = getClassificationScheme(bqm, blm);
            Classification classification = blm.createClassification(cScheme, "category", category);
            service.addClassification(classification);
            
            saveRegistryObject(serviceBinding, jaxrConnectionFactory);
        } catch (JAXRException je) {
            throw new RegistryException(je.getLocalizedMessage(), je);
        } catch (MarshalException me) {
            throw new RegistryException(me.getLocalizedMessage(), me);
        } finally {
            jaxrConnectionFactory.closeConnection(connection);
        }
    }
	/** 
	 * Remove an EPR from the Registry
	 */
	public void unRegisterEPR(String category, String serviceName, EPR toBeDeletedEPR) throws RegistryException, ServiceNotFoundException{
        //first find the ServiceBindings for this service
        Connection connection = jaxrConnectionFactory.getConnection();
        Service service = null;
        try {
            service = findService(category, serviceName);
            if (service==null) {
                throw new ServiceNotFoundException("No such Service found for service with category= "
                        + category + " and name = " + serviceName);
            }
            Collection serviceBindings = findServiceBindings(service);
            String toBeDeletedEPRXml = EPRHelper.toXMLString(toBeDeletedEPR).trim();
            for (Iterator i=serviceBindings.iterator();i.hasNext();){
                ServiceBinding serviceBinding = (ServiceBinding) i.next();
                String eprXML = serviceBinding.getAccessURI().trim();
                // for backwards compatibility still have the decoder if
                // unmarchalling fails
                try {
                    EPRHelper.fromXMLString(eprXML);
                } catch (UnmarshalException unme) {
                    try {
                        eprXML = URLDecoder.decode(eprXML, "UTF-8").trim();
                    } catch (UnsupportedEncodingException ue) {
                        logger.error(unme.getMessage(), unme);
                        logger.error(ue.getMessage(), ue);
                    }
                }
                if (eprXML.equals(toBeDeletedEPRXml)) {
                    RegistryService rs = connection.getRegistryService();
                    BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
                    Collection<Key> serviceBindingKeys = new ArrayList<Key>();
                    serviceBindingKeys.add(serviceBinding.getKey());
                    blm.deleteServiceBindings(serviceBindingKeys);
                    return;
                }
            }
            throw new RegistryException("No such EPR found for service with name = "
                    + serviceName + " and EPR=" + toBeDeletedEPR);
        } catch (JAXRException je) {
            throw new RegistryException(je.getLocalizedMessage(), je);
        } catch (MarshalException me) {
            throw new RegistryException(me.getLocalizedMessage(), me);
        } finally {
            jaxrConnectionFactory.closeConnection(connection);
        }
    }
	
	/** 
	 * {@inheritDoc}
	 * @return collection services
	 */
	public List<String> findAllServices() throws RegistryException
	{
		List<String> serviceNames = new ArrayList<String>();
		try {
			Collection services = getJBossESBOrganization().getServices();
			for (Iterator i=services.iterator();i.hasNext();) {
				String serviceName = ((Service)i.next()).getName().getValue();
				serviceNames.add(serviceName);
			}
		} catch (JAXRException je) {
			throw new RegistryException(je.getLocalizedMessage(), je);
		}
		return serviceNames;
	}
	/**
	 * Find Services based on a category ("transformation").
	 * 
	 * @return collection services
	 */
	public List<String> findServices(String category) throws RegistryException
	{
		List<String>serviceNames = new ArrayList<String>();

        try {
			Collection<Service>services = findServicesForCategory(category);
			for (Iterator<Service> i=services.iterator();i.hasNext();) {
				String serviceName = i.next().getName().getValue();
				serviceNames.add(serviceName);
			}
		} catch (JAXRException je) {
			throw new RegistryException(je.getLocalizedMessage(), je);
		}
		return serviceNames;
	}
	/**
	 * 
	 * @return
	 */
	public List<EPR> findEPRs(String category, String serviceName) throws RegistryException, ServiceNotFoundException
	{
		List<EPR> eprs = new ArrayList<EPR>();

        Connection connection = jaxrConnectionFactory.getConnection();
		try {
			final Concept jbossTModel = getJBossESBTModel(connection);
			RegistryService rs = connection.getRegistryService();
			BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();

			Service service = findService(category, serviceName);
			if (service==null){
                throw new ServiceNotFoundException("Could not find service with category=" + category + " and serviceName=" + serviceName);
			}
			// Get registry service and business query manager
			Collection<ServiceBinding> serviceBindings = findServiceBindings(service);
			//Converting them to EPRs
			for (Iterator i=serviceBindings.iterator();i.hasNext();) {
				ServiceBinding serviceBinding = (ServiceBinding) i.next();

				String eprXML = serviceBinding.getAccessURI();
                EPR epr = null;
                //for backwards compatibility still have the decoder if
                //unmarchalling fails
                try {
                    epr = EPRHelper.fromXMLString(eprXML);
                    eprs.add(epr);
                } catch (UnmarshalException unme) {
                    try {
                        eprXML = URLDecoder.decode(eprXML, "UTF-8");
                        epr = EPRHelper.fromXMLString(eprXML);
                        eprs.add(epr);
                    } catch (UnsupportedEncodingException ue) {
                        logger.error(ue.getMessage(), ue);
                        throw new UnmarshalException(ue.getMessage(), ue);
                    }
                }
			}
		} catch (JAXRException je) {
			throw new RegistryException(je.getLocalizedMessage(), je);
        } catch (UnmarshalException me) {
            throw new RegistryException(me.getLocalizedMessage(), me);
		} finally {
			jaxrConnectionFactory.closeConnection(connection);
		}
		return eprs;
	}
	/**
	 * 
	 * @return
	 */
	public EPR findEPR(String category, String serviceName) throws RegistryException, ServiceNotFoundException
	{
        EPR epr = null ;
        Connection connection = jaxrConnectionFactory.getConnection();
        try {
			final Concept jbossTModel = getJBossESBTModel(connection);
			RegistryService rs = connection.getRegistryService();
			BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();

            Service service = findService(category, serviceName);
            if (service==null){
                throw new ServiceNotFoundException("Could not find service with category=" + category + " and serviceName=" + serviceName);
            }
            // Get registry service and business query manager
            Collection<ServiceBinding> serviceBindings = findServiceBindings(service);
            if (serviceBindings.iterator().hasNext()) {
                ServiceBinding serviceBinding = (ServiceBinding) serviceBindings.iterator().next();

                String eprXML = serviceBinding.getAccessURI();
                // for backwards compatibility still have the decoder if
                // unmarchalling fails
                try {
                    epr = EPRHelper.fromXMLString(eprXML);
                } catch (UnmarshalException unme) {
                    try {
                        eprXML = URLDecoder.decode(eprXML, "UTF-8");
                        epr = EPRHelper.fromXMLString(eprXML);
                    } catch (UnsupportedEncodingException ue) {
                        logger.error(ue.getMessage(), ue);
                        throw new UnmarshalException(ue.getMessage(), ue);
                    }
                }
            }
        } catch (JAXRException je) {
            throw new RegistryException(je.getLocalizedMessage(), je);
        } catch (UnmarshalException me) {
            throw new RegistryException(me.getLocalizedMessage(), me);
        } finally {
            jaxrConnectionFactory.closeConnection(connection);
        }
        
        return epr;
	}

    /**
	 * Find all Organizations with a name mathing the queryString parameter.
	 *
	 * @param organizationName used to match with the name of the organization.
	 * @return the Organization.
	 */	
    public static Organization findOrganization(String organizationName, JAXRConnectionFactory jaxrConnectionFactory) throws JAXRException {
	    if (organizationName==null) {
	    	organizationName="";
	    }
		Connection connection = jaxrConnectionFactory.getConnection();
		try {
			// Get registry service and business query manager
			RegistryService rs = connection.getRegistryService();
			BusinessQueryManager bqm = rs.getBusinessQueryManager();
			// Define find qualifiers and name patterns
			Collection<String> findQualifiers = new ArrayList<String>();
			findQualifiers.add(FindQualifier.SORT_BY_NAME_ASC);
			Collection<String> namePatterns = new ArrayList<String>();
		
			if (isV3) {
				namePatterns.add(organizationName);			
			} else {
				namePatterns.add("%" + organizationName + "%");
			}
	        //Find based upon qualifier type and values
			logger.log(Level.DEBUG,"Going to query the registry for name pattern " + namePatterns);
			BulkResponse response = bqm.findOrganizations(findQualifiers,
					namePatterns, null, null, null, null);
			if (response.getStatus()==JAXRResponse.STATUS_SUCCESS) {
				for (Iterator orgIter = response.getCollection().iterator(); orgIter.hasNext();)
				{
					Organization org = (Organization) orgIter.next();
					logger.log(Level.DEBUG, "Organization name: " + org.getName().getValue());
					logger.log(Level.DEBUG, "Description: " + org.getDescription().getValue());
					logger.log(Level.DEBUG, "Key id: " + org.getKey().getId());
					User primaryContact = org.getPrimaryContact();
					logger.log(Level.DEBUG, "Primary Contact: " + primaryContact.getPersonName().getFullName());
					if (orgIter.hasNext()) {
						logger.log(Level.ERROR, "Found " + response.getCollection().size()
								+ " Organization, while expecting only one of name " + organizationName);
					}
					return org;
				}
			}
			return null;
		} finally {
			jaxrConnectionFactory.closeConnection(connection);
		}
	}

    /**
	 *
	 * @param category
	 * @param serviceName
	 * @return Service
	 */
	protected Service findService(String category, String serviceName) throws JAXRException
	{
		if ((category==null) || (serviceName == null)) {
			return null ;
		}
		Connection connection = jaxrConnectionFactory.getConnection();
		try {
			final Concept jbossTModel = getJBossESBTModel(connection);
			// Get registry service and business query manager
			RegistryService rs = connection.getRegistryService();
			BusinessQueryManager bqm = rs.getBusinessQueryManager();
            BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
            final ClassificationScheme cScheme = getClassificationScheme(bqm, blm) ;
            Collection<Classification> classifications = new ArrayList<Classification>();
            Classification classification =
                blm.createClassification(
                  cScheme,
                  "category", category );
            classifications.add(classification);
			Collection<String> namePatterns = new ArrayList<String>();
			namePatterns.add(serviceName);

            Collection<SpecificationLink> specifications = new ArrayList<SpecificationLink>();
            SpecificationLink sl = new SpecificationLinkImpl(blm);
            sl.setSpecificationObject(jbossTModel);
			specifications.add(sl);
            
	        //Find based upon qualifier type and values
			Collection<String> findQualifiers = new ArrayList<String>();
			findQualifiers.add(FindQualifier.AND_ALL_KEYS);
			findQualifiers.add(FindQualifier.EXACT_NAME_MATCH);
			findQualifiers.add(FindQualifier.SORT_BY_NAME_DESC);
			logger.log(Level.DEBUG, "Going to query the registry for name pattern " + namePatterns);
			BulkResponse response = bqm.findServices(null, findQualifiers,
					namePatterns, classifications, specifications);
			if (response.getStatus()==JAXRResponse.STATUS_SUCCESS) {
				for (Iterator servIter = response.getCollection().iterator(); servIter.hasNext();)
				{
					Service service = (Service) servIter.next();
					logger.log(Level.DEBUG, "Service name: " + service.getName().getValue());
					if (service.getDescription()!=null) {
						logger.log(Level.DEBUG, "Description: " + service.getDescription().getValue());
					}
					logger.log(Level.DEBUG, "Key id: " + service.getKey().getId());
					if (servIter.hasNext()) {
						logger.log(Level.ERROR, "Found " + response.getCollection().size()
								+ " Services, while expecting only one by the name of "
								+ serviceName + " in category " + category);
					}
					return service;
				}
			}
			return null;
		} finally {
			jaxrConnectionFactory.closeConnection(connection);
		}
	}

    /**
	 * Finds all services for a given category.
	 * @param category
	 * @return Service
	 */
	@SuppressWarnings("unchecked")
	protected Collection<Service> findServicesForCategory(String category) throws JAXRException
	{
		Collection<Service> services = new ArrayList<Service>();
		if (category == null) {
			return services;
		}
		Connection connection = jaxrConnectionFactory.getConnection();
		try {
			// Get registry service and business query manager
			RegistryService rs = connection.getRegistryService();
			BusinessQueryManager bqm = rs.getBusinessQueryManager();
            BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
            ClassificationScheme cScheme = getClassificationScheme(bqm, blm);
            //Create classification
            Classification classification =
                blm.createClassification(
                  cScheme,
                  "category", category );

			// Define find qualifiers and name patterns
            Collection<String> findQualifiers = new ArrayList<String>();
            findQualifiers.add(FindQualifier.AND_ALL_KEYS);
			findQualifiers.add(FindQualifier.SORT_BY_NAME_DESC);
			Collection<Classification> classifications = new ArrayList<Classification>();
            classifications.add(classification);
	        //Find based upon qualifier type and values
			logger.log(Level.DEBUG, "Going to query the registry for classification " + classifications);
			BulkResponse response = bqm.findServices(null, findQualifiers,
					null, classifications, null);
			if (response.getStatus()==JAXRResponse.STATUS_SUCCESS) {
				services = (Collection<Service>) response.getCollection();
			}
			return services;
		} finally {
			jaxrConnectionFactory.closeConnection(connection);
		}
	}

    /**
	 * Create a jbossesb organization under which we will register all our services.
	 *
	 * @return
	 * @throws JAXRException
	 */
    protected static Organization createJBossESBOrganization(JAXRConnectionFactory jaxrConnectionFactory) throws JAXRException
	{
//		Getting the connection to the Registry (reading config)
		Connection connection = jaxrConnectionFactory.getConnection();
		try {
			//Logging in
			RegistryService rs = connection.getRegistryService();
			//Building organization
			BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
			Organization organization = blm.createOrganization(blm.createInternationalString("Red Hat/JBossESB"));
				organization.setDescription(blm.createInternationalString("Red Hat/JBoss Professional Open Source, Services for the JBossESB"));

			User user = blm.createUser();
			organization.setPrimaryContact(user);
			PersonName personName = blm.createPersonName("JBossESB");
			TelephoneNumber telephoneNumber = blm.createTelephoneNumber();
			telephoneNumber.setNumber("404 467-8555");
			telephoneNumber.setType(null);
			PostalAddress address = blm.createPostalAddress("3340",
					"Peachtree Road, NE, Suite 1200", "Atlanta", "GA", "USA",
					"30326", "Office");
			Collection<PostalAddress> postalAddresses = new ArrayList<PostalAddress>();
			postalAddresses.add(address);
			Collection<EmailAddress> emailAddresses = new ArrayList<EmailAddress>();
			EmailAddress emailAddress = blm
					.createEmailAddress("jbossesb@jboss.com");
			emailAddresses.add(emailAddress);

			Collection<TelephoneNumber> numbers = new ArrayList<TelephoneNumber>();
			numbers.add(telephoneNumber);
			user.setPersonName(personName);
			user.setPostalAddresses(postalAddresses);
			user.setEmailAddresses(emailAddresses);
			user.setTelephoneNumbers(numbers);

	//		Scout does not support this (yet), so leaving it out for now.
	//		ClassificationScheme cScheme = getClassificationScheme(blm,
	//				blm.createInternationalString("uddi-org:general_keywords"), blm.createInternationalString(""));
	//		Classification classification = blm.createClassification(cScheme,
	//				blm.createInternationalString("JBoss ESB"), blm.createInternationalString("JBESB"));
	//		org.addClassification(classification);

			saveRegistryObject(organization, jaxrConnectionFactory);
			return organization;
		} finally {
			jaxrConnectionFactory.closeConnection(connection);
		}
	}

    /**
	 * Create a jbossesb organization under which we will register all our services.
	 *
	 * @return
	 * @throws JAXRException
	 */
    private synchronized Concept getJBossESBTModel(Connection connection) throws JAXRException
	{
		if (jbossTModelXXX == null) {
			RegistryService rs = connection.getRegistryService();
			BusinessQueryManager bqm = rs.getBusinessQueryManager();
            
            Collection<String> findQualifiers = new ArrayList<String>();
            findQualifiers.add(FindQualifier.SORT_BY_NAME_ASC);
            Collection<String> namePatterns = new ArrayList<String>();
            if (isV3) {
                namePatterns.add(orgCategory);
            } else {
            	namePatterns.add("%" + orgCategory + "%");
            }
            
            BulkResponse response = bqm.findConcepts(findQualifiers, namePatterns, null, null, null);
			if (response.getStatus()==JAXRResponse.STATUS_SUCCESS) {
				Collection<RegistryObject> regs = (Collection<RegistryObject>) response.getCollection();
				Iterator<RegistryObject> orgIter = regs.iterator();
				if (orgIter.hasNext())
				{
					jbossTModelXXX = (Concept) orgIter.next();
				}
			}
		}
		return jbossTModelXXX;
	}

    
    /**
	 * Save Registry Object
	 */
	private static void saveRegistryObject(RegistryObject registryObject, JAXRConnectionFactory jaxrConnectionFactory) throws JAXRException
	{
//		Getting the connection to the Registry (reading config)
		Connection connection = jaxrConnectionFactory.getConnection();
		try {

			BulkResponse br = null;
			//Logging in
			RegistryService rs = connection.getRegistryService();
			//Building organization
			BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();

			final String interfaceName ;
			if (registryObject instanceof Organization) {
				interfaceName = Organization.class.getCanonicalName() ;
				Collection<Organization> orginations = new ArrayList<Organization>();
				
				orginations.add((Organization) registryObject);
				br = blm.saveOrganizations(orginations);
			} else if (registryObject instanceof Service) {
				interfaceName = Organization.class.getCanonicalName() ;
				Collection<Service> services = new ArrayList<Service>();
				services.add((Service) registryObject);
				br = blm.saveServices(services);
			} else if (registryObject instanceof ServiceBinding) {
				interfaceName = Organization.class.getCanonicalName() ;
				Collection<ServiceBinding> serviceBindings = new ArrayList<ServiceBinding>();
				serviceBindings.add((ServiceBinding) registryObject);
				br = blm.saveServiceBindings(serviceBindings);
			} else {
				logger.log(Level.ERROR, "Trying to save an unsupported RegistryObject");
				throw new JAXRException("Trying to save an unsupported RegistryObject");
			}
            //Verify the return
			if (br!=null && br.getStatus() == JAXRResponse.STATUS_SUCCESS) {
				logger.log(Level.DEBUG, interfaceName + " successfully saved");
				Collection coll = br.getCollection();
				Iterator iter = coll.iterator();
				while (iter.hasNext()) {
					Key key = (Key) iter.next();
					registryObject.setKey(key);
					logger.log(Level.DEBUG, "Saved Key=" + key.getId());
				}
			} else {
				logger.log(Level.ERROR, "Errors occurred during save.");
				if (br!=null) {
					Collection exceptions = br.getExceptions();
					Iterator iter = exceptions.iterator();
					StringBuilder errors = new StringBuilder();
					JAXRException je = new JAXRException("JAXRExceptions occurred during save");
					while (iter.hasNext()) {
						Exception e = (Exception) iter.next();
						errors.append(e.getLocalizedMessage()).append("\n");
						je.setStackTrace(e.getStackTrace());
						logger.log(Level.ERROR, e.getLocalizedMessage(), e);
						//if it's the last error, throw it now and set the current stacktrace
						if (!iter.hasNext()) {
							throw new JAXRException(errors.toString(), e);
						}
					}
					throw new JAXRException("Errors occurred during save. Response status=" + br.getStatus());
				}
				throw new JAXRException("Errors occurred during save");
			}
		} finally {
			jaxrConnectionFactory.closeConnection(connection);
		}
	}

    /**
	 * finds the JBossESB Organizationa and creates one if it is not there.
	 * @return JBossESB Organization
	 * @throws JAXRException
	 */
	private Organization getJBossESBOrganization() throws JAXRException
	{
		Organization jbossESBOrganization = findOrganization("Red Hat/JBossESB", jaxrConnectionFactory);

		if (jbossESBOrganization==null) {
			jbossESBOrganization = createJBossESBOrganization(jaxrConnectionFactory);
		}
		
		return jbossESBOrganization;
	}
	
	/**
	 * finds the JBossESB category classification scheme and creates one if it is not there.
	 * 
	 * @return JBossESB category classification scheme
	 * @throws JAXRException For errors during querying/creating.
	 */
	private synchronized ClassificationScheme getClassificationScheme(final BusinessQueryManager bqm, final BusinessLifeCycleManager blm) throws JAXRException
	{
		if (jbossESBScheme == null) {
			// Define find qualifiers and name patterns
			Collection<String> findQualifiers = new ArrayList<String>();
			findQualifiers.add(FindQualifier.AND_ALL_KEYS);
			findQualifiers.add(FindQualifier.EXACT_NAME_MATCH);
			findQualifiers.add(FindQualifier.SORT_BY_NAME_DESC);
			ClassificationScheme cScheme = bqm.findClassificationSchemeByName(findQualifiers, JBOSS_ESB_CATEGORY);
			// If the scheme returned null, then we have to create a org.jboss.soa.esb.:category scheme to use
			if (cScheme == null) {
				try {
					ClassificationScheme scheme = blm.createClassificationScheme(JBOSS_ESB_CATEGORY, JBOSS_ESB_CATEGORY);
					ArrayList<ClassificationScheme> cSchemes = new ArrayList<ClassificationScheme>();
					cSchemes.add(scheme);
					BulkResponse br = blm.saveClassificationSchemes(cSchemes);
				} catch (Exception e) {
					throw new JAXRException(e);
				}
				cScheme = bqm.findClassificationSchemeByName(findQualifiers, JBOSS_ESB_CATEGORY);
			}
			jbossESBScheme = cScheme ;
		}
		return jbossESBScheme ;
	}

    /**
	 * Find the ServiceBindings for a given Service
	 * @param service
	 * @return
	 * @throws RegistryException
	 */
	@SuppressWarnings("unchecked")
	private Collection<ServiceBinding> findServiceBindings(Service service) throws RegistryException
	{
		final Collection currentBindings ;
		try {
			currentBindings = service.getServiceBindings() ;
		} catch (final JAXRException jaxre) {
			throw new RegistryException(jaxre.getLocalizedMessage(), jaxre);
		}
		if ((currentBindings != null) && (currentBindings.size() > 0)) {
			return currentBindings;
		}
		
		Collection<ServiceBinding> serviceBindings = new ArrayList<ServiceBinding>();
		Connection connection = jaxrConnectionFactory.getConnection();
		try {
			final Concept jbossTModel = getJBossESBTModel(connection);
			RegistryService rs = connection.getRegistryService();
			BusinessQueryManager bqm = rs.getBusinessQueryManager();
			BusinessLifeCycleManager blm = rs.getBusinessLifeCycleManager();
			Collection<String> findQualifiers = null;
			//Collection<String> findQualifiers = new ArrayList<String>();
			//findQualifiers.add(FindQualifier.SORT_BY_DATE_ASC);
			ArrayList<SpecificationLink> specs = new ArrayList<SpecificationLink>();
            SpecificationLink sl = new SpecificationLinkImpl(blm);
            sl.setSpecificationObject(jbossTModel);

			specs.add(sl);
			BulkResponse bulkResponse = bqm.findServiceBindings(service.getKey(),findQualifiers,null,specs);
			if (bulkResponse.getStatus()==JAXRResponse.STATUS_SUCCESS){
				serviceBindings = (Collection<ServiceBinding>)bulkResponse.getCollection();
				for (Iterator bindIter = serviceBindings.iterator(); bindIter.hasNext();)
				{
					ServiceBinding binding = (ServiceBinding) bindIter.next();
					Collection<SpecificationLink> bindingSpecs = binding.getSpecificationLinks();					
		            SpecificationLink slink = new SpecificationLinkImpl(blm);
		            slink.setSpecificationObject(jbossTModel);

					bindingSpecs.add(slink);

				}
			}
			return serviceBindings;
		} catch (Exception je) {
			throw new RegistryException(je.getLocalizedMessage(), je);
		} finally {
			jaxrConnectionFactory.closeConnection(connection);
		}
	}
	
	static {
		isV3 = "3.0".equals(Configuration.getRegistryUDDIVersion());
	}
}

	
