/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.soa.esb.listeners.config;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.Map.Entry;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;

import javax.management.JMException;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;

import org.jboss.deployment.DeploymentException;
import org.jboss.deployment.DeploymentInfo;
import org.jboss.deployment.SubDeployer;
import org.jboss.deployment.SubDeployerSupport;
import org.jboss.internal.soa.esb.listeners.war.Filter;
import org.jboss.internal.soa.esb.listeners.war.Servlet;
import org.jboss.internal.soa.esb.listeners.war.WebDeploymentArchive;
import org.jboss.internal.soa.esb.listeners.war.WebModel;
import org.jboss.internal.soa.esb.publish.ContractReferencePublisher;
import org.jboss.internal.soa.esb.util.JBossDeployerUtil;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.internal.soa.esb.webservice.ESBContractGenerator;
import org.jboss.internal.soa.esb.webservice.ESBResponseFilter;
import org.jboss.internal.soa.esb.webservice.ESBServiceContractReferencePublisher;
import org.jboss.internal.soa.esb.webservice.ESBServiceEndpointInfo;
import org.jboss.internal.soa.esb.webservice.JAXWSProviderClassGenerator;
import org.jboss.logging.Logger;
import org.jboss.metadata.MetaData;
import org.jboss.metadata.XmlFileLoader;
import org.jboss.mx.loading.LoaderRepositoryFactory;
import org.jboss.mx.util.MBeanProxyExt;
import org.jboss.mx.util.ObjectNameConverter;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.lifecycle.LifecycleResourceManager;
import org.jboss.soa.esb.listeners.config.model.ModelAdapter;
import org.jboss.soa.esb.util.DeploymentArchive;
import org.jboss.soa.esb.util.FileUtil;
import org.jboss.system.ServiceControllerMBean;
import org.jboss.web.AbstractWebContainer;
import org.w3c.dom.Element;

/**
 * comment
 *
 * @author <a href="bill@jboss.com">Bill Burke</a>
 * @version $Revision: 1.1 $
 */
public class JBoss4ESBDeployer extends SubDeployerSupport
        implements SubDeployer, JBoss4ESBDeployerMBean
{
   private final static Logger log = Logger.getLogger(JBoss4ESBDeployer.class);

   private ServiceControllerMBean serviceController;
   private Properties actionArtifactProperties;
   private File esbWarFiles ;
   private final Set<String> esbNames = new HashSet<String>() ;
   
   private static final String PREFIX_CANONICAL_NAME = "jboss.esb:deployment=" ;
   private static final String ESB_ARTIFACT_NAME = "jbossesb.esb" ;
   private static final String SUB_DEPLOYMENTS = JBoss4ESBDeployer.class.getName() + "#SUB_DEPLOYMENTS";
    
   /**
    * The path the the directory that will be used to generate the war file
    * for all web gateway deployments (EBWS and HTTP Gateways).
    */
   private String warFilesDir;
   
   /**
    * Default CTOR used to set default values to the Suffixes and RelativeOrder
    * attributes. Those are read at subdeployer registration time by the MainDeployer
    * to alter its SuffixOrder.
    */
   public JBoss4ESBDeployer()
   {
      setSuffixes(new String[]{".esb"});
      setRelativeOrder(1000); // before old EJB 2.1 deployer
   }
   
   public static boolean hasFile(DeploymentInfo di, String filePath)
   {
      String urlStr = di.url.getFile();
      try
      {
         URL dd = di.localCl.findResource(filePath);
         if (dd != null)
         {

            // If the DD url is not a subset of the urlStr then this is coming
            // from a jar referenced by the deployment jar manifest and the
            // this deployment jar it should not be treated as persistence
            if (di.localUrl != null)
            {
               urlStr = di.localUrl.toString();
            }

            String ddStr = dd.toString();
            if (ddStr.indexOf(urlStr) >= 0)
            {
               return true;
            }
         }
      }
      catch (Exception ignore)
      {
      }
      return false;
   }
   
    /**
    * Returns true if this deployer can deploy the given DeploymentInfo.
    *
    * @return True if this deployer can deploy the given DeploymentInfo.
    * @jmx:managed-operation
    */
   public boolean accepts(DeploymentInfo di)
   {
      String urlStr = di.url.toString();
      return urlStr.endsWith(".esb") || urlStr.endsWith(".esb/") ||
              urlStr.endsWith("-esb.xml");
   }

   /**
    * Get a reference to the ServiceController
    */
   protected void startService() throws Exception
   {
      actionArtifactProperties = JBossDeployerUtil.getArtifactProperties("/actionArtifactMap.properties");
      
      serviceController = (ServiceControllerMBean)
              MBeanProxyExt.create(ServiceControllerMBean.class,
                      ServiceControllerMBean.OBJECT_NAME, server);

      mainDeployer.addDeployer(this);
      LifecycleResourceManager.deactivateHook() ;
   }
   
   @Override
   protected void stopService() throws Exception
   {
      LifecycleResourceManager.getSingleton().cleanupAllResources() ;
      super.stopService();
   }

   protected URL getDocumentUrl(DeploymentInfo di)
   {
      String urlStr = di.url.toString();
      if (urlStr.endsWith(".esb") || urlStr.endsWith(".esb/"))
      {
         return di.localCl.getResource("META-INF/jboss-esb.xml");
      }
      return di.url;
   }

   public void init(DeploymentInfo di) throws DeploymentException
   {
      if (warFilesDir == null)
      {
          final String errorMsg = String.format("No property named '%s' was configured in jbossesb.sar/META-INF/jboss-service.xml for %s", "warFilesDir", getClass().getName());
          throw new DeploymentException(errorMsg);
      }
          
      final File tmpDir = new File(warFilesDir);
      if (!tmpDir.exists())
      {
          final String errorMsg = String.format("The directory configured for %s='%s' does not exist.", "warFilesDir", tmpDir);
          throw new DeploymentException(errorMsg);
      }
      
      esbWarFiles = JBossDeployerUtil.createDir(tmpDir, "esbwarfiles");
      
      try
      {
         if (di.url.getProtocol().equalsIgnoreCase("file"))
         {
            File file = new File(di.url.getFile());

            if (!file.isDirectory())
            {
               // If not directory we watch the package
               di.watch = di.url;
            }
            else
            {
               // If directory we watch the xml files
               di.watch = new URL(di.url, "META-INF/jboss-esb.xml");
            }
         }
         else
         {
            // We watch the top only, no directory support
            di.watch = di.url;
         }

         XmlFileLoader xfl = new XmlFileLoader();
         InputStream in = di.localCl.getResourceAsStream("META-INF/deployment.xml");
         if (in != null)
         {
            try
            {
               Element jboss = xfl.getDocument(in, "META-INF/deployment.xml").getDocumentElement();
               // Check for a ejb level class loading config
               Element loader = MetaData.getOptionalChild(jboss, "loader-repository");
               if (loader != null)
               {
                  LoaderRepositoryFactory.LoaderRepositoryConfig config =
                          LoaderRepositoryFactory.parseRepositoryConfig(loader);
                  di.setRepositoryInfo(config);
               }

            }
            finally
            {
               in.close();
            }
         }
         
         final URL document = getDocumentUrl(di);
         
         if (document == null)
         {
            throw new DeploymentException("Unable to find document url of META-INF/jboss-esb.xml in: "
                    + di.url);
         }
         final String jbossEsbXml = JBossDeployerUtil.readEsbConfig(document.openStream()); ;
         
         final Set<ObjectName> deps = new HashSet<ObjectName>();
         final ModelAdapter model = JBossDeployerUtil.getJbossEsbModel(jbossEsbXml) ;
         addActionDependencies(di.shortName, model, deps) ;
         
         JBoss4ESBDeployment deployment = new JBoss4ESBDeployment(jbossEsbXml, di.shortName);
         initialiseDeploymentName(deployment) ;
         final String deploymentName = deployment.getDeploymentName() ;
         di.context.put(JBoss4ESBDeploymentMetaData.class, new JBoss4ESBDeploymentMetaData(deployment, deploymentName, deps)) ;
         di.context.put(SUB_DEPLOYMENTS, new ArrayList<File>()) ;

         // invoke super-class initialization
         super.init(di);

         // Deploy web gateways...
          final ClassLoader origCL = Thread.currentThread().getContextClassLoader() ;
          try
          {
              Thread.currentThread().setContextClassLoader(di.localCl) ;
              deployWebGateways(deployment, model, di);
          }
          finally
          {
              Thread.currentThread().setContextClassLoader(origCL) ;
          }
      }
      catch (Exception e)
      {
         if (e instanceof DeploymentException)
         {
            throw(DeploymentException) e;
         }
         throw new DeploymentException("failed to initialize", e);
      }
   }

   public synchronized void create(DeploymentInfo di) throws DeploymentException
   {
      log.info("create esb service, " + di.shortName);
      try
      {
         final JBoss4ESBDeploymentMetaData metaData = (JBoss4ESBDeploymentMetaData)di.context.get(JBoss4ESBDeploymentMetaData.class) ;
         final Set<ObjectName> deps = metaData.getDependencies() ;
         InputStream in = di.localCl.getResourceAsStream("META-INF/deployment.xml");
         if (in != null)
         {
            try
            {
               XmlFileLoader xfl = new XmlFileLoader();
               Element jboss = xfl.getDocument(in, "META-INF/deployment.xml").getDocumentElement();
               // Check for a ejb level class loading config
               Iterator depends = MetaData.getChildrenByTagName(jboss, "depends");
               if (depends != null)
               {
                  while (depends.hasNext())
                  {
                     Element depend = (Element)depends.next();
                     ObjectName depOn = new ObjectName(MetaData.getElementContent(depend));
                     deps.add(depOn);
                  }
               }
               Iterator esbDepends = MetaData.getChildrenByTagName(jboss, "esb-depends");
               if ((esbDepends != null) && esbDepends.hasNext())
               {
                  final Map<String, DeploymentInfo> subDeploymentLocationMap ;
                  if (di.subDeployments.size() > 0)
                  {
                     subDeploymentLocationMap = new HashMap<String, DeploymentInfo>() ;
                     final Set<DeploymentInfo> subDeployments = (Set<DeploymentInfo>)di.subDeployments ;
                     for(DeploymentInfo subDI : subDeployments)
                     {
                        final String urlPath = subDI.url.getPath() ;
                        final String deployablePath = (urlPath.endsWith("/") ? urlPath.substring(0, urlPath.length()-1) : urlPath) ;
                        final int lastSeparator = deployablePath.lastIndexOf('/') ;
                        final String deployable = (lastSeparator >= 0 ? deployablePath.substring(lastSeparator+1) : deployablePath) ;
                        if (subDeploymentLocationMap.put(deployable, subDI) != null)
                        {
                           throw new DeploymentException("Duplicate subDeployment name: " + deployable) ;
                        }
                     }
                  }
                  else
                  {
                     throw new DeploymentException("No subdeployments to match esb-depends") ;
                  }
                  
                  do
                  {
                     Element depend = (Element)esbDepends.next();
                     final String deployable = MetaData.getElementContent(depend) ;
                     final DeploymentInfo subDI = subDeploymentLocationMap.get(deployable) ;
                     if ((subDI != null) && subDI.context.containsKey(AbstractWebContainer.WEB_MODULE))
                     {
                        final ObjectName jmxName = (ObjectName) subDI.context.get(AbstractWebContainer.WEB_MODULE) ;
                        deps.add(jmxName) ;
                     }
                     else
                     {
                         throw new DeploymentException("Could not locate WAR subdeployment matching: " + deployable) ;
                     }
                  }
                  while (esbDepends.hasNext()) ;
               }
            }
            finally
            {
               in.close();
            }
         }

         String name = PREFIX_CANONICAL_NAME + metaData.getDeploymentName();
         ObjectName on = ObjectNameConverter.convert(name);
         // Check that the name is not registered
         if (server.isRegistered(on) == true)
         {
            throw new DeploymentException("Duplicate registration for " + name) ;
         }
         
         final JBoss4ESBDeployment deployment = metaData.getDeployment() ;
         deployment.setClassloader(di.ucl);
         server.registerMBean(deployment, on);
         di.deployedObject = on;
         log.debug("Deploying: " + di.url);
         // Invoke the create life cycle method
         serviceController.create(di.deployedObject, deps);
      }
      catch (Exception e)
      {
         throw new DeploymentException("Error during create of ESB Module: "
                 + di.url, e);
      }
      super.create(di);
   }

   private void addActionDependencies(final String deploymentName,
      final ModelAdapter model, final Set<ObjectName> deps)
      throws MalformedObjectNameException
   {
      final Set<String> artifacts = new HashSet<String>() ;
      artifacts.add(ESB_ARTIFACT_NAME) ;

      final Set<String> actionClasses = model.getActions() ;
      
      final int numActionClasses = (actionClasses == null ? 0 : actionClasses.size()) ;
      if (numActionClasses > 0)
      {
          for(final String actionClass: actionClasses)
          {
              final String artifact = (String)actionArtifactProperties.get(actionClass) ;
              if (artifact != null)
              {
                  artifacts.add(artifact) ;
              }
          }
      }
      
      for(final String artifact: artifacts)
      {
          if (!deploymentName.equals(artifact))
          {
              final String canonicalName = PREFIX_CANONICAL_NAME + artifact ;
              final ObjectName on = ObjectNameConverter.convert(canonicalName) ;
              deps.add(on) ;
          }
      }
   }

   private void createWebserviceWars(final DeploymentInfo di, final ModelAdapter model, final String deploymentName,
                                     final List<ContractReferencePublisher> publishers, WebDeploymentArchive webDeployment)
       throws DeploymentException
   {
       final List<WebserviceInfo> endpointServices =  model.getWebserviceServices() ;
       if (endpointServices != null)
       {
           if (endpointServices.size() > 0)
           {
               try
               {
                   final JAXWSProviderClassGenerator generator = new JAXWSProviderClassGenerator() ;
                   
                   for(WebserviceInfo webserviceInfo: endpointServices)
                   {
                       final Service service = webserviceInfo.getService() ;

                       // Copy all schemas to the wsdl directory to support imports.
                       final File esbArchive = new File(di.watch.getFile());
                       final Map<String, String> schemasMap = JBoss4ESBDeployer.getSchemas(esbArchive);
                       final String wsdlDir = "WEB-INF/wsdl/" + service.getCategory().replace('/', '_') + "/";
                       for (Entry<String, String> schemaEntry : schemasMap.entrySet())
                       {
                           webDeployment.addEntry(wsdlDir + schemaEntry.getKey(), schemaEntry.getValue().getBytes("UTF-8"));
                           log.debug("Added schema " + wsdlDir + schemaEntry.getKey());
                       }
                       
                       final ESBServiceEndpointInfo serviceInfo = new ESBServiceEndpointInfo(webserviceInfo) ;
                       final String wsdl = ESBContractGenerator.generateWSDL(webserviceInfo, serviceInfo, di.localCl) ;
                       webDeployment.addEntry(serviceInfo.getWSDLFileName(), wsdl.getBytes("UTF-8"));

                       final String handlers = JBossDeployerUtil.getHandlers(serviceInfo) ;
                       final boolean includeHandlers = (handlers != null) ;
                       if (includeHandlers)
                       {
                           final String wsHandlerName = "WEB-INF/classes/" + serviceInfo.getPackageName().replace('.', '/') + "/esb-jaxws-handlers.xml" ;
                           webDeployment.addEntry(wsHandlerName, handlers.getBytes("UTF-8"));
                       }
                       
                       final byte[] wsClass = generator.generate(service.getCategory(),
                           service.getName(), serviceInfo, includeHandlers) ;
                       final String wsClassName = serviceInfo.getClassName().replace('.', '/') + ".class" ;
                       webDeployment.addEntry("WEB-INF/classes/" + wsClassName, wsClass);

                       // Servlet....
                       final Servlet servlet = new Servlet(serviceInfo.getServletName(), serviceInfo.getClassName(), webDeployment.getWebModel());
                       servlet.getUrlMappings().add("/ebws" + serviceInfo.getServletPath());

                       // Filter...
                       // At the moment we only need the filter to overcome a bug in JBossWS re In-Only endpoints, so
                       // we only include for one way services...
                       if(serviceInfo.isOneWay()) {
                           new Filter(serviceInfo.getServletName() + "_Filter", ESBResponseFilter.class.getName(), servlet);
                       }

                       final ContractReferencePublisher publisher = new ESBServiceContractReferencePublisher(service, webserviceInfo.getDescription(), serviceInfo.getServletName()) ;
                       publishers.add(publisher) ;
                   }
               }
               catch (final Exception ex)
               {
                   throw new DeploymentException("Failed to create webservice artifact", ex) ;
               }
           }
       }
   }

    private void deployWebGateways(JBoss4ESBDeployment deployment, ModelAdapter model, DeploymentInfo di) throws ConfigurationException, DeploymentException, JMException {
        final String deploymentName = deployment.getDeploymentName() ;
        final List<ContractReferencePublisher> publishers = new ArrayList<ContractReferencePublisher>() ;
        WebDeploymentArchive webDeployment = new WebDeploymentArchive(getESBWarFileName(di));
        WebModel webModel = webDeployment.getWebModel();
        
        // Set the global security domain and global security method.
        // These setting are shared for all http-providers and EBWSs 
        // in a jboss-esb.xml file. 
        webModel.setAuthDomain(model.getAuthDomain());
        webModel.setAuthMethod(model.getAuthMethod());

        // Add the EBWS components...
        createWebserviceWars(di, model, deploymentName, publishers, webDeployment);

        // Add the web deployments info to the WebModel...
        ModelUtil.updateWebModel(ModelUtil.getListenerGroups(model), webModel);

        // Add a sub-deloyment for the web model...
        List<Servlet> servlets = webModel.getServlets();
        if(!servlets.isEmpty()) {
            
            // Finalize the webDeployment...
            webDeployment.finalizeArchive();

            // Now create and deploy the web deployment...
            File file = createGatewayTempDeployment(webDeployment, di);
            try {
                deployUrl(di, file.toURL(), webDeployment.getArchiveName());

                // Maintain a list of sub deployments so we can clean them
                // up on undeploy...
                getSubDeployments(di).add(file);
            } catch (Throwable throwable) {
                if(file.exists()) {
                    file.delete();
                }
                throw new ConfigurationException("Error deploying '" + file.getAbsolutePath() + "'.", throwable);
            }

            deployment.setPublishers(publishers) ;
            deployment.setServlets(servlets);
        }
    }

    private File createGatewayTempDeployment(DeploymentArchive deploymentArchive, DeploymentInfo di) throws ConfigurationException {

        File gatewayDeployFile = createTempDeploymentFile(esbWarFiles, deploymentArchive.getArchiveName(), true);
        FileOutputStream fileOutStream;

        try {
            fileOutStream = new FileOutputStream(gatewayDeployFile);
        } catch (FileNotFoundException e) {
            throw new ConfigurationException("Unexpected exception.", e);
        }

        try {
            deploymentArchive.toOutputStream(fileOutStream);
        } catch (IOException e) {
            throw new ConfigurationException("Error writing deployment archive '" + gatewayDeployFile.getAbsolutePath() + "'.", e);
        } finally {
            try {
                fileOutStream.close();
            } catch (IOException e) {
                log.warn("Error closing deployment archive stream '" + gatewayDeployFile.getAbsolutePath() + "'", e);
            }
        }

        return gatewayDeployFile;
    }

    private File createTempDeploymentFile(final File tempDir, final String archiveName, final boolean createDir) throws ConfigurationException {
        if (!tempDir.exists())
        {
            if (!createDir)
            {
                return null ;
            }
            tempDir.mkdir() ;
        }

        File file = new File(tempDir, archiveName);
        if(file.exists()) {
            log.debug("Deployment archive '" + archiveName + "' already exists.  Overwriting!!");
        }

        return file;
    }
   
    private String getESBWarFileName(final DeploymentInfo di)
    {
        final String esbName = di.shortName;
        final int lastSeparator = esbName.lastIndexOf('.') ;
        return ((lastSeparator >= 0) ? esbName.substring(0, lastSeparator) : esbName) + ".war" ;
    }

   public synchronized void start(DeploymentInfo di)
           throws DeploymentException
   {
      try
      {
         serviceController.start(di.deployedObject);
      }
      catch (Exception e)
      {
         try
         {
            stop(di);
            destroy(di);
         }
         catch (DeploymentException ignore)
         {
         }
         throw new DeploymentException("Error during start of ESB Module: "
                 + di.url, e);
      }

      super.start(di);
   }

   public void stop(DeploymentInfo di)
           throws DeploymentException
   {
      if (di.deployedObject != null)
      {
         try
         {
            serviceController.stop(di.deployedObject);
         }
         catch (Exception e)
         {
            throw new DeploymentException("Error during stop of ESB Module: "
                    + di.url, e);
         }
      }
      super.stop(di);
   }

   public void destroy(DeploymentInfo di)
           throws DeploymentException
   {
      final JBoss4ESBDeploymentMetaData metaData = (JBoss4ESBDeploymentMetaData)di.context.get(JBoss4ESBDeploymentMetaData.class) ;

       for(File subDeployment : getSubDeployments(di)) {
           try {
               if(subDeployment.exists()) {
                   if(!subDeployment.delete()) {
                       log.debug("Failed to delete sub deployment '" + subDeployment.getAbsolutePath() + "'.");
                   }
               }
           } catch (Exception e) {
               log.debug("Failed to delete deployment file '" + subDeployment.getAbsolutePath() + "'.  Exception: " + e.getMessage());
           }
       }

      if (metaData != null)
      {
          final String deploymentName = metaData.getDeploymentName() ;
          removeDeploymentName(deploymentName) ;
      }
      if (di.deployedObject != null)
      {
         try
         {
            serviceController.destroy(di.deployedObject);
            server.unregisterMBean(di.deployedObject);
         }
         catch (Exception e)
         {
            throw new DeploymentException("Error during stop of ESB Module: "
                    + di.url, e);
         }
      }
      super.destroy(di);
   }
   
    private synchronized void initialiseDeploymentName(final JBoss4ESBDeployment deployment)
    {
        final String deploymentName = deployment.getDeploymentName() ;
        if (!esbNames.add(deploymentName))
        {
            deployment.setDeploymentName(deploymentName + ",uid=" + System.identityHashCode(deployment)) ;
        }
    }
    
    private synchronized void removeDeploymentName(final String deploymentName)
    {
        esbNames.remove(deploymentName) ;
    }

    private List<File> getSubDeployments(DeploymentInfo di) {
        return (List<File>) di.context.get(SUB_DEPLOYMENTS);
    }
    
    /**
     * Sets the directory that will be used for generating ESWS wars.
     * @param dir The directory to be used.
     */
    public void setWarFilesDir(final String dir)
    {
        this.warFilesDir = dir;
    }

    static Map<String, String> getSchemas(final File file) throws DeploymentException
    {
        if (file == null)
        {
            return Collections.emptyMap();
        }
        
        if (file.isDirectory()) 
        {
            return getSchemasFromDirectory(file);
        }
        
        final String fileName = file.getName();
        if (fileName.endsWith(".esb"))
            return getSchemasFromArchive(file);
	        
        if (fileName.endsWith("-esb.xml"))
        {
            final File metaInf = file.getParentFile();
            final File deploymentRoot = metaInf.getParentFile();
            return getSchemasFromDirectory(deploymentRoot);
        }
        
        throw new DeploymentException("Unrecognized deployment : " + file + ".");
    }
    
    static Map<String, String> getSchemasFromDirectory(final File directory) throws DeploymentException
    {
        return getSchemasFromDir(directory, directory);
    }

    private static Map<String, String> getSchemasFromDir(final File root, final File directory) throws DeploymentException
    {
        if (directory != null)
        {
	        final Map<String, String> schemasMap = new HashMap<String, String>();
            
            if (directory.isDirectory())
            {
                final File[] files = directory.listFiles();
                for (File file : files)
                {
                    if (file.isDirectory())
                    {
                        schemasMap.putAll(getSchemasFromDir(root, file));
                    }
                    else if (isXsd(file.getName()))
                    {
                        try
                        {
                            final String filename = determineSchemaFilename(root, file);
                            schemasMap.put(filename, FileUtil.readTextFile(file));
                        } 
                        catch (final IOException e)
                        {
                            throw new DeploymentException("IOException while trying to read xsd '" + file.getName() + "'", e);
                        }
                    }
                }
            }
            return schemasMap;
        }
        
        return Collections.emptyMap();
    }
    
    private static String determineSchemaFilename(final File rootDir, final File file)
    {
        final String rootName = rootDir.getName();
        final String path = file.getPath();
        int idx = path.indexOf(rootName);
        if (idx != -1)
        {
            return path.substring(idx + rootName.length() + 1);
        }
        
        return file.getName();
    }
    
    private static boolean isXsd(final String fileName)
    {
        return fileName.endsWith(".xsd");
    }

    static Map<String, String> getSchemasFromArchive(final File archive) throws DeploymentException
    {
        if (archive == null || !archive.isFile() || !archive.getName().endsWith(".esb"))
        {
            return Collections.emptyMap();
        }
        
        final HashMap<String, String> schemas = new HashMap<String, String>();
        final ZipFile zip = getZipFile(archive);
                
        final Enumeration<? extends ZipEntry> entries = zip.entries();
        while (entries.hasMoreElements()) 
        {
            ZipEntry entry = entries.nextElement();
                    
            final String fileName = entry.getName();
            if (isXsd(fileName))
            {
                InputStream in = null;
                try
                {
                    in = zip.getInputStream(entry);
                    schemas.put(fileName, StreamUtils.readStreamString(in, "UTF-8"));
                } 
                catch (final IOException e)
                {
                    throw new DeploymentException("IOException while trying to read from file '" + fileName + "' from archive '" + archive + "'", e);
                }
                finally
                {
                    try { in.close(); } catch (IOException ignore) { log.error("Error while trying to close in stream for file '" + fileName + "'", ignore); }
                }
            }
        }
        return schemas;
    }

    private static ZipFile getZipFile(final File archive) throws DeploymentException
    {
        try
        {
            return new ZipFile(archive);
        } 
        catch (ZipException e)
        {
            throw new DeploymentException(e.getMessage(), e);
        } 
        catch (IOException e)
        {
            throw new DeploymentException(e.getMessage(), e);
        }
    }
}
