/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.deployers.mc;

import java.util.List;

import javax.management.ObjectName;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.listeners.war.Servlet;
import org.jboss.internal.soa.esb.publish.ContractReferencePublisher;
import org.jboss.soa.esb.lifecycle.LifecycleResourceManager;
import org.jboss.soa.esb.listeners.config.Configuration;
import org.jboss.soa.esb.listeners.config.ServicePublisher;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleController;

/**
 * EsbDeployment for the Microcontainer. 
 * <p/> 
 * 
 * Microcontainer will call the lifecycle methods:
 * <lu>
 *  <li><i>create</i> will create the ESB deployment.</i>
 *  <li><i>start</i> will start the ESB deployment.</i>
 *  <li><i>stop</i> will stop the ESB deployment.</i>
 *  <li><i>destroy</i> will destroy the ESB deployment.</i>
 * </lu>
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 */
public class EsbDeployment implements EsbDeploymentMBean
{
    private Logger log = Logger.getLogger(EsbDeployment.class);
    
    private final String jbossEsbXml;
    private String deploymentName;
    private List<ContractReferencePublisher> publishers;
    private List<Servlet> servlets;
    private ManagedLifecycleController controller;
    private final String mbeanName;
    
    public EsbDeployment(final String config, final String deploymentName, final String mbeanName) throws Exception
    {
        this.jbossEsbXml = config;
        this.deploymentName = deploymentName;
        this.mbeanName = mbeanName;
    }
    
    public String getJbossEsbXml()
    {
        return jbossEsbXml;
    }

    public String getJbossEsbXmlAsHtml()
    {
        return jbossEsbXml.replace("<", "&lt;").replace(">", "&gt;");
    }

    public void setPublishers(final List<ContractReferencePublisher> publishers)
    {
        this.publishers = publishers;
    }
    
    public void setServlets(final List<Servlet> servlets)
    {
 	   this.servlets = servlets;
    }
    
    public List<Servlet> getServlets()
    {
 	   return servlets;
    }

    public ManagedLifecycleController getController()
    {
        return controller;
    }

    public String getDeploymentName()
    {
        return deploymentName;
    }

    public void setDeploymentName(final String deploymentName)
    {
        this.deploymentName = deploymentName;
    }

    public List<ContractReferencePublisher> getPublishers()
    {
        return publishers;
    }

    public void create()
    {
        LifecycleResourceManager.getSingleton().associateDeployment(deploymentName) ;
        LifecycleResourceManager.deactivateHook() ;
    }

    public void start() throws Exception
    {    
        if (controller == null)
        {
        	log.info("Starting ESB Deployment '" + deploymentName + "'");
            controller = Configuration.create(jbossEsbXml, new ObjectName(mbeanName), publishers, servlets);
            controller.start();
        }
    }

	public boolean isStarted() {
		return (controller != null);
	}

    public void stop() throws Exception
    {        
        if (controller != null)
        {
        	log.info("Stopping '" + deploymentName + "'");
	        ServicePublisher.removeServicePublishers(controller);
	        controller.stop();
	        controller = null;
        }
    }

    public void destroy() throws Exception
    {
        log.info("Destroying '" + deploymentName + "'");
        
        LifecycleResourceManager.getSingleton().disassociateDeployment(deploymentName);
        LifecycleResourceManager.getSingleton().destroyResources();
    }
}
