/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.registry;

import java.util.List;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * Returns an Instance of the Registry.
 * 
 * @author kstam
 * 
 */
public class RegistryFactory
{
	private static Logger logger = Logger.getLogger(RegistryFactory.class);

   private static Registry singleton;
   /**
    * The current registry implementation.
    */
   private static Registry currentRegistry ;
   /**
    * The interceptors applied to the registry.
    */
   private static RegistryInterceptor[] interceptors ;

   /**
    * Set the singleton. Any existing instance will be overwritten.
    * 
    * @param registry
    */
   
   public static synchronized void setRegistry(Registry registry)
   {
      currentRegistry = registry;
      singleton = configureInterceptors(registry) ;
   }

/**
    * Get the singleton.
    * 
    * @return the singleton or <code>null</code> if one has not been created.
    */
   
    public static synchronized Registry getRegistrySingleton() {
        return currentRegistry;
    }

    /**
     * Obtain the current registry singleton or create a new instance if one
     * is not available. This will assign the singleton as well.
     * 
     * @return the singleton
     * @throws RegistryException
     */
    
    public static synchronized Registry getRegistry() throws RegistryException {
        if (singleton != null) {
            return singleton;
        }
        setRegistry(createRegistry());
        return singleton ;
    }

    /**
     * Create a new registry (singleton) or return a pre-existing instance.
     * 
     * @return the registry singleton
     * @throws RegistryException
     */
    
    public static synchronized Registry createRegistry()
           throws RegistryException
   {
      if (currentRegistry != null)
         return currentRegistry;
        
      Registry registry = null;
      String className = Configuration.getRegistryImplementationClass();
      logger.log(Level.DEBUG, "Going to load " + className);

      try
		{
            // instruct class loader to load the Registry Implementation
         Class registryClass = ClassUtil.forName(className, RegistryFactory.class);
         // Try to instance the Registry
         registry = (Registry) registryClass.newInstance();
      }
      catch (ClassNotFoundException cnfex)
      {
         throw new RegistryException("Registry Implementation=" + className
               + " not found", cnfex);
      }
      catch (Exception e)
      {
         throw new RegistryException("Invocation exception. "
               + e.getLocalizedMessage(), e);
      }

      currentRegistry = registry;
      
      return registry;
   }
   
   /**
    * Configure the interceptors around the registry
    * @return The intercepted registry.
    */
   private static Registry configureInterceptors(final Registry registry)
   {
      if (interceptors != null)
      {
         final int numInterceptors = interceptors.length ;
         for(int count = 0 ; count < numInterceptors ; count++)
         {
            interceptors[count].setRegistry(null) ;
         }
         interceptors = null ;
      }
      
      if (registry == null)
      {
          return null ;
      }
      
      final String registryInterceptorConfig = Configuration.getRegistryInterceptors() ;
      final String[] registryInterceptorNames = (registryInterceptorConfig == null ? null : registryInterceptorConfig.split(",")) ;
      final int numInterceptors = (registryInterceptorNames == null ? 0 : registryInterceptorNames.length) ;
      final RegistryInterceptor[] registryInterceptors = new RegistryInterceptor[numInterceptors + 1] ;
      
      Registry current = registry ;
      try
      {
          for(int count = numInterceptors ; count > 0 ; count--)
          {
              final Class interceptorClass ;
              try
              {
                  interceptorClass = ClassUtil.forName(registryInterceptorNames[count-1].trim(), RegistryFactory.class) ;
              }
              catch (final ClassNotFoundException cnfe)
              {
                  throw new RegistryException("Failed to instantiate registry interceptor, disabling all interceptors", cnfe) ;
              }
              if (!RegistryInterceptor.class.isAssignableFrom(interceptorClass))
              {
                  throw new RegistryException("Registry interceptor does not implement RegistryInterceptor interface: " + interceptorClass.getName()) ;
              }
              final Object interceptor ;
              try
              {
                  interceptor = interceptorClass.newInstance() ;
              }
              catch (final Throwable th)
              {
                  throw new RegistryException("Failed to instantiate registry interceptor", th) ;
              }
              final RegistryInterceptor registryInterceptor = (RegistryInterceptor)interceptor ;
              registryInterceptor.setRegistry(current) ;
              registryInterceptors[count] = registryInterceptor ;
              current = registryInterceptor ;
          }
      }
      catch (final RegistryException re)
      {
          for(int count = 1 ; count <= numInterceptors ; count++)
          {
              final RegistryInterceptor interceptor = registryInterceptors[count] ;
              if (interceptor != null)
              {
                  interceptor.setRegistry(null) ;
              }
          }
          logger.error("Failed to create registry interceptors, falling back to registry implementation", re) ;
          return registry ;
      }
      final HeadRegistryInterceptor headInterceptor = new HeadRegistryInterceptor() ;
      headInterceptor.setRegistry(current) ;
      registryInterceptors[0] = headInterceptor ;
      interceptors = registryInterceptors ;
      return headInterceptor ;
   }
   
   /**
    * Registry Interceptor that throws an exception if a registry is used after it has been disabled.
    * @author <a href='Kevin.Conner@jboss.com'>Kevin Conner</a>
    */
   private static final class HeadRegistryInterceptor extends AbstractRegistryInterceptor
   {
      public List<String> findAllServices() throws RegistryException
      {
         return getRegistry().findAllServices() ;
      }

      public EPR findEPR(final String serviceCategoryName, final String serviceName)
         throws RegistryException, ServiceNotFoundException
      {
         return getRegistry().findEPR(serviceCategoryName, serviceName) ;
      }

      public List<EPR> findEPRs(final String serviceCategoryName, final String serviceName)
         throws RegistryException, ServiceNotFoundException
      {
         return getRegistry().findEPRs(serviceCategoryName, serviceName) ;
      }

      public List<String> findServices(final String serviceCategoryName)
         throws RegistryException
      {
         return getRegistry().findServices(serviceCategoryName) ;
      }

      public void registerEPR(final String serviceCategoryName, final String serviceName,
         final String serviceDescription, final EPR epr, final String eprDescription)
         throws RegistryException
      {
          getRegistry().registerEPR(serviceCategoryName, serviceName,
            serviceDescription, epr, eprDescription) ;
      }

      public void unRegisterEPR(final String serviceCategoryName, final String serviceName,
         final EPR epr)
         throws RegistryException, ServiceNotFoundException
      {
          getRegistry().unRegisterEPR(serviceCategoryName, serviceName, epr) ;
      }

      public void unRegisterService(final String category, final String serviceName)
         throws RegistryException, ServiceNotFoundException
      {
          getRegistry().unRegisterService(category, serviceName) ;
      }
   }
}
