/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.internal.soa.esb.services.registry;

import java.net.URI;
import java.util.List;
import java.util.concurrent.BrokenBarrierException;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;

import junit.framework.TestCase;

import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.services.registry.Registry;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.RegistryInterceptor;
import org.jboss.soa.esb.services.registry.ServiceNotFoundException;


/**
 * Unit tests for the caching registry interceptor
 * 
 * @author <a href="mailto:Kevin.Conner@jboss.com">Kevin Conner</a>
 */
public class CachingRegistryInterceptorUnitTest extends TestCase
{
    /**
     * Tests of caching, intended to run within the current cache validity period
     */
    public void testCaching()
        throws Exception
    {
        final String category = "category" ;
        final String name = "name" ;
        final EPR epr1 = new EPR() ;
        final EPR epr2 = new EPR() ;
        final EPR epr3 = new EPR(URI.create("urn:test")) ;
        
        final CachingRegistryInterceptor interceptor = new CachingRegistryInterceptor() ;
        final CacheRegistryStatsInterceptor stats = new CacheRegistryStatsInterceptor() ;
        final MockRegistry registry = new MockRegistry() ;
        stats.setRegistry(registry) ;
        interceptor.setRegistry(stats) ;

        registry.registerEPR(category, name, "Description", epr1, "EPR description") ;
        
        // findAllServices passes through to registry
        assertEquals("findAllServices count", 0, stats.getFindAllServicesCount()) ;
        interceptor.findAllServices() ;
        assertEquals("findAllServices count", 1, stats.getFindAllServicesCount()) ;
        interceptor.findAllServices() ;
        assertEquals("findAllServices count", 2, stats.getFindAllServicesCount()) ;

        // findServices passes through to registry
        assertEquals("findServices count", 0, stats.getFindServicesCount()) ;
        interceptor.findServices(category) ;
        assertEquals("findServices count", 1, stats.getFindServicesCount()) ;
        interceptor.findServices(category) ;
        assertEquals("findServices count", 2, stats.getFindServicesCount()) ;

        // findEPR/findEPRs go to registry on first call, then cache.
        assertEquals("findEPRs count", 0, stats.getFindEPRsCount()) ;
        interceptor.findEPRs(category, name) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        interceptor.findEPRs(category, name) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        interceptor.findEPR(category, name) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertEquals("findEPR count", 0, stats.getFindEPRCount()) ;

        // registerEPR goes to registry and adds to cache
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertEquals("registerEPR count", 0, stats.getRegisterEPRCount()) ;
        final List<EPR> registerOrigEPRs = interceptor.findEPRs(category, name) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertNotNull("Original EPRs", registerOrigEPRs) ;
        assertEquals("Original EPR count", 1, registerOrigEPRs.size()) ;
        interceptor.registerEPR(category, name, "Description", epr2, "EPR description") ;
        assertEquals("registerEPR count", 1, stats.getRegisterEPRCount()) ;
        final List<EPR> duplicateEPRs = interceptor.findEPRs(category, name) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertNotNull("Duplicate EPRs", duplicateEPRs) ;
        assertEquals("Duplicate EPR count", 1, duplicateEPRs.size()) ;
        interceptor.registerEPR(category, name, "Description", epr3, "EPR description") ;
        assertEquals("registerEPR count", 2, stats.getRegisterEPRCount()) ;
        final List<EPR> registerNewEPRs = interceptor.findEPRs(category, name) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertNotNull("New EPRs", registerNewEPRs) ;
        assertEquals("New EPR count", 2, registerNewEPRs.size()) ;

        // unRegister goes to registry and removes from cache
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertEquals("unRegisterEPR count", 0, stats.getUnRegisterEPRCount()) ;
        final List<EPR> unRegisterOrigEPRs = interceptor.findEPRs(category, name) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertNotNull("Original EPRs", unRegisterOrigEPRs) ;
        assertEquals("Original EPR count", 2, unRegisterOrigEPRs.size()) ;
        interceptor.unRegisterEPR(category, name, epr3) ;
        assertEquals("unRegisterEPR count", 1, stats.getUnRegisterEPRCount()) ;
        final List<EPR> unRegisterNewEPRs = interceptor.findEPRs(category, name) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertNotNull("New EPRs", unRegisterNewEPRs) ;
        assertEquals("New EPR count", 1, unRegisterNewEPRs.size()) ;
        interceptor.unRegisterEPR(category, name, epr2) ;
        assertEquals("unRegisterEPR count", 2, stats.getUnRegisterEPRCount()) ;
        final List<EPR> unRegisterDuplicateEPRs = interceptor.findEPRs(category, name) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertNotNull("Duplicate EPRs", unRegisterDuplicateEPRs) ;
        assertEquals("Duplicate EPR count", 1, unRegisterDuplicateEPRs.size()) ;
        interceptor.unRegisterEPR(category, name, epr1) ;
        assertEquals("unRegisterEPR count", 3, stats.getUnRegisterEPRCount()) ;
        final List<EPR> finalEPRs = interceptor.findEPRs(category, name) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertNotNull("Final EPRs", finalEPRs) ;
        assertEquals("Final EPR count", 0, finalEPRs.size()) ;
        final EPR finalEPR = interceptor.findEPR(category, name) ;
        assertNull("Final EPR", finalEPR) ;
        assertEquals("findEPR count", 0, stats.getFindEPRCount()) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;

        // Need to fix MockRegistry as it does not correctly handle
        // service information.  We cannot use unRegisterService unless
        // there is an EPR present so we re-register the first one.
        // Also findEPR and findEPRs should throw ServiceNotFoundException
        // after unRegister but MockRegistry does not support it.
        registry.registerEPR(category, name, "Description", epr1, "EPR description") ;
        assertEquals("unRegisterService count", 0, stats.getUnRegisterServiceCount()) ;
        interceptor.unRegisterService(category, name) ;
        assertEquals("unRegisterService count", 1, stats.getUnRegisterServiceCount()) ;
    }

    /**
     * Test concurrent access to next interceptor in chain.
     * @throws Exception
     */
    public void testConcurrency()
        throws Exception
    {
        final int numServices = 5 ;
        final String category = "category" ;
        final String name = "name" ;
        
        final CachingRegistryInterceptor interceptor = new CachingRegistryInterceptor() ;
        final CacheRegistryConcurrencyInterceptor concurrency = new CacheRegistryConcurrencyInterceptor(new CyclicBarrier(numServices)) ;
        final MockRegistry registry = new MockRegistry() ;
        concurrency.setRegistry(registry) ;
        interceptor.setRegistry(concurrency) ;
        
        final String[] names = new String[numServices] ;
        for (int count = 0 ; count < numServices ; count++)
        {
            names[count] = name + count ;
            registry.registerEPR(category, names[count], "description", new EPR(), "EPR description") ;
        }
        final int numThreads = numServices*2 ;
        
        final CyclicBarrier barrier = new CyclicBarrier(numThreads) ;
        final ConcurrencyTest[] tests = new ConcurrencyTest[numThreads] ;
        for(int count = 0 ; count < numThreads ; count++)
        {
            tests[count] = new ConcurrencyTest(barrier, interceptor, category, names[count%numServices]) ;
            tests[count].start() ;
        }
        
        for(int count = 0 ; count < numThreads ; count++)
        {
            tests[count].join();
        }
        
        for(int count = 0 ; count < numThreads ; count++)
        {
            assertNull("Throwable occurred", tests[count].getThrowable()) ;
        }
        
        assertEquals("Registry findEPRs invocation", numServices, concurrency.getFindEPRsCount()) ;
        assertFalse("Barrier timeout", concurrency.isTimeout()) ;
    }

    private static final class CacheRegistryStatsInterceptor implements RegistryInterceptor
    {
        private Registry registry ;
        final AtomicInteger findAllServicesCount = new AtomicInteger() ;
        final AtomicInteger findEPRCount = new AtomicInteger() ;
        final AtomicInteger findEPRsCount = new AtomicInteger() ;
        final AtomicInteger findServicesCount = new AtomicInteger() ;
        final AtomicInteger registerEPRCount = new AtomicInteger() ;
        final AtomicInteger unRegisterEPRCount = new AtomicInteger() ;
        final AtomicInteger unRegisterServiceCount = new AtomicInteger() ;
        
        public List<String> findAllServices()
            throws RegistryException
        {
            findAllServicesCount.incrementAndGet() ;
            return registry.findAllServices() ;
        }

        public int getFindAllServicesCount()
        {
            return findAllServicesCount.get() ;
        }

        public EPR findEPR(final String serviceCategoryName, final String serviceName)
            throws RegistryException, ServiceNotFoundException
        {
            findEPRCount.incrementAndGet() ;
            return registry.findEPR(serviceCategoryName, serviceName) ;
        }

        public int getFindEPRCount()
        {
            return findEPRCount.get() ;
        }

        public List<EPR> findEPRs(final String serviceCategoryName, final String serviceName)
            throws RegistryException, ServiceNotFoundException
        {
            findEPRsCount.incrementAndGet() ;
            return registry.findEPRs(serviceCategoryName, serviceName) ;
        }

        public int getFindEPRsCount()
        {
            return findEPRsCount.get() ;
        }

        public List<String> findServices(final String serviceCategoryName)
            throws RegistryException
        {
            findServicesCount.incrementAndGet() ;
            return registry.findServices(serviceCategoryName) ;
        }

        public int getFindServicesCount()
        {
            return findServicesCount.get();
        }

        public void registerEPR(final String serviceCategoryName, final String serviceName,
            final String serviceDescription, final EPR epr, final String eprDescription)
            throws RegistryException
        {
            registerEPRCount.incrementAndGet() ;
            registry.registerEPR(serviceCategoryName, serviceName, serviceDescription, epr, eprDescription) ;
        }

        public int getRegisterEPRCount()
        {
            return registerEPRCount.get();
        }

        public void unRegisterEPR(final String serviceCategoryName,
            final String serviceName, final EPR epr)
            throws RegistryException, ServiceNotFoundException
        {
            unRegisterEPRCount.incrementAndGet() ;
            registry.unRegisterEPR(serviceCategoryName, serviceName, epr) ;
        }

        public int getUnRegisterEPRCount()
        {
            return unRegisterEPRCount.get();
        }

        public void unRegisterService(final String category, final String serviceName)
            throws RegistryException, ServiceNotFoundException
        {
            unRegisterServiceCount.incrementAndGet() ;
            registry.unRegisterService(category, serviceName) ;
        }

        public int getUnRegisterServiceCount()
        {
            return unRegisterServiceCount.get();
        }

        public void setRegistry(final Registry registry)
        {
            this.registry = registry ;
        }
    }

    private static final class ConcurrencyTest extends Thread
    {
        private final CyclicBarrier barrier ;
        private final Registry registry ;
        private final String category ;
        private final String name ;
        
        private Throwable throwable ;

        ConcurrencyTest(final CyclicBarrier barrier, final Registry registry, 
            final String category, final String name)
        {
            this.barrier = barrier ;
            this.registry = registry ;
            this.category = category ;
            this.name = name ;
        }

        public void run()
        {
            try
            {
                barrier.await() ;
                registry.findEPRs(category, name) ;
            }
            catch (final Throwable th)
            {
                throwable = th ;
            }
        }

        Throwable getThrowable()
        {
            return throwable ;
        }
    }

    private static final class CacheRegistryConcurrencyInterceptor implements RegistryInterceptor
    {
        private final CyclicBarrier barrier ;

        private volatile boolean timeout ;
        private Registry registry ;
        private final AtomicInteger findEPRsCount = new AtomicInteger() ;

        public CacheRegistryConcurrencyInterceptor(final CyclicBarrier barrier)
        {
            this.barrier = barrier ;
        }

        public List<String> findAllServices()
            throws RegistryException
        {
            return registry.findAllServices() ;
        }

        public EPR findEPR(final String serviceCategoryName, final String serviceName)
            throws RegistryException, ServiceNotFoundException
        {
            return registry.findEPR(serviceCategoryName, serviceName) ;
        }

        public List<EPR> findEPRs(final String serviceCategoryName, final String serviceName)
            throws RegistryException, ServiceNotFoundException
        {
            findEPRsCount.incrementAndGet() ;
            if (!timeout)
            {
                try
                {
                    barrier.await(10, TimeUnit.SECONDS) ;
                }
                catch (final TimeoutException te)
                {
                    timeout = true ;
                }
                catch (final InterruptedException ie)
                {
                    throw new RegistryException("Interrupted", ie) ;
                }
                catch (final BrokenBarrierException bbe)
                {
                    throw new RegistryException("Broken barrier", bbe) ;
                }
            }
            return registry.findEPRs(serviceCategoryName, serviceName) ;
        }

        public int getFindEPRsCount()
        {
            return findEPRsCount.get() ;
        }

        public boolean isTimeout()
        {
            return timeout ;
        }

        public List<String> findServices(final String serviceCategoryName)
            throws RegistryException
        {
            return registry.findServices(serviceCategoryName) ;
        }

        public void registerEPR(final String serviceCategoryName, final String serviceName,
            final String serviceDescription, final EPR epr, final String eprDescription)
            throws RegistryException
        {
            registry.registerEPR(serviceCategoryName, serviceName, serviceDescription, epr, eprDescription) ;
        }

        public void unRegisterEPR(final String serviceCategoryName,
            final String serviceName, final EPR epr)
            throws RegistryException, ServiceNotFoundException
        {
            registry.unRegisterEPR(serviceCategoryName, serviceName, epr) ;
        }

        public void unRegisterService(final String category, final String serviceName)
            throws RegistryException, ServiceNotFoundException
        {
            registry.unRegisterService(category, serviceName) ;
        }

        public void setRegistry(final Registry registry)
        {
            this.registry = registry ;
        }
    }
}
