/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.addressing.eprs.tests;

import static org.junit.Assert.assertTrue;

import java.io.File;
import java.net.URISyntaxException;
import java.sql.SQLException;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.couriers.PickUpOnlyCourier;
import org.jboss.soa.esb.addressing.eprs.JDBCEpr;
import org.jboss.soa.esb.addressing.util.DefaultReplyTo;
import org.jboss.soa.esb.addressing.MalformedEPRException;
import org.jboss.soa.esb.couriers.CourierFactory;
import org.jboss.soa.esb.couriers.CourierUtil;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.couriers.CourierTimeoutException;
import org.jboss.soa.esb.helpers.persist.JdbcCleanConn;
import org.jboss.soa.esb.helpers.persist.SimpleDataSource;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.testutils.HsqldbUtil;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Unit tests for file default reply to
 *
 * @author <a href="mailto:schifest@heuristica.com.ar">Esteban</a>
 */


public class DefaultJdbcReplyToEprUnitTest
{

	private static Class thisClass = DefaultJdbcReplyToEprUnitTest.class;
	static Logger _logger = Logger.getLogger(thisClass);
	
    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(thisClass);
    }
    
    static File WORKDIR;
    static
    {
    	String os = System.getProperty("os.name","").toLowerCase();
    	String dflt = (os.indexOf("win")>=0) ? "/temp": "/tmp";
    	WORKDIR	= new File(System.getProperty("java.io.tmpdir",dflt));
    }

	private static String _dbFileName = WORKDIR + "/defaultReplyEprDB";
	private static String mDbDriver ="org.hsqldb.jdbcDriver";
	private static String mDbUrl = "jdbc:hsqldb:" + (new File(_dbFileName)).toURI();
	private static String mDbUsername = "sa";
	private static String mDbPassword ="";
	private static Object server;
	private static JdbcCleanConn _dbConn; 
	
    @BeforeClass
    public static void runBeforeAllTests()
    {
    	_logger.info("@Starting HSQL database");
        try
        {
        	Class.forName(mDbDriver);
            server = HsqldbUtil.startHsqldb(_dbFileName, "defaultReplyEprDB");
            _dbConn = new JdbcCleanConn
  				(new SimpleDataSource(mDbDriver,mDbUrl,mDbUsername,mDbPassword));
            createMessageTable("foo");
            createMessageTable("foo_reply_table");
        }
        catch (Exception e)
        {
            _logger.error("No Database Available - Stop Testing", e);
            assertTrue(false);
        }
    }


    @AfterClass
    public static void runAfterAllTests() throws Exception
    {
    	_logger.info("_________________________________________");
    	try
        {
    		dropTable("foo");
            dropTable("foo_reply_table");
        	_dbConn.release();
        }
    	finally
    	{
    		HsqldbUtil.stopHsqldb(server);
    		_logger.info("Database Shutdown Complete");
    	}
    }

    
  @Test
    public void testJdbcReplyEpr() throws MalformedEPRException, CourierException, URISyntaxException, CourierTimeoutException {
  		_logger.info("_________________________________________");
    	_logger.info("testJdbcReplyEpr() invoked");
        //  Send a Message that will be picked up by a listener, and specify replyTo
        JDBCEpr toEpr = getEpr("foo");
        JDBCEpr replyToEpr = (JDBCEpr)DefaultReplyTo.getReplyTo(toEpr);

        String text_1 = "Outgoing";
        Message outgoingMsg = MessageFactory.getInstance().getMessage();
        outgoingMsg.getHeader().getCall().setTo(toEpr);
        outgoingMsg.getHeader().getCall().setReplyTo(replyToEpr);
        outgoingMsg.getBody().add(text_1.getBytes());
        CourierUtil.deliverMessage(outgoingMsg);

        // Mock a service that picks up the original message and replies
        JDBCEpr serviceEpr = getEpr("foo");
        PickUpOnlyCourier listener = CourierFactory.getPickupCourier(serviceEpr);
        Message received = listener.pickup(100);
        String text_2 = new String((byte[]) received.getBody().get());
        assertTrue(text_1.equals(text_2));
//        	assertTrue(replyToEpr.equals(received.getHeader().getCall().getReplyTo()));

        // now respond to replyTo
        text_2	+= " + processed by listener";
        Message response = MessageFactory.getInstance().getMessage();
        response.getHeader().getCall().setTo(received.getHeader().getCall().getReplyTo());
        response.getBody().add(text_2.getBytes());
        CourierUtil.deliverMessage(response);

        // try to pick up reply
        PickUpOnlyCourier waiter = CourierFactory.getPickupCourier(replyToEpr);
        Message finalMsg = waiter.pickup(100);
        assertTrue(text_2.equals(new String((byte[]) finalMsg.getBody().get())));

        _logger.info(text_2+"... and back from jdbc ReplyTo EPR");
        _logger.info("getDefaultReplyToEpr test succeeded for JDBC message transport");

    }
	private static void dropTable(String tableName) throws Exception
	{
			StringBuilder sb = new StringBuilder("drop table ")
				.append(tableName).append(" if exists");
			_dbConn.execUpdWait(_dbConn.prepareStatement(sb.toString()),1);
            _dbConn.commit();
    }
  	private static void createMessageTable(String tableName) throws Exception
  	{
  		try
  		{
  			dropTable(tableName);
  			JDBCEpr epr = getEpr(tableName);
  			StringBuilder sb = new StringBuilder("create table ").append(tableName)
  				.append("(").append(epr.getMessageIdColumn())	.append(" varchar")
  				.append(",").append(epr.getStatusColumn())		.append(" varchar")
  				.append(",").append(epr.getTimestampColumn())	.append(" bigint")
  				.append(",").append(epr.getDataColumn())		.append(" varchar")
  				.append(")")
  			;
  			_dbConn.execUpdWait(_dbConn.prepareStatement(sb.toString()),1);
            _dbConn.commit();
  		}
  		catch (SQLException e) 
  		{
  			_logger.error(e);
  			throw e;
  		}
  	}
  	
  	private static JDBCEpr getEpr(String tableName) throws URISyntaxException
  	{
		JDBCEpr epr = new JDBCEpr(mDbUrl,true,true);
		epr.setDriver	(mDbDriver);
		epr.setUserName	(mDbUsername);
		epr.setPassword	(mDbPassword);

		epr.setTableName		(tableName);
		epr.setMessageIdColumn	("message_id_col");
		epr.setStatusColumn		("status_col");
		epr.setDataColumn		("data_col");
		epr.setTimestampColumn	("stamp_col");

		return epr;
  	}
}
