/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.message;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.fail;

import java.net.URI;

import junit.framework.JUnit4TestAdapter;

import org.jboss.internal.soa.esb.couriers.MockCourier;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
/**
 * Tests for Call modifications while invoking through ServiceInvoker.
 *
 * @author <a href="mailto:kevin.conner@jboss.com">Kevin Conner</a>
 */
public class ServiceInvokerCallUnitTest
{
    private EPR deliverEPR ;
    private EPR failEPR ;
    private EPR pickupEPR ;
    private JMSEpr jmsDeliverEPR ;
    private MockCourier deliverCourier ;
    private MockCourier failCourier ;
    private Message responseMessage ;

    @Before
    public void setUp()
        throws Exception
    {
        MockCourierFactory.install() ;
        MockRegistry.install() ;

        deliverEPR = new JMSEpr(JMSEpr.QUEUE_TYPE, "test:deliver", "ConnectionFactory") ;
        failEPR = new JMSEpr(JMSEpr.QUEUE_TYPE, "test:fail", "ConnectionFactory") ;
        pickupEPR = new EPR(new URI("test:pickup")) ;
        jmsDeliverEPR = new JMSEpr(JMSEpr.QUEUE_TYPE, "deliver", "ConnectionFactory") ;

        deliverCourier = new MockCourier(true);
        failCourier = new MockCourier(false);
        responseMessage = MessageFactory.getInstance().getMessage() ;
        deliverCourier.pickupMessage = responseMessage ;

        MockRegistry.register("test", "deliver", deliverEPR, deliverCourier);
        MockRegistry.register("test", "fail", failEPR, failCourier);
        MockRegistry.register("test", "jmsdeliver", jmsDeliverEPR, deliverCourier);
        
        final EPR deadLetterEPR = new EPR(new URI("internal:DLQ")) ;
        MockRegistry.register(ServiceInvoker.INTERNAL_SERVICE_CATEGORY, ServiceInvoker.DEAD_LETTER_SERVICE_NAME, deadLetterEPR, deliverCourier);
    }
    
    @After
    public void tearDown()
        throws Exception
    {
        MockRegistry.uninstall() ;
        MockCourierFactory.uninstall() ;
    }
    
    @Test
    public void testNullSyncEPRs()
        throws Exception
    {
        final Message message = MessageFactory.getInstance().getMessage() ;
        
        final ServiceInvoker si = new ServiceInvoker("test", "jmsdeliver") ;
        final Message resp = si.deliverSync(message, 10) ;
        
        assertEquals("Response message", resp, responseMessage) ;
        
        assertNull("Call To EPR", message.getHeader().getCall().getTo()) ;
        assertNull("Call ReplyTo EPR", message.getHeader().getCall().getReplyTo()) ;
    }
    
    @Test
    public void testNonNullToSyncEPRs()
        throws Exception
    {
        final Message message = MessageFactory.getInstance().getMessage() ;
        message.getHeader().getCall().setTo(failEPR) ;
        
        final ServiceInvoker si = new ServiceInvoker("test", "jmsdeliver") ;
        final Message resp = si.deliverSync(message, 10) ;
        
        assertEquals("Response message", resp, responseMessage) ;
        
        assertEquals("Call To EPR", failEPR, message.getHeader().getCall().getTo()) ;
        assertNull("Call ReplyTo EPR", message.getHeader().getCall().getReplyTo()) ;
    }
    
    @Test
    public void testNonNullSyncEPRs()
        throws Exception
    {
        final Message message = MessageFactory.getInstance().getMessage() ;
        message.getHeader().getCall().setTo(failEPR) ;
        message.getHeader().getCall().setReplyTo(pickupEPR) ;
        
        final ServiceInvoker si = new ServiceInvoker("test", "jmsdeliver") ;
        final Message resp = si.deliverSync(message, 10) ;
        
        assertEquals("Response message", resp, responseMessage) ;
        
        assertEquals("Call To EPR", failEPR, message.getHeader().getCall().getTo()) ;
        assertEquals("Call ReplyTo EPR", pickupEPR, message.getHeader().getCall().getReplyTo()) ;
    }
    
    @Test
    public void testNullAsyncEPRs()
        throws Exception
    {
        final Message message = MessageFactory.getInstance().getMessage() ;
        
        final ServiceInvoker si = new ServiceInvoker("test", "jmsdeliver") ;
        si.deliverAsync(message) ;
        
        assertNull("Call To EPR", message.getHeader().getCall().getTo()) ;
        assertNull("Call ReplyTo EPR", message.getHeader().getCall().getReplyTo()) ;
    }
    
    @Test
    public void testNonNullToAsyncEPRs()
        throws Exception
    {
        final Message message = MessageFactory.getInstance().getMessage() ;
        message.getHeader().getCall().setTo(failEPR) ;
        
        final ServiceInvoker si = new ServiceInvoker("test", "jmsdeliver") ;
        si.deliverAsync(message) ;
        
        assertEquals("Call To EPR", failEPR, message.getHeader().getCall().getTo()) ;
        assertNull("Call ReplyTo EPR", message.getHeader().getCall().getReplyTo()) ;
    }
    
    @Test
    public void testNonNullAsyncEPRs()
        throws Exception
    {
        final Message message = MessageFactory.getInstance().getMessage() ;
        message.getHeader().getCall().setTo(failEPR) ;
        message.getHeader().getCall().setReplyTo(pickupEPR) ;
        
        final ServiceInvoker si = new ServiceInvoker("test", "jmsdeliver") ;
        si.deliverAsync(message) ;
        
        assertEquals("Call To EPR", failEPR, message.getHeader().getCall().getTo()) ;
        assertEquals("Call ReplyTo EPR", pickupEPR, message.getHeader().getCall().getReplyTo()) ;
    }
    
    @Test
    public void testFailNullSyncEPRs()
        throws Exception
    {
        final Message message = MessageFactory.getInstance().getMessage() ;
        
        final ServiceInvoker si = new ServiceInvoker("test", "fail") ;
        try
        {
            si.deliverSync(message, 10) ;
            fail("Expected MessageDeliveryException") ;
        }
        catch (final MessageDeliverException mde) {} // expected
        
        assertNull("Call To EPR", message.getHeader().getCall().getTo()) ;
        assertNull("Call ReplyTo EPR", message.getHeader().getCall().getReplyTo()) ;
    }
    
    @Test
    public void testFailNonNullToSyncEPRs()
        throws Exception
    {
        final Message message = MessageFactory.getInstance().getMessage() ;
        message.getHeader().getCall().setTo(deliverEPR) ;
        
        final ServiceInvoker si = new ServiceInvoker("test", "fail") ;
        try
        {
            si.deliverSync(message, 10) ;
            fail("Expected MessageDeliveryException") ;
        }
        catch (final MessageDeliverException mde) {} // expected
        
        assertEquals("Call To EPR", deliverEPR, message.getHeader().getCall().getTo()) ;
        assertNull("Call ReplyTo EPR", message.getHeader().getCall().getReplyTo()) ;
    }
    
    @Test
    public void testFailNonNullSyncEPRs()
        throws Exception
    {
        final Message message = MessageFactory.getInstance().getMessage() ;
        message.getHeader().getCall().setTo(deliverEPR) ;
        message.getHeader().getCall().setReplyTo(pickupEPR) ;
        
        final ServiceInvoker si = new ServiceInvoker("test", "fail") ;
        try
        {
            si.deliverSync(message, 10) ;
            fail("Expected MessageDeliveryException") ;
        }
        catch (final MessageDeliverException mde) {} // expected
        
        assertEquals("Call To EPR", deliverEPR, message.getHeader().getCall().getTo()) ;
        assertEquals("Call ReplyTo EPR", pickupEPR, message.getHeader().getCall().getReplyTo()) ;
    }
    
    @Test
    public void testFailNullAsyncEPRs()
        throws Exception
    {
        final Message message = MessageFactory.getInstance().getMessage() ;
        
        final ServiceInvoker si = new ServiceInvoker("test", "fail") ;
        si.deliverAsync(message) ;
        
        assertNull("Call To EPR", message.getHeader().getCall().getTo()) ;
        assertNull("Call ReplyTo EPR", message.getHeader().getCall().getReplyTo()) ;
    }
    
    @Test
    public void testFailNonNullToAsyncEPRs()
        throws Exception
    {
        final Message message = MessageFactory.getInstance().getMessage() ;
        message.getHeader().getCall().setTo(deliverEPR) ;
        
        final ServiceInvoker si = new ServiceInvoker("test", "fail") ;
        si.deliverAsync(message) ;
        
        assertEquals("Call To EPR", deliverEPR, message.getHeader().getCall().getTo()) ;
        assertNull("Call ReplyTo EPR", message.getHeader().getCall().getReplyTo()) ;
    }
    
    @Test
    public void testFailNonNullAsyncEPRs()
        throws Exception
    {
        final Message message = MessageFactory.getInstance().getMessage() ;
        message.getHeader().getCall().setTo(deliverEPR) ;
        message.getHeader().getCall().setReplyTo(pickupEPR) ;
        
        final ServiceInvoker si = new ServiceInvoker("test", "fail") ;
        si.deliverAsync(message) ;
        
        assertEquals("Call To EPR", deliverEPR, message.getHeader().getCall().getTo()) ;
        assertEquals("Call ReplyTo EPR", pickupEPR, message.getHeader().getCall().getReplyTo()) ;
    }
    
    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(ServiceInvokerCallUnitTest.class) ;
    }
}
