/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.routing.cbr;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.services.rules.RuleInfoBuilder;
import org.jboss.internal.soa.esb.services.rules.RuleServiceCallHelper;
import org.jboss.internal.soa.esb.services.rules.RuleServiceException;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.routing.MessageRouterException;
import org.jboss.soa.esb.services.routing.cbr.ContentBasedRouter;
import org.jboss.soa.esb.services.rules.RuleInfo;

/**
 * The Implementation of a rule based Content Router. Here we use JBossRules.
 * <p/>
 * We keep a HashMap of so called working memories for performance reasons.
 * <br><br>
 * @author kstam@redhat.com
 * 
 */
public class JBossRulesRouter implements ContentBasedRouter
{
	private static Logger logger = Logger.getLogger(JBossRulesRouter.class);
	
    private RuleServiceCallHelper ruleServiceCallHelper;

	/**
     * Route the message with a reference to the ruleSets supplied in the message.
     * Not implemented.
	 * 
	 * @param message - Message that needs routing.
	 */
	public List<String> route (Message message) throws MessageRouterException
	{
        logger.error("Not implemented");
		return null;
	}

	/**
	 * Route the message, using the given ruleSet (drl).
	 * 
	 * @param ruleSet -
	 *            String reference to a file which contains a ruleSet.
	 * @param ruleLanguage -
	 *            String reference to a file which contains a custom rule language
	 *            definition.
	 * @param message -
	 *            Message that needs routing.
	 * @param objectList -
	 *            List of objects that should be extracted from the message
	 *            object and which will be inserted into the Drools working 
	 *            memory.
	 * @return List<String> -
	 *            List of destinations
	 * 			
	 */
	public List<String> route (
			final String ruleSet, 
			final boolean ruleReload,
			final Message message, 
			final List<Object> objectList) throws MessageRouterException
	{
	    return route(message, objectList);
	}

	/**
	 * Route the message, where the routing rules are supplied as part of the
	 * message itself. 
	 * 
	 * @param ruleSet -
	 *            String reference to a file which contains a ruleSet.
	 * @param ruleLanguage -
	 *            String reference to a file which contains a custom rule language
	 *            definition.
	 * @param ruleReload -
	 *            if set to true, a ruleSet update should result in reloading the
	 *            ruleSet.
	 * @param message -
	 *            Message that needs routing.
	 * @param objectList - 
	 *			  a list with additional objects (typically pulled from the message) to be inserted into 
	 *            working memory.
	 * @return List<String> -
	 *            List of Service destinations.
	 */
	public List<String> route (
			final String ruleSet, 
			final String ruleLanguage,
			final boolean ruleReload, 
			Message message, 
			final List<Object> objectList ) throws MessageRouterException
	{
	    return route(message, objectList);
	}
	
	private List<String> route(Message message, final List<Object> objectList) throws MessageRouterException
    {
        try
        {
            final Map<String,Object> globals = new HashMap<String,Object>();
            List<String> destinations = new ArrayList<String>();
            globals.put("destinations", destinations);
            
            RuleInfoBuilder builder =  ruleServiceCallHelper.getRuleInfoBuilder();
            builder.globals(globals);
            builder.defaultFacts(objectList);
            final RuleInfo ruleInfo = builder.build();
            ruleServiceCallHelper.executeRulesService(message, ruleInfo);
        
            return destinations;
        } 
        catch (final RuleServiceException e)
        {
            throw new MessageRouterException(e.getMessage(), e);
        }
    }
	
	public void setConfigTree(final ConfigTree configTree) 
	{
	    try
        {
            this.ruleServiceCallHelper = new RuleServiceCallHelper(configTree);
        }
        catch (ConfigurationException e)
        {
            throw new IllegalStateException(e.getMessage(), e);
        }
	}
        
}
