/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.internal.soa.esb.services.rules;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.io.UnsupportedEncodingException;
import java.util.Properties;

import org.apache.log4j.Logger;
import org.drools.RuleBase;
import org.drools.RuleBaseFactory;
import org.drools.agent.RuleAgent;
import org.drools.compiler.DroolsParserException;
import org.drools.compiler.PackageBuilder;
import org.drools.compiler.PackageBuilderConfiguration;
import org.drools.compiler.PackageBuilderErrors;
import org.drools.decisiontable.InputType;
import org.drools.decisiontable.SpreadsheetCompiler;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * A helper class, it returns rulebases based on various methods of creating them.
 * <p/> 
 * 
 * @author jdelong@redhat.com
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 * 
 */
public class DroolsRuleBaseHelper {

	private static Logger logger = Logger.getLogger(DroolsRuleBaseHelper.class);
	
	private DroolsRuleBaseHelper()  {}
	
	/**
	 * Factory method that returns an instance of this class.
	 * </p>
	 * The current implementation returns a new instance of this
	 * class for every call.
	 * 
	 * @return {@link DroolsRuleBaseHelper}
	 */
	public static DroolsRuleBaseHelper getInstance()
	{
		return new DroolsRuleBaseHelper();
	}

	/**
	 * Creates a rulebase using rules and dsl from files
	 * <p/>
	 * 
	 * @param ruleFile - 
	 * 			file name which contains the Drools rules. Can be a file on the local
	 *  		filesystem, a file on the classpath or an URL. Must not be null.
	 * @param dsl - 
	 *			file containing the Drools Domain Specific Language (dsl)
	 * @throws DroolsParserException -
	 * 			if an exception occurs during Drools package building	
	 * @throws IOException -
	 * 			if an IOException occurs during Drools package building	
	 * @throws RuleServiceException -
	 * 			if the ruleFile cannot be located or if it was not possible to
	 * 			add the package to the RuleBase.
	 */
	public RuleBase createRuleBaseFromRuleFiles(
			final String ruleFile, 
			final String dsl)
			throws DroolsParserException, IOException, RuleServiceException, RuleServiceBuilderException
	{
		assertNotNull( ruleFile, "ruleFile" );
		
		// read in the rules
		InputStream rulesInputStream = getRulesInputStream( ruleFile );

		Reader ruleReader = getRulesReader( rulesInputStream );

		PackageBuilder builder = new PackageBuilder( new PackageBuilderConfiguration() );
		try 
		{
			if ( dsl == null ) 
			{
				builder.addPackageFromDrl(ruleReader);
			} 
			else 
			{
				InputStream dslInputStream = getRulesInputStream( dsl );
				Reader dslReader = getRulesReader( dslInputStream );
				if ( dslReader == null ) 
				{
					logger.error("Could not find dsl file [" + dsl + "]");
				} 
				else 
				{
					try 
					{
						builder.addPackageFromDrl(ruleReader, dslReader);
					} 
					finally 
					{
						safeClose(dslInputStream);
					}
				}
			}
		} 
		finally 
		{
			safeClose(rulesInputStream);
		}

		return getNewRuleBaseWithPackage( builder );
	}
	
	

	/**
	 * Reads the rules and dsl from files and returning as a string
	 * 
	 * @param ruleFile - 
	 * 			file name which contains the Drools rules. Can be a file on the local
	 *  		filesystem, a file on the classpath or an URL. Must not be null.
	 * @param dsl - 
	 *			file containing the Drools Domain Specific Language (dsl)
	 * @return String -
	 * 			String representation of the rules
	 * @throws IOException -
	 * 			if an IOException occurs during Drools package building	
	 * @throws RuleServiceException -
	 * 			if the ruleFile cannot be located or if it was not possible to
	 * 			add the package to the RuleBase.
	 * @throws RuleServiceException 
	 */
	public String getRulesAsString( 
			final String ruleFile, 
			final String dsl) throws IOException, RuleServiceException 
	{
		assertNotNull( ruleFile, "rulefile" );
		
		final String rules = getFileContents( ruleFile ); 
		return dsl == null ? rules : rules + getFileContents( dsl );
	}
	
	/**
	 * Reading the rules from a decision table.
	 * <p/>
	 * @param decisionTable -
	 * 			file name which contains the Drools decision table. Can be a file on the local
	 *  		filesystem, a file on the classpath or an URL. Must not be null.
	 * @throws DroolsParserException -
	 * 			
	 * @throws IOException -
	 * 			
	 * @throws RuleServiceException -
	 */
	public RuleBase createRuleBaseFromDecisionTable(
			final String decisionTable)
			throws DroolsParserException, IOException, RuleServiceException, RuleServiceBuilderException
	{
		assertNotNull( decisionTable, "decisionTable" );
		
		final String drl = getSpreadsheetRules(decisionTable);
		PackageBuilder builder = new PackageBuilder();
		builder.addPackageFromDrl( new StringReader(drl) );
		return getNewRuleBaseWithPackage( builder );
	}

	/**
	 * Reading the decision table and creating a drl
	 * </p>
	 * @throws RuleServiceException 
	 */
	public String getSpreadsheetRules(final String decisionTable) throws RuleServiceException 
	{
		InputStream inputStreamDT = getRulesInputStream(decisionTable);
		try 
		{
			return new SpreadsheetCompiler().compile(inputStreamDT, InputType.XLS);
		} 
		finally 
		{
			safeClose(inputStreamDT);
		}
	}

	/**
	 * This shows how rules are loaded up from a deployed package.
	 */
	public RuleAgent createRuleAgent(final String ruleAgentProperties) throws IOException, Exception 
	{
		return RuleAgent.newRuleAgent( "/" + ruleAgentProperties, new LogAgentEventListener() );
	}
	
	private String getFileContents( final String fileName ) throws RuleServiceException
	{
		InputStream inputStream = getRulesInputStream( fileName );
		try 
		{
    		return StreamUtils.readStreamString( inputStream, "UTF-8" );
		} 
		catch ( final UnsupportedEncodingException e)
		{
			throw new RuleServiceException("Could not read from file [" + fileName + "].", e);
		} 
		finally 
		{
			safeClose( inputStream );
		}
	}
	
	//	private instance methods
	
	private Reader getRulesReader(final InputStream inputStream) 
	{
		return new InputStreamReader( inputStream );
	}

	private InputStream getRulesInputStream(final String rulesFile) throws RuleServiceException 
	{
		InputStream resourceAsStream = ClassUtil.getResourceAsStream( "/" + rulesFile, getClass() );
		if ( resourceAsStream == null )
    		throw new RuleServiceException("Could not locate file [" + rulesFile + "], neither as a file on the local filesystem, on the classpath nor as a URL.");
		else
			return resourceAsStream;
	}
	
	private RuleBase getNewRuleBaseWithPackage( final PackageBuilder builder ) throws RuleServiceException, RuleServiceBuilderException
	{
		final PackageBuilderErrors errors = builder.getErrors();
		if ((errors != null) && (errors.size() > 0))
		{
			throw new RuleServiceException("Generation raised the following errors: " + errors) ;
		}
		RuleBase ruleBase = RuleBaseFactory.newRuleBase();
		try 
		{
			ruleBase.addPackage( builder.getPackage() );
		} 
		catch (final Exception ex) 
		// 	need to catch Exception as RuleBase.addPackage throws it
		{
			throw new RuleServiceException(ex.getMessage(), ex);
		}
		
		if ( builder.hasErrors() )
		{
			throw new RuleServiceBuilderException( "PackageBuilder generated errors: ", builder.getErrors() );
		}
		
		return ruleBase;
	}
	
	//	private static methods
	
	private static void safeClose( final InputStream is ) 
	{
		try 
		{
			is.close();
		} 
		catch (final IOException e) 
		{
			logger.warn("Caught an IOException while trying to close as inputstream.", e );
		} 
	}

	private static void assertNotNull( final String argumentValue, final String argumentName ) 
	{
		if( argumentValue == null )
			throw new NullPointerException("[" + argumentName + "] argument must not be null");
	}
	
}
