/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.rules;

import static org.jboss.soa.esb.listeners.ListenerTagNames.RULE_RELOAD_TAG;
import static org.jboss.soa.esb.services.rules.RuleServicePropertiesNames.CONTINUE;
import static org.jboss.soa.esb.services.rules.RuleServicePropertiesNames.DECISION_TABLE;
import static org.jboss.soa.esb.services.rules.RuleServicePropertiesNames.DISPOSE;
import static org.jboss.soa.esb.services.rules.RuleServicePropertiesNames.IMPL_CLASS;
import static org.jboss.soa.esb.services.rules.RuleServicePropertiesNames.RULE_AGENT_PROPERTIES;
import static org.jboss.soa.esb.services.rules.RuleServicePropertiesNames.STATEFUL;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.rules.RuleInfo;
import org.jboss.soa.esb.services.rules.RuleService;
import org.jboss.soa.esb.services.rules.StatefulRuleInfo;

import com.arjuna.common.util.propertyservice.PropertyManager;

/**
 * RuleServiceCallHelper is a class for calling
 * methods on a {@link RuleService} implementation.
 * </p>
 * 
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class RuleServiceCallHelper
{
    private static Logger logger = Logger.getLogger(RuleServiceCallHelper.class);
    
    /**
     * The default continue state expected.
     */
    private static final boolean DEFAULT_CONTINUE_STATE;
    
    /**
     * The {@link RuleService} implementation to use.
     */
    private RuleService ruleService;
    
    /**
     * The rule set.
     */
    private String ruleSet;
    
    /**
     * The optional rule language. (DSL)
     */
    private String ruleLanguage;
    
    /**
     * True if a ruleSet is being used. 
     */
    private boolean useRuleSet;

    /** 
     * The decision table to be use, if configured.
     */
    private String decisionTable;
    private boolean useDecisionTable;

    /** 
     * The rule agent to be use, if configured.
     */
    private String ruleAgent;
    private boolean useRuleAgent;
    
    /**
     * Should this be a stateful rules execution.
     */
    private boolean stateful;
    
    /**
     * Controll the reloading of rules.
     */
    private boolean ruleReload;

    public RuleServiceCallHelper(final ConfigTree config) throws ConfigurationException
	{
        final String ruleServiceImpl = config.getAttribute(IMPL_CLASS.getTagName(), DroolsRuleService.class.getName());
        try
        {
            ruleService = RuleServiceFactory.getRuleService( ruleServiceImpl );
            ruleService.setConfigTree( config );
        } 
        catch (RuleServiceException e)
        {
            throw new ConfigurationException("Could not create RuleService", e);
        }
        
        ruleSet = config.getAttribute(ListenerTagNames.RULE_SET_TAG);
        ruleLanguage = config.getAttribute(ListenerTagNames.RULE_LANGUAGE_TAG);
        
        String ruleReloadStr = config.getAttribute(ListenerTagNames.RULE_RELOAD_TAG);
        if (ruleReloadStr != null && "true".equals(ruleReloadStr)) 
        {
            ruleReload = true;
        }
        
        useRuleSet = ruleSet != null;
        if (ruleSet == null)
        {
            // Extract the decicion table from the configuration(if configured).
            decisionTable = config.getAttribute( DECISION_TABLE.getTagName() );
            useDecisionTable = decisionTable != null;
            
            if (useDecisionTable == false)
            {
                // Extract the ruleAgent from the configuration(if configured).
                ruleAgent = config.getAttribute( RULE_AGENT_PROPERTIES.getTagName() );
                useRuleAgent = true;
        
                if(logger.isDebugEnabled()) 
                {
                    if (ruleAgent != null && ruleReload) 
                    {
                        logger.debug("'" + RULE_RELOAD_TAG + "' is specified on the same configuration as a Rule Agent configuration is specified.  Ignoring the '" + RULE_RELOAD_TAG + "' configuration.");
                    }
                }
            }
        }
        
        stateful = Boolean.valueOf(config.getAttribute(STATEFUL.getTagName()));
	}
    
    public Message executeRulesService(Message message, final RuleInfo ruleInfo) throws RuleServiceException 
    {
        if (isStateful())
        {
            return executeStateful(ruleInfo, message);
        }
        else
        {
            return executeStateless(ruleInfo, message);
        }
    }
    
    public Message executeStateless(final RuleInfo ruleInfo, final Message message) throws RuleServiceException
    {
        if (useRuleSet)
        {
            return ruleService.executeStatelessRules(ruleInfo, message);
        }
        else if (useDecisionTable)
        {
            return ruleService.executeStatelessRulesFromDecisionTable(ruleInfo, message);
        }
        else if (useRuleAgent)
        {
            return ruleService.executeStatelessRulesFromRuleAgent(ruleInfo, message);
        }
        else
        {
            throw new RuleServiceException( "One of '" + ListenerTagNames.RULE_SET_TAG + "', '" + DECISION_TABLE.getTagName() + "', or ' " + RULE_AGENT_PROPERTIES.getTagName() + "'must be specified as properties in jboss-esb.xml");
        }
        
    }
    
    public Message executeStateful(final RuleInfo ruleInfo, final Message message) throws RuleServiceException
    {
        final boolean dispose = getDisposeProperty(message);
        final boolean explicitContinueState = getContinueProperty(message);
        final boolean continueState = explicitContinueState || getContinuePropertyOrDefault(message) ;
        StatefulRuleInfo statefulRuleInfo = new StatefulRuleInfoImpl(ruleInfo, dispose, continueState);
            
        if (explicitContinueState)
        {
            checkDisposePropertyIsSpecified(message);
            return ruleService.continueStatefulRulesExecution(statefulRuleInfo, message);
        }
        else
        {
            if (useRuleSet)
            {
                return ruleService.executeStatefulRules(statefulRuleInfo, message);
            }
            else if (useDecisionTable)
            {
                return ruleService.executeStatefulRulesFromDecisionTable(statefulRuleInfo, message);
            }
            else if (useRuleAgent)
            {
                return ruleService.executeStatefulRulesFromRuleAgent(statefulRuleInfo, message);
            }
            else
            {
                throw new RuleServiceException( "One of '" + ListenerTagNames.RULE_SET_TAG + "', '" + DECISION_TABLE.getTagName() + "', or ' " + RULE_AGENT_PROPERTIES.getTagName() + "'must be specified as properties in jboss-esb.xml");
            }
        }
    }
    
    public String determineRuleSource()
    {
        if (useRuleSet)
        {
            return ruleSet;
        }
        else if (useDecisionTable)
        {
            return decisionTable;
        }
        else 
        {
            return ruleAgent;
        }
    }
    
    public boolean isUseRuleSet()
    {
        return useRuleSet;
    }
    
    public boolean isUseDecisionTable()
    {
        return useDecisionTable;
    }
    
    public boolean isUseRuleAgent()
    {
        return useRuleAgent;
    }
    
    public boolean isStateful()
    {
        return stateful;
    }
    
    public String getRuleSet()
    {
        return ruleSet;
    }

    public String getRuleLanguage()
    {
        return ruleLanguage;
    }

    public String getDecisionTable()
    {
        return decisionTable;
    }

    public String getRuleAgent()
    {
        return ruleAgent;
    }

    public boolean isRuleReload()
    {
        return ruleReload;
    }

    public RuleInfoBuilder getRuleInfoBuilder()
    {
        final RuleInfoBuilder builder = new RuleInfoBuilder(determineRuleSource());
        builder.dslSource(ruleLanguage);
        builder.reload(ruleReload);
        return builder;
    }

    private static boolean getDisposeProperty(final Message message) throws RuleServiceException
    {
        Object dispose = message.getProperties().getProperty(DISPOSE.getTagName());
        return Boolean.TRUE.equals(dispose);
    }

    private static boolean getContinueProperty(final Message message) throws RuleServiceException
    {
        Object continueState = message.getProperties().getProperty(CONTINUE.getTagName());
        return Boolean.TRUE.equals(continueState);
    }

    private static boolean getContinuePropertyOrDefault(final Message message) throws RuleServiceException
    {
        Object continueState = message.getProperties().getProperty(CONTINUE.getTagName());
        if (continueState != null)
        {
            return Boolean.TRUE.equals(continueState);
        }
        else
        {
            return DEFAULT_CONTINUE_STATE;
        }
    }

    private static void checkDisposePropertyIsSpecified( final Message message ) throws RuleServiceException
    {
        Object dispose = message.getProperties().getProperty( DISPOSE.getTagName() );
        if ( dispose == null )
        {
            throw new RuleServiceException("The property [" + DISPOSE.getTagName() + "] must be specified when [" +
                    CONTINUE.getTagName() + "] is true. This is required as it is important that the rules working memory "+
                    " be disposed or memory leaks can occur.");
        }
    }
    
    static
    {
        final PropertyManager prop = ModulePropertyManager.getPropertyManager(ModulePropertyManager.RULES_MODULE);
        final String value = prop.getProperty(Environment.RULES_CONTINUE_STATE) ;
        DEFAULT_CONTINUE_STATE = Boolean.parseBoolean(value);
    }
}
    