/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.internal.soa.esb.services.rules;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import junit.framework.JUnit4TestAdapter;

import org.drools.RuleBase;
import org.jboss.internal.soa.esb.services.routing.cbr.Order;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.Counter;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.message.mapping.ObjectMapper;
import org.jboss.soa.esb.message.mapping.ObjectMappingException;
import org.jboss.soa.esb.services.rules.StatefulRuleInfo;
import org.jboss.soa.esb.util.ClassUtil;
import org.junit.Before;
import org.junit.Test;

/**
 * Unit test for {@link DroolsRuleService}
 * <p/>
 *
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class DroolsRuleServiceUnitTest
{
	private DroolsRuleService ruleService = new DroolsRuleService();

	private DroolsRuleBaseState ruleBaseState;
	private Message message;

	private Order order;

	private ArrayList<String> messagePathList;

	@Test
	public void executeStatelessRules()
	{
		Map<String,Object> globals = getGlobalsWithDestAndMessage();
		message = ruleBaseState.executeStatelessRules( message , globals, null );
		ArrayList<String> destinations = getDistinations( globals );
		assertTrue( destinations.size() == 1 );
	}

	@Test
    public void executeStatelessRulesWithRuleInfo() throws RuleServiceException
    {
	    RuleInfoBuilder builder = new RuleInfoBuilder("JBossESBRules.drl");
		Map<String,Object> globals = getGlobalsWithDestAndMessage();
	    builder.globals(globals);

        message = ruleService.executeStatelessRules(builder.build(), message);
        ArrayList<String> destinations = getDistinations( globals );
        assertTrue( destinations.size() == 1 );
    }

	@Test
	public void executeStatelessRulesFromDecisionTableReload() throws RuleServiceException
	{
		Map<String,Object> globals = getGlobalsWithDest();
		final String decisionTable = "RuleBaseHelper.xls";
		message = ruleService.executeStatelessRulesFromDecisionTable( decisionTable, true, message, globals, null );
		ArrayList<String> destinations = getDistinations( globals );
		assertTrue( destinations.size() == 1 );
	}

	@Test
    public void executeStatelessRulesFromDecisionTableReloadWithRuleInfo() throws RuleServiceException
    {
	    RuleInfoBuilder builder = new RuleInfoBuilder("RuleBaseHelper.xls");
        Map<String,Object> globals = getGlobalsWithDest();
	    builder.globals(globals);
	    builder.reload(true);

        message = ruleService.executeStatelessRulesFromDecisionTable(builder.build(), message);
        ArrayList<String> destinations = getDistinations( globals );
        assertTrue( destinations.size() == 1 );
    }

	@Test
	public void executeStatelessRulesFromDecisionTable() throws RuleServiceException
	{
		Map<String,Object> globals = getGlobalsWithDest();
		final String decisionTable = "RuleBaseHelper.xls";
		message = ruleService.executeStatelessRulesFromDecisionTable( decisionTable, false, message, globals, null );
		ArrayList<String> destinations = getDistinations( globals );
		assertTrue( destinations.size() == 1 );
	}

	@Test
    public void executeStatelessRulesFromDecisionTableWithRuleInfo() throws RuleServiceException
    {
	    RuleInfoBuilder builder = new RuleInfoBuilder("RuleBaseHelper.xls");
        Map<String,Object> globals = getGlobalsWithDest();
	    builder.globals(globals);
	    builder.reload(false);

        message = ruleService.executeStatelessRulesFromDecisionTable(builder.build(), message);
        ArrayList<String> destinations = getDistinations( globals );
        assertTrue( destinations.size() == 1 );
    }

	@Test
	public void executeStatfulRulesFromDecisionTableReload() throws RuleServiceException
	{
		Map<String,Object> globals = getGlobalsWithDest();
		final String decisionTable = "RuleBaseHelper.xls";
		message = ruleService.executeStatefulRulesFromDecisionTable( decisionTable, true, message, globals, null );
		ArrayList<String> destinations = getDistinations( globals );
		assertTrue( destinations.size() == 1 );
	}

	@Test
	public void executeStatfulRulesFromDecisionTable() throws RuleServiceException
	{
		Map<String,Object> globals = getGlobalsWithDest();
		final String decisionTable = "RuleBaseHelper.xls";
		message = ruleService.executeStatefulRulesFromDecisionTable( decisionTable, false, message, globals, null );
		ArrayList<String> destinations = getDistinations( globals );
		assertTrue( destinations.size() == 1 );
	}

	@Test
	public void executeStatefulRules() throws RuleServiceException
	{
		Map<String,Object> globals = getGlobalsWithDestAndMessage();
		message = ruleService.executeStatefulRules( "JBossESBRules.drl", null, true, message , globals, null );
		ArrayList<String> destinations = getDistinations( globals );
		assertTrue( destinations.size() == 1 );
	}

	@Test
	public void executeStatefulRulesDrl() throws RuleServiceException
	{
		Map<String,Object> globals = getGlobalsWithDestAndMessage();
		message = ruleService.executeStatefulRules( "JBossESBRules.drl", null, true, message, globals, null );
		ArrayList<String> destinations = getDistinations( globals );
		assertTrue( destinations.size() == 1 );
	}

	@Test
	public void executeStatelessRulesDrlWithDsl() throws RuleServiceException, ConfigurationException, UnsupportedEncodingException
	{
		Message msg = MessageFactory.getInstance().getMessage();
		InputStream resourceAsStream = ClassUtil.getResourceAsStream( "/" + "5KB_message.xml", getClass() );
		String contents = StreamUtils.readStreamString( resourceAsStream, "UTF-8" );
		msg.getBody().add( contents );
		Map<String,Object> globals = getGlobalsWithDest();
		boolean ruleReload = true;

		// first run
		long startTime = System.nanoTime();

		message = ruleService.executeStatelessRules( "RulesWithDsl.drl", "XPathLanguage.dsl", ruleReload, msg, globals, null );
		ArrayList<String> destinations = getDistinations( globals );
		assertTrue( destinations.size() == 1 );

		long procTime = System.nanoTime() - startTime;
		long firstRun = TimeUnit.NANOSECONDS.toMillis( procTime ) ;
		System.out.println( "Timed First run : " +  firstRun + "ms" );

		// second run
		startTime = System.nanoTime();

		message = ruleService.executeStatelessRules( "RulesWithDsl.drl", "XPathLanguage.dsl", ruleReload, msg, globals, null );
		procTime = System.nanoTime() - startTime;
		long secondRun = TimeUnit.NANOSECONDS.toMillis( procTime ) ;
		System.out.println( "Timed Second run : " + secondRun + "ms" );

		destinations = getDistinations( globals );
		assertTrue( destinations.size() == 2 );
	}

	@Test
	public void executeStatelessRulesDrlWithDslAndNamespaces() throws RuleServiceException, ConfigurationException, UnsupportedEncodingException
	{
		Message msg = MessageFactory.getInstance().getMessage();
		InputStream resourceAsStream = ClassUtil.getResourceAsStream( "/" + "5KBNS_message.xml", getClass() );
		String contents = StreamUtils.readStreamString( resourceAsStream, "UTF-8" );
		msg.getBody().add( contents );
		Map<String,Object> globals = getGlobalsWithDest();
		boolean ruleReload = true;

		message = ruleService.executeStatelessRules( "RulesWithDslNS.drl", "XPathLanguage.dsl", ruleReload, msg, globals, null );
		ArrayList<String> destinations = getDistinations( globals );
		assertEquals( 3 , destinations.size() );
	}

	@Test
	public void executeStatefulRulesContinueSession() throws RuleServiceException, ObjectMappingException
	{
		Message message = createMessageWithOrder( order );
		Map<String,Object> globals = getGlobalsWithMessage( message );
    	ArrayList<String> messagePathList = new ArrayList<String>();
        messagePathList.add("body.Order");
        messagePathList.add("body.Counter");

        List<Object> objectList = new ObjectMapper().createObjectList(message, messagePathList);

		// process message
		message = ruleService.executeStatefulRules( "JBossESBPricingRulesStateful.drl", null, true, message, globals, objectList );
        assertEquals( 20.0, order.getDiscount(), 0 );
        assertEquals( "20%" ,message.getBody().get("DiscountObject"));

        //	process message again with a counter instance
        objectList = new ObjectMapper().createObjectList(message, messagePathList);
		message = ruleService.continueStatefulRulesExecution( "JBossESBPricingRulesStateful.drl", true, message, globals, objectList );

        Counter counter = (Counter) message.getBody().get("Counter");

        assertEquals( 2 , counter.getCounter() );
	}

	@Test
    public void continueSessionWithEntryPoint() throws RuleServiceException, ObjectMappingException, InterruptedException
    {
        Message message = createMessageWithOrder(order);
        long timestamp = System.currentTimeMillis();
        message.getProperties().setProperty(Environment.MESSAGE_ENTRY_TIME, timestamp);

        final RuleInfoBuilder builder = new RuleInfoBuilder("PricingRulesStatefulEntryPoint.drl");
		builder.global("message", message);
		final ObjectMapper objectMapper = new ObjectMapper();
		builder.defaultFact(objectMapper.getObjectFromMessage(message, "body.Counter"));
		builder.defaultFact(message);

		// OrderEntryPoint that matches the entry-point name in PricingRulesStatfulEntryPoint.drl.
		builder.fact("OrderEntryPoint", objectMapper.getObjectFromMessage(message, "body.Order"));
        // process message
		StatefulRuleInfo statefulInfo = new StatefulRuleInfoImpl(builder.build(), true, true);
        message = ruleService.executeStatefulRules(statefulInfo, message);

        assertEquals( 20.0, order.getDiscount(), 0 );
        assertEquals( "20%" ,message.getBody().get("DiscountObject"));

    }

	@Test (expected = RuleServiceException.class)
    public void shouldThrowIfEntryPointDoesNotExistInSession() throws RuleServiceException, ObjectMappingException
    {
        Message message = createMessageWithOrder( order );
        long timestamp = System.currentTimeMillis();
        message.getProperties().setProperty(Environment.MESSAGE_ENTRY_TIME, timestamp);

        final ArrayList<String> messagePathList = new ArrayList<String>();
        messagePathList.add("body.Order");
        messagePathList.add("body.Counter");

        final RuleInfoBuilder builder = new RuleInfoBuilder("PricingRulesStatefulEntryPoint.drl");
        builder.global("message", message);
        final ObjectMapper objectMapper = new ObjectMapper();
		builder.defaultFact(objectMapper.getObjectFromMessage(message, "body.Counter"));

        // Non Existing entry-point name.
        builder.fact("Bajja", objectMapper.getObjectFromMessage(message, "body.Order"));
        // process message
		StatefulRuleInfo statefulInfo = new StatefulRuleInfoImpl(builder.build(), true, false);
        ruleService.executeStatefulRules(statefulInfo, message);
    }

	//	Test setup methods

	@Before
	public void setup() throws RuleServiceException
	{
		ruleBaseState = ruleService.getRuleBaseStateForFileBasedRules( "JBossESBRules.drl", null, true );
		message = MessageFactory.getInstance().getMessage();

		order = new Order();
        order.setQuantity(20);
        order.setUnitPrice( new BigDecimal("20.0") );

        messagePathList = new ArrayList<String>();
        messagePathList.add("body.Order");
        messagePathList.add("body.Counter");
	}

	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter( DroolsRuleServiceUnitTest.class );
	}

	@SuppressWarnings("unchecked")
	private ArrayList<String> getDistinations( final Map<String,Object> globals )
	{
		return (ArrayList<String>) globals.get( "destinations" );
	}

	private Map<String,Object> getGlobalsWithDestAndMessage()
	{
		Map<String,Object> globals = getGlobalsWithDest();
		globals.putAll( getGlobalsWithMessage( message ));
		return globals;
	}

	private Map<String,Object> getGlobalsWithDest()
	{
		Map<String,Object> globals = new HashMap<String,Object>();
		ArrayList<String> destinations =  new ArrayList<String>();
		globals.put("destinations", destinations );
		return globals;
	}

	private Map<String,Object> getGlobalsWithMessage( final Message message )
	{
		Map<String,Object> globals = new HashMap<String,Object>();
		globals.put("message", message );
		return globals;
	}

	private Message createMessageWithOrder( final Order order )
	{
		Message message = MessageFactory.getInstance().getMessage(MessageType.JAVA_SERIALIZED);
		message.getBody().add("Order", order);
		return message;
	}
}
