/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import org.jboss.soa.esb.actions.ContentBasedWiretap;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.services.rules.RuleServicePropertiesNames;

/**
 * Builder for CBR ConfigTree instances.
 * </p>
 * 
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class CBRConfigTreeBuilder
{
	//	mandatory
	private boolean reload;
 		
	//	optional 
	private String ruleFile;
	private String ruleLanguage;
	private List<String> messagePathList;
	private Map<String, List<String>> entryPointFacts;
	private boolean stateful;
	private String decisionTable;
	private String ruleAgent;
	private String ruleServiceImpl;
	
		
	public CBRConfigTreeBuilder( final boolean reload )
	{
		this.reload = reload;
	}
	
	public CBRConfigTreeBuilder ruleFile( final String ruleFile )
	{
		this.ruleFile = ruleFile;
		return this;
	}
	
	public CBRConfigTreeBuilder ruleLanguage( final String dsl )
	{
		this.ruleLanguage = dsl;
		return this;
	}
	
	public CBRConfigTreeBuilder decisionTable( final String decisionTable )
	{
		this.decisionTable = decisionTable;
		return this;
	}
	
	public CBRConfigTreeBuilder ruleAgent( final String ruleAgent )
	{
		this.ruleAgent = ruleAgent;
		return this;
	}
		
	public CBRConfigTreeBuilder messagePaths( List<String> messagePathList )
	{
		this.messagePathList = messagePathList;
		return this;
	}
	
	public CBRConfigTreeBuilder entryPoints( final Map<String, List<String>> entryPointFacts )
	{
		this.entryPointFacts = entryPointFacts;
		return this;
	}
	
	public CBRConfigTreeBuilder ruleServiceImpl( String ruleServiceImpl )
	{
		this.ruleServiceImpl = ruleServiceImpl;
		return this;
	}
	
	public CBRConfigTreeBuilder stateful( final boolean stateful )
	{
		this.stateful = stateful;
		return this;
	}
		
	public ConfigTree build()
	{
        ConfigTree configTree = new ConfigTree("cbr-config");
        
        configTree.setAttribute( ListenerTagNames.RULE_RELOAD_TAG, Boolean.toString( reload ) );
        
        if ( ruleFile != null )
            configTree.setAttribute( ListenerTagNames.RULE_SET_TAG, ruleFile );
        else if ( decisionTable != null )
            configTree.setAttribute( RuleServicePropertiesNames.DECISION_TABLE.getTagName(), decisionTable );
        else if ( ruleAgent != null )
            configTree.setAttribute( RuleServicePropertiesNames.RULE_AGENT_PROPERTIES.getTagName(), ruleAgent );
        else
        	throw new IllegalStateException("One of ruleFile, decisionTable or ruleAgent must be set");
        
        if (ruleLanguage != null)
	        configTree.setAttribute( ListenerTagNames.RULE_LANGUAGE_TAG, ruleLanguage);
        
            
        if ( messagePathList != null )
        {
        	for (String messagePath : messagePathList)
			{
        		ConfigTree objectPath = new ConfigTree( ContentBasedWiretap.OBJECT_PATH_TAG, configTree );
        		objectPath.setAttribute( ContentBasedWiretap.OBJECT_PATH, messagePath );
			}
        }
        
        if ( entryPointFacts != null )
        {
            Set<Entry<String, List<String>>> entrySet = entryPointFacts.entrySet();
            for (Entry<String, List<String>> entry : entrySet)
            {
                final String entryPointName = entry.getKey();
                List<String> value = entry.getValue();
                for (String path : value)
                {
            		ConfigTree objectPath = new ConfigTree( ContentBasedWiretap.OBJECT_PATH_TAG, configTree );
            		objectPath.setAttribute( ContentBasedWiretap.OBJECT_PATH, path );
            		objectPath.setAttribute( ContentBasedWiretap.ENTRY_POINT, entryPointName );
                }
            }
        }
        
        if ( stateful )
            configTree.setAttribute( RuleServicePropertiesNames.STATEFUL.getTagName(), Boolean.toString( stateful ) );
        
        if ( ruleServiceImpl != null )
            configTree.setAttribute( RuleServicePropertiesNames.IMPL_CLASS.getTagName(), ruleServiceImpl );
        
        return configTree;
	}
}
