/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2009
 */
package org.jboss.soa.esb.actions.soap.proxy;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import javax.management.JMException;
import javax.management.MBeanServer;
import javax.management.ObjectName;

import org.jboss.internal.soa.esb.publish.ContractInfo;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.mx.util.MBeanServerLocator;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.soap.AuthBASICWsdlContractPublisher;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.util.xml.DOMUtils;
import org.jboss.util.xml.DOMWriter;
import org.jboss.ws.metadata.umdm.ServerEndpointMetaData;
import org.jboss.wsf.spi.SPIProvider;
import org.jboss.wsf.spi.SPIProviderResolver;
import org.jboss.wsf.spi.deployment.Endpoint;
import org.jboss.wsf.spi.management.EndpointRegistry;
import org.jboss.wsf.spi.management.EndpointRegistryFactory;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * SOAPProxy WSDL loader.
 * 
 * @author dward at jboss.org
 */
public abstract class SOAPProxyWsdlLoader
{
	
	private ConfigTree config;
	private String address;
	private boolean rewriteHost;
	private String serviceCat;
	private String serviceName;
	private Puller puller = null;
	private File tempDir = null;
	
	private ContractInfo contract;
	private boolean rewriteLocation;
	private File primary_file;
	private List<File> cleanup_files = new ArrayList<File>();
	
	protected SOAPProxyWsdlLoader(ConfigTree config, String address, boolean rewriteHost)
	{
		this.config = config;
		this.address = address;
		this.rewriteHost = rewriteHost;
		ConfigTree parent_config = config.getParent();
		serviceCat = (parent_config != null ? parent_config.getAttribute(ListenerTagNames.SERVICE_CATEGORY_NAME_TAG) : null);
		serviceName = (parent_config != null ? parent_config.getAttribute(ListenerTagNames.SERVICE_NAME_TAG) : null);
	}
	
	public String getAddress()
	{
		return address;
	}
	
	public synchronized Puller getPuller()
	{
		if (puller == null)
		{
			puller = new Puller(config);
		}
		return puller;
	}
	
	private File getTempDir() throws IOException
	{
		if (tempDir == null)
		{
			synchronized (SOAPProxyWsdlLoader.class)
			{
				if (tempDir == null)
				{
					MBeanServer mbeanServer = MBeanServerLocator.locateJBoss();
					try
					{
						tempDir = (File)mbeanServer.getAttribute(new ObjectName("jboss.system:type=ServerConfig"), "ServerTempDir");
					}
					catch (JMException ignored) {}
					String tempName = SOAPProxyWsdlLoader.class.getSimpleName();
					if ( tempDir == null || !tempDir.exists() )
					{
						File tempFile = File.createTempFile(tempName + "-", ".tmp");
						tempDir = tempFile.getParentFile();
						tempFile.delete();
					}
					tempDir = new File(tempDir, tempName);
					tempDir.mkdirs();
				}
			}
		}
		return tempDir;
	}
	
	public File createTempFile(String prefix, String suffix) throws IOException
	{
		return File.createTempFile( prefix + "-", suffix, getTempDir() );
	}
	
	public ContractInfo load(boolean rewriteLocation) throws IOException
	{
		contract = new ContractInfo();
		this.rewriteLocation = rewriteLocation;
		String address = getAddress();
		String protocol = URI.create(address).getScheme();
		primary_file = createTempFile(protocol, ".wsdl");
		cleanup_files.add(primary_file);
		getPuller().pull(address, primary_file);
		primary_file = filterFile(address, primary_file, ".wsdl", protocol);
		// populate the primary member variables (mimeType + data)
		String mimeType = "text/xml";
		String data = null;
		InputStream is = null;
		try
		{
			URL url = getURL();
			if (url != null)
			{
				is = new BufferedInputStream( url.openStream() );
				data = StreamUtils.readStreamString(is, "UTF-8");
			}
		}
		finally
		{
			try { if (is != null) is.close(); } catch (Throwable t) {}
		}
		contract.setMimeType(mimeType);
		contract.setData(data);
		return contract;
	}
	
	private File filterFile(String url, File origFile, String suffix, String protocol) throws IOException
	{
		BufferedInputStream bis = null;
		Element wsdlElement;
		try
		{
			bis = new BufferedInputStream( origFile.toURL().openStream() );
			wsdlElement = DOMUtils.parse(bis);
		}
		finally
		{
			try { if (bis != null) bis.close(); } catch (Throwable t) {}
		}
		if ( findRelative(url, wsdlElement, protocol) )
		{
			File rewriteFile = createTempFile(protocol, suffix);
			cleanup_files.add(rewriteFile);
			BufferedOutputStream bos = null;
			try
			{
				bos = new BufferedOutputStream( new FileOutputStream(rewriteFile) );
				DOMWriter writer = new DOMWriter(bos);
				writer.setPrettyprint(true);
				writer.print(wsdlElement);
			}
			finally
			{
				try { if (bos != null) bos.close(); } catch (Throwable t) {}
			}
			origFile.delete();
			cleanup_files.remove(origFile);
			origFile = rewriteFile;
		}
		return origFile;
	}
	
	private boolean findRelative(String url, Element wsdlElement, String protocol) throws IOException
	{
		boolean rewrite = false;
		NodeList nlist = wsdlElement.getChildNodes();
		for (int i = 0; i < nlist.getLength(); i++)
		{
			Node childNode = nlist.item(i);
			if (childNode.getNodeType() == Node.ELEMENT_NODE)
			{
				Element childElement = (Element)childNode;
				String nodeName = childElement.getLocalName();
				if ( "import".equals(nodeName) || "include".equals(nodeName) )
				{
					Attr locationAttr = childElement.getAttributeNode("schemaLocation");
					if (locationAttr == null)
					{
						locationAttr = childElement.getAttributeNode("location");
					}
					if (locationAttr != null)
					{
						String location = fixRelative( url, locationAttr.getNodeValue() );
						File locationFile = createTempFile(protocol, ".tmp");
						cleanup_files.add(locationFile);
						getPuller().pull(location, locationFile);
						locationFile = filterFile(location, locationFile, ".tmp", protocol);
						String locationFileName = locationFile.getName();
						if (rewriteLocation)
						{
							String locationResource = getPuller().pull(locationFile);
							contract.putResource(locationFileName, locationResource);
							StringBuilder locationBuilder = new StringBuilder("@REQUEST_URL@?wsdl");
							locationBuilder.append("&resource=").append(locationFileName);
							if (serviceCat != null)
							{
								locationBuilder.append("&serviceCat=").append(serviceCat);
							}
							if (serviceName != null)
							{
								locationBuilder.append("&serviceName=").append(serviceName);
							}
							String contractProtocol = (!protocol.toLowerCase().equals("https") ? "http" : "https");
							locationBuilder.append("&protocol=").append(contractProtocol);
							locationAttr.setNodeValue( locationBuilder.toString() );
						}
						else
						{
							locationAttr.setNodeValue(locationFileName);		
						}
						rewrite = true;
					}
				}
				if ( findRelative(url, childElement, protocol) )
				{
					rewrite = true;
				}
			}
		}
		return rewrite;
	}
	
	String fixRelative(String base_url, String location)
	{
		location = location.replaceAll("/./", "/");
		if ( !location.startsWith("http://") && !location.startsWith("https://") )
		{
			while ( location.startsWith("./") )
			{
				location = location.substring( 2, location.length() );
			}
			if ( location.startsWith("/") )
			{
				if (rewriteHost)
				{
					String base_host = base_url.substring( 0, base_url.indexOf("/", base_url.indexOf("://")+3) );
					location = base_host + location;
				}
			}
			else
			{
				int count = 0;
				while ( location.startsWith("../") )
				{
					location = location.substring( 3, location.length() );
					count++;
				}
				String newLocation = base_url.substring( 0, base_url.lastIndexOf("/") );
				for (int c = 0; c < count; c++)
				{
					newLocation = newLocation.substring( 0, newLocation.lastIndexOf("/") );
				}
				location = newLocation + "/" + location;
			}
		}
		return location;
	}
	
	public URL getURL() throws MalformedURLException
	{
		return primary_file.toURL();
	}
	
	public void cleanup()
	{
		for (File file : cleanup_files)
		{
			file.delete();
		}
	}
	
	public static SOAPProxyWsdlLoader newLoader(ConfigTree config) throws ConfigurationException
	{
		SOAPProxyWsdlLoader loader;
		String address = config.getRequiredAttribute("wsdl");
		if ( address.startsWith("http://") || address.startsWith("https://") )
		{
			loader = new DefaultSOAPProxyWsdlLoader(config, address, true);
		}
		else if ( address.startsWith("file://") || address.startsWith("classpath://") )
		{
			loader = new DefaultSOAPProxyWsdlLoader(config, address, false);
		}
		else if ( address.startsWith("internal://") )
		{
			loader = new InternalSOAPProxyWsdlLoader(config, address);
		}
		else
		{
			throw new ConfigurationException("unrecognized wsdl address: " + address);
		}
		return loader;
	}
	
	static class DefaultSOAPProxyWsdlLoader extends SOAPProxyWsdlLoader
	{
		
		public DefaultSOAPProxyWsdlLoader(ConfigTree config, String address, boolean rewriteHost)
		{
			super(config, address, rewriteHost);
		}
		
	}
	
	private static class InternalSOAPProxyWsdlLoader extends SOAPProxyWsdlLoader
	{
		
		public InternalSOAPProxyWsdlLoader(ConfigTree config, String address)
		{
			super(config, address, true);
		}
		
		@Override
		public String getAddress()
		{
			String end_addr = super.getAddress();
			String end_name = end_addr.substring( 11, end_addr.length() );
			SPIProvider spi_prov = SPIProviderResolver.getInstance().getProvider();
			EndpointRegistryFactory end_reg_fact = spi_prov.getSPI(EndpointRegistryFactory.class);
			EndpointRegistry end_reg = end_reg_fact.getEndpointRegistry();
			Endpoint end = null;
			for ( ObjectName obj_name : end_reg.getEndpoints() )
			{
				//if ( end_name.equals(obj_name.getKeyProperty(Endpoint.SEPID_PROPERTY_ENDPOINT)) )
				if ( obj_name.toString().equals(end_name) )
				{
					end = end_reg.getEndpoint(obj_name);
					break;
				}
			}
			if (end != null)
			{
				ServerEndpointMetaData end_meta = end.getAttachment(ServerEndpointMetaData.class);
				if (end_meta == null)
				{
					throw new RuntimeException("cannot obtain metadata for internal endpoint: " + end_name);
				}
				return end_meta.getServiceMetaData().getWsdlLocation().toString();
			}
			else
			{
				throw new RuntimeException("unrecognized internal endpoint: " + end_name);
			}
		}
		
	}
	
	private static class Puller extends AuthBASICWsdlContractPublisher
	{
		
		private ConfigTree config;
		
		public Puller(ConfigTree config)
		{
			this.config = config;
		}
		
		@Override
		public Properties getActionProperties()
		{
			Properties props = new Properties();
			for ( String key : config.getAttributeNames() )
			{
				String value = config.getAttribute(key);
				if (value != null)
				{
					props.setProperty(key, value);
				}
			}
			return props;
		}
		
		public void pull(String url, File file) throws IOException
		{
			BufferedInputStream bis = null;
			BufferedOutputStream bos = null;
			try
			{
				// re-using the HTTPClient and AuthBASIC stuff...
				String data = getWsdl( url, config.getAttribute("wsdlCharset") );
				bis = new BufferedInputStream( new ByteArrayInputStream(data.getBytes()) );
				bos = new BufferedOutputStream( new FileOutputStream(file) );
				byte[] buf = new byte[1024];
				int read = 0;
				while ( (read = bis.read(buf)) != -1 )
				{
					bos.write(buf, 0, read);
				}
				bos.flush();
			}
			finally
			{	
				try { if (bos != null) bos.close(); } catch (Throwable t) {}
				try { if (bis != null) bis.close(); } catch (Throwable t) {}
			}
		}
		
		public String pull(File file) throws IOException
		{
			Reader reader = new BufferedReader( new FileReader(file) );
			try
			{
				return StreamUtils.readReader(reader);
			}
			finally
			{
				reader.close();
			}
		}
		
	}

}
