/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.services.soapui;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.wsdl.Part;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLEventWriter;
import javax.xml.stream.XMLStreamException;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.jboss.system.server.ServerConfigImplMBean;
import org.jboss.mx.util.MBeanProxyExt;

import org.apache.commons.httpclient.HttpClient;
import org.apache.log4j.Logger;
import org.apache.xmlbeans.XmlCursor;
import org.apache.xmlbeans.XmlObject;
import org.apache.xmlbeans.XmlOptions;
import org.jboss.internal.soa.esb.soap.OGNLUtils;
import org.jboss.internal.soa.esb.util.ESBProperties;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.http.HttpClientFactory;
import org.jboss.system.ServiceMBeanSupport;
import org.jboss.system.server.ServerConfig;
import org.milyn.Smooks;
import org.w3c.dom.Comment;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import com.eviware.soapui.SoapUI;
import com.eviware.soapui.impl.wsdl.WsdlInterface;
import com.eviware.soapui.impl.wsdl.WsdlOperation;
import com.eviware.soapui.impl.wsdl.WsdlProject;
import com.eviware.soapui.impl.wsdl.support.soap.SoapMessageBuilder;
import com.eviware.soapui.impl.wsdl.support.xsd.SampleXmlUtil;
import com.eviware.soapui.model.iface.MessagePart;
import com.eviware.soapui.model.iface.Operation;
import com.eviware.soapui.settings.WsdlSettings;


/**
 * Soap UI Soap Client Service MBean.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class SoapUIClientService extends ServiceMBeanSupport implements SoapUIClientServiceMBean {

    private static final String SOAPUI_CLONE_COMMENT = " repetitions:";

    private static final String IS_CLONE_ATTRIB = "is-clone";
    private static Logger logger = Logger.getLogger(SoapUIClientService.class);
    private Map<String, WsdlInterface[]> wsdls = new HashMap<String, WsdlInterface[]>();
    private DocumentBuilderFactory docBuilderFactory ;
    private SmooksCache smooksCache;
    private ESBProperties properties;
    private static final String CLONED_POSTFIX = " - cloned";
    /**
     * The SoapUI property file.
     */
    private String propertyFile ;
    /**
     * The name of the generated property file.
     */
    private static final String SOAP_UI_PROPERTY_FILE = "soapui-settings.xml" ;
    /**
     * The additional schema resources.
     */
    private String schemas ;

    private String serverDataDir;
    /**
     * The name of the SoapUI schema directory.
     */
    private static final String SOAP_UI_SCHEMA_DIRECTORY = "soapui-schemas" ;
    /**
     * The name of the property specifying the created SoapUI schema directory.
     */
    private static final String PROPERTY_ESB_SCHEMA_DIRECTORY = "jboss.esb.soapui.schema" ;

    /**
     * Public default constructor.
     */
    public SoapUIClientService() throws ConfigurationException {
        properties = new ESBProperties("/soapui-client.sar.properties");

        docBuilderFactory = DocumentBuilderFactory.newInstance();
        docBuilderFactory.setNamespaceAware(true);
        int smooksLRUCacheSize = properties.getIntProperty("smooks.lru.cache.size", 30);
        smooksCache = new SmooksCache(smooksLRUCacheSize);
    }

    protected void startService() throws Exception {
        super.startService();
        
        if (propertyFile != null) {
            File dataDir;
            if (serverDataDir != null) {
                
                dataDir = new File(serverDataDir);
            }
            else {
                final ServerConfig serverConfig = (ServerConfig) MBeanProxyExt.create(ServerConfig.class, ServerConfigImplMBean.OBJECT_NAME);
                dataDir = serverConfig.getServerDataDir() ;
            }
            
            if (schemas != null)
            {
                initialiseSchemas(dataDir) ;
            }
            
            final File soapUIPropertyFile = new File(dataDir, SOAP_UI_PROPERTY_FILE) ;
            
            final File baseFile = new File(propertyFile) ;
            final InputStream xmlPropertyIS = getInputStream(baseFile);

            
            try {
                final FileOutputStream fos = new FileOutputStream(soapUIPropertyFile) ;
                XMLHelper.replaceSystemProperties(XMLHelper.getXMLStreamReader(xmlPropertyIS),
                    XMLHelper.getXMLStreamWriter(fos)) ;
            } finally {
                xmlPropertyIS.close() ;
            }
            
            SoapUI.initSettings(soapUIPropertyFile.getAbsolutePath()) ;
        }
    }
    
    @Override
    protected void stopService() throws Exception
    {
        smooksCache.clean();
        if (propertyFile != null)
        {
            File dataDir;
            if (serverDataDir != null) {
                dataDir = new File(serverDataDir);
            }
            else
            {
                final ServerConfig serverConfig = (ServerConfig) MBeanProxyExt.create(ServerConfig.class, ServerConfigImplMBean.OBJECT_NAME);
                dataDir = serverConfig.getServerDataDir() ;
            }
            final File schemaDir = new File(dataDir, SOAP_UI_SCHEMA_DIRECTORY) ;
            if (schemaDir.exists())
            {
                deleteFiles(schemaDir) ;
            }
        }
        
        // TODO Auto-generated method stub
        super.stopService();
    }
    
    private void initialiseSchemas(final File dataDir)
        throws IOException
    {
        final String[] schemaResources = schemas.split("[, ]") ;
        if (schemaResources != null)
        {
            final File schemaDir = new File(dataDir, SOAP_UI_SCHEMA_DIRECTORY) ;
            schemaDir.mkdir();
            
            for(String schema: schemaResources)
            {
                final File schemaFile = new File(schema) ;
                final InputStream is = getInputStream(schemaFile) ;
                try
                {
                    writeToFile(is, schemaDir, schemaFile.getName()) ;
                }
                finally
                {
                    is.close() ;
                }
            }
            System.setProperty(PROPERTY_ESB_SCHEMA_DIRECTORY, schemaDir.getAbsolutePath()) ;
        }
    }
    
    private void deleteFiles(final File file)
    {
        if (file.isDirectory())
        {
            final File[] files = file.listFiles() ;
            for(File child: files)
            {
                deleteFiles(child) ;
            }
        }
        file.delete() ;
    }
    
    private void writeToFile(final InputStream is, final File dir, final String name)
        throws IOException
    {
        final File output = new File(dir, name) ;
        final byte[] buffer = new byte[256] ;
        final FileOutputStream fos = new FileOutputStream(output) ;
        try
        {
            for(;;)
            {
                final int count = is.read(buffer) ;
                if (count <= 0)
                {
                    break ;
                }
                fos.write(buffer, 0, count) ;
            }
        }
        finally
        {
            fos.close() ;
        }
    }
    
    private InputStream getInputStream(final File file)
        throws IOException
    {
        if (!file.isAbsolute()) {
            final URL resourceURL = Thread.currentThread().getContextClassLoader().getResource(file.getPath()) ;
            return  resourceURL.openStream() ;
        } else {
            return new FileInputStream(file) ;
        }
    }
    
    /**
     * Get the property file.
     * @return The name of the property file being used.
     */
    public String getPropertyFile()
    {
        return propertyFile ;
    }
    /**
     * Set the property file.
     * @param propertyFile The name of the property file being used.
     */
    public void setPropertyFile(final String propertyFile)
    {
        this.propertyFile = propertyFile ;
    }
    
    /**
     * Get the additional schema resources.
     * @return The additional schema resources.
     */
    public String getSchemas()
    {
        return schemas ;
    }
    
    /**
     * Set the additional schemes.
     * @param schemas The additional schema resources to setup for SoapUI.
     * This needs support through the soapui-settings.xml file
     * @see setPropertyFile
     */
    public void setSchemas(final String schemas)
    {
        this.schemas = schemas ;
    }

    /**
     * Build a SOAP request for the specified operation on the specified WSDL.
     *
     * @param wsdl            WSDL URL.
     * @param operation       Operation name.
     * @param params          Message parameter map.
     * @param httpClientProps {@link org.apache.commons.httpclient.HttpClient} creation properties.
     * @param smooksResource  {@link org.milyn.Smooks} transformation configuration resource.  This is the actual
     *                        Smooks resource configuration XML, not a file name.
     *                        Null if no transformations are to be performed on the SOAP message before serializing it
     *                        for return.
     * @param soapNs 		  optional SOAP namespace
     * @return The SOAP Message.
     * @throws IOException Failed to load WSDL.
     */
    public String buildRequest(String wsdl, String operation, Map params, Properties httpClientProps, String smooksResource, String soapNs) throws IOException, UnsupportedOperationException, SAXException {
        Operation operationInst = getOperation(wsdl, operation, httpClientProps);
        String requestTemplate = operationInst.getRequestAt(0).getRequestContent();

        return buildSOAPMessage(requestTemplate, params, smooksResource, soapNs);
    }
    
    /**
     * Use soapUI to build a SOAP response for the specified operation on the specified WSDL.
     *
     * @param wsdl            WSDL URL 
     * @param operation       Operation name.
     * @param params          Message parameter map.
     * @param httpClientProps {@link org.apache.commons.httpclient.HttpClient} creation properties.
     * @param smooksResource  {@link org.milyn.Smooks} transformation configuration resource file.
     *                        Null if no transformations are to be performed on the SOAP message before serializing it
     *                        for return.
     * @param soapNs 		  optional SOAP namespace
     * @return The SOAP Message.
     * @throws IOException                   Failed to load WSDL.
     * @throws UnsupportedOperationException Operation not supported on specified WSDL.
     * @throws SAXException                  Failed to parse the SOAP UI generated request message.
     */
    public String buildResponse(String wsdl, String operation, Map params, Properties httpClientProps, String smooksResource, String soapNs) throws IOException, UnsupportedOperationException, SAXException {
    	Operation operationInst = getOperation(wsdl, operation, httpClientProps);
        WsdlOperation wsdlOperation = (WsdlOperation)operationInst;
        String responseTemplate = wsdlOperation.createResponse(true);
        return buildSOAPMessage(responseTemplate, params, smooksResource, soapNs);   	
    }
    
    
    public String buildFault(String wsdl, String operation, String faultName, Map params, Properties httpClientProps, String smooksResource, String soapNs) throws IOException, UnsupportedOperationException, SAXException {
    	String faultDetail = null;
    	Operation operationInst = getOperation(wsdl, operation, httpClientProps);
        WsdlOperation wsdlOperation = (WsdlOperation)operationInst;
        MessagePart[] faultParts = wsdlOperation.getFaultParts();
        params.put("Fault.faultcode","soapenv:server");
	    SoapMessageBuilder soapMessageBuilder = wsdlOperation.getInterface().getMessageBuilder();
	    String faultTemplate = "<soapenv:Envelope xmlns:soapenv=\"http://schemas.xmlsoap.org/soap/envelope/\">"
	                            + "<soapenv:Body><soapenv:Fault><faultcode>?</faultcode><faultstring>?</faultstring><detail>";
		XmlObject detail = XmlObject.Factory.newInstance();
        for (int i = 0 ; i < faultParts.length; i++) {
        	MessagePart.FaultPart faultPart = (MessagePart.FaultPart)faultParts[i];
        	if (faultPart.getName().equalsIgnoreCase(faultName)) {
        		SampleXmlUtil generator = new SampleXmlUtil( false );
        		generator.setExampleContent( false );
        		generator.setTypeComment( false );
        		XmlCursor cursor = detail.newCursor();   			
    			cursor.toFirstContentToken();
    			generator.setTypeComment( true );
    			generator.setIgnoreOptional(wsdlOperation.getInterface().getSettings().getBoolean( WsdlSettings.XML_GENERATION_ALWAYS_INCLUDE_OPTIONAL_ELEMENTS ) );
    			for( Part part : faultPart.getWsdlParts() ) {
    				try {
						soapMessageBuilder.createElementForPart(part, cursor, generator);
					} catch (Exception e) {
						logger.error("Unable to create soap fualt template", e);
					}
        		}
        	}
        }
        faultTemplate = faultTemplate + detail.xmlText( new XmlOptions().setSaveAggressiveNamespaces().setSavePrettyPrint())
                        + "</detail></soapenv:Fault></soapenv:Body></soapenv:Envelope>";
        faultDetail = buildSOAPMessage(faultTemplate, params, smooksResource, soapNs);        
        return faultDetail;         	
    }

    /**
     * Get the 1st endpoint from the specified WSDL.
     *
     * @param wsdl            WSDL URL.
     * @param httpClientProps {@link org.apache.commons.httpclient.HttpClient} creation properties.
     * @return The operation endpoint URL.
     * @throws IOException Failed to load WSDL.
     */
    public String getEndpoint(String wsdl, Properties httpClientProps) throws IOException {
        WsdlInterface[] wsdlInterfaces = getWsdlInterfaces(wsdl, httpClientProps);

        return wsdlInterfaces[0].getEndpoints()[0];
    }

    private WsdlInterface[] getWsdlInterfaces(String wsdl, Properties httpClientProps) throws IOException {
        try {
            WsdlInterface[] wsdlInterfaces = wsdls.get(wsdl);
            if (wsdlInterfaces == null) {
                WsdlProject wsdlProject = new WsdlProject();
                HttpClient httpClient = HttpClientFactory.createHttpClient(httpClientProps);

                try {
                   wsdlInterfaces = wsdlProject.importWsdl(wsdl, true, new EsbWsdlLoader(wsdl, httpClient));
                } finally {
                    HttpClientFactory.shutdown(httpClient);
                }
                wsdls.put(wsdl, wsdlInterfaces);
            }
            return wsdlInterfaces;
        } catch (Exception e) {
            IOException ioe = new IOException("Failed to import WSDL '" + wsdl + "'.");
            ioe.initCause(e);
            throw ioe;
        }
    }

    private Operation getOperation(String wsdl, String operation, Properties httpClientProps) throws IOException, UnsupportedOperationException {
        WsdlInterface[] wsdlInterfaces = getWsdlInterfaces(wsdl, httpClientProps);

        for (WsdlInterface wsdlInterface : wsdlInterfaces) {
            Operation operationInst = wsdlInterface.getOperationByName(operation);

            if (operationInst != null) {
                return operationInst;
            }
        }
        
        // Try clearing WSDL cache, WSDL may have updated 
        wsdls.remove(wsdl);
        wsdlInterfaces = getWsdlInterfaces(wsdl, httpClientProps);

        for (WsdlInterface wsdlInterface : wsdlInterfaces) {
            Operation operationInst = wsdlInterface.getOperationByName(operation);

            if (operationInst != null) {
                return operationInst;
            }
        }
        
        throw new UnsupportedOperationException("Operation '" + operation + "' not supported by WSDL '" + wsdl + "'.");
    }

    private String buildSOAPMessage(String soapMessageTemplate, Map params, String smooksResource, String soapNs) throws IOException, SAXException {
        Document messageDoc = getDocument(soapMessageTemplate) ;

        Element docRoot = messageDoc.getDocumentElement();

        // Purposely not using log levels to output because I want to make
        // it as easy as possible for the developer to dump the SOAP during dev. They
        // just need to set "dumpSOAP" in the param Map....
        boolean dumpSOAP = params.containsKey("dumpSOAP");
        if(dumpSOAP) {
            dumpSOAP("SOAP Template (Unexpanded):", docRoot);
        }

        expandMessage(docRoot, params);

        if(dumpSOAP) {
            dumpSOAP("SOAP Template (Expanded):", docRoot);
        }

        injectParameters(docRoot, params, soapNs);

        if(smooksResource != null) {
            applySmooksTransform(smooksResource, messageDoc);
        }

        if(dumpSOAP) {
            dumpSOAP("SOAP Message (Populated Template):", docRoot);
        }
        
        try
        {
            final StringWriter sw = new StringWriter() ;
            final XMLEventWriter writer = XMLHelper.getXMLEventWriter(sw) ;
            XMLHelper.readDomNode(docRoot, writer, true) ;
	        return sw.toString();
        }
        catch (final XMLStreamException xmlse)
        {
            final IOException ioe = new IOException("Failed to serialize the output SOAP message") ;
            ioe.initCause(xmlse) ;
            throw ioe ;
        }
    }

    private Document getDocument(String soapMessageTemplate) throws IOException {
        try {
            final XMLEventReader reader = XMLHelper.getXMLEventReader(new StringReader(soapMessageTemplate)) ;
            return XMLHelper.createDocument(reader) ;
        } catch (final Exception ex) {
            final IOException ioe = new IOException("Failed to deserialize the SOAP message template") ;
            ioe.initCause(ex) ;
            throw ioe ;
        }
    }

    private void dumpSOAP(String message, Element docRoot) {
        System.out.println("------------------------------------------------------------------------------------------------------------------------------------------");
        System.out.println(message + "\n");
        try {
            final XMLEventWriter writer = XMLHelper.getXMLEventWriter(new StreamResult(System.out)) ;
            XMLHelper.readDomNode(docRoot, writer, false) ;
        } catch (Exception e) {
            logger.error("Unable to dump SOAP.", e);
        }
        System.out.println("------------------------------------------------------------------------------------------------------------------------------------------");
    }

    private void applySmooksTransform(String smooksResource, Document messageDoc) throws IOException, SAXException {
        if(smooksResource != null) {
            SmooksEntry smooksEntry = smooksCache.get(smooksResource);
            try {
                final Smooks smooks = smooksEntry.getSmooks();
                smooks.filter(new DOMSource(messageDoc), new DOMResult(), smooks.createExecutionContext());        
            }
            finally
            {
                smooksCache.release(smooksEntry);
            }
        }
    }

    /**
     * Expand the message to accommodate data collections.
     * <p/>
     * It basically just clones the message where appropriate.
     *
     * @param element The element to be processed.
     * @param params  The message params.  Uses the message params to
     *                decide whether or not cloning is required.
     */
    private void expandMessage(Element element, Map params) {

        // If this element is not a cloned element, check does it need to be cloned...
        if (!element.hasAttributeNS(OGNLUtils.JBOSSESB_SOAP_NS, IS_CLONE_ATTRIB)) {
            String ognl = OGNLUtils.getOGNLExpression(element);
            Element clonePoint = getClonePoint(element);

            if(clonePoint != null) {
                int collectionSize;

                collectionSize = calculateCollectionSize(ognl, params);

                if(collectionSize == -1) {
                    // It's a collection, but has no entries that match the OGNL expression for this element...
                    if(clonePoint == element) {
                        // If the clonePoint is the element itself, we remove it... we're done with it...
                        clonePoint.getParentNode().removeChild(clonePoint);
                    } else {
                        // If the clonePoint is not the element itself (it's a child element), leave it
                        // and check it again when we get to it...
                        resetClonePoint(clonePoint);
                    }
                } else if(collectionSize == 0) {
                    // It's a collection, but has no entries, remove it...
                    clonePoint.getParentNode().removeChild(clonePoint);
                } else if(collectionSize == 1) {
                    // It's a collection, but no need to clone coz we
                    // already have an entry for it...
                    clonePoint.setAttributeNS(OGNLUtils.JBOSSESB_SOAP_NS, OGNLUtils.JBOSSESB_SOAP_NS_PREFIX + OGNLUtils.OGNL_ATTRIB, ognl + "[0]");
                } else {
                    // It's a collection and we need to do some cloning
                    if(clonePoint != null) {
                        // We already have one, so decrement by one...
                        cloneCollectionTemplateElement(clonePoint, (collectionSize - 1), ognl);
                    } else {
                        logger.warn("Collection/array template element <" + element.getLocalName() + "> would appear to be invalid.  It doesn't contain any child elements.");
                    }
                }
            }
        }

        // Now do the same for the child elements...
        List<Node> children = YADOMUtil.copyNodeList(element.getChildNodes());
        for (Node node : children) {
            if (node.getNodeType() == Node.ELEMENT_NODE) {
                expandMessage((Element) node, params);
            }
        }
    }

    private int calculateCollectionSize(String ognl, Map params) {
        // Try for an Object graph based collection...
        Object param = OGNLUtils.getParameter(ognl, params);
        if (param != null) {
            Class paramRuntime = param.getClass();
            if (paramRuntime.isArray()) {
                return ((Object[]) param).length;
            } else if (Collection.class.isAssignableFrom(paramRuntime)) {
                return ((Collection) param).size();
            }
        }

        // Try for <String, Object> collection based map entries...
        Set<Map.Entry> entries = params.entrySet();
        String collectionPrefix = ognl + "[";
        int maxIndex = -1;
        for (Map.Entry entry : entries) {
            Object keyObj = entry.getKey();
            if(keyObj instanceof String) {
                String key = (String)keyObj;
                if(key.startsWith(collectionPrefix)) {
                    int endIndex = key.indexOf(']', collectionPrefix.length());
                    String ognlIndexValString = key.substring(collectionPrefix.length(), endIndex);
                    try {
                        int ognlIndexVal = Integer.valueOf(ognlIndexValString);
                        maxIndex = Math.max(maxIndex, ognlIndexVal);
                    } catch(NumberFormatException e) {}
                }
            }
        }

        if(maxIndex != -1) {
            return maxIndex + 1;
        }

        // It's a collection, but nothing in this message for it collection...
        return -1;
    }

    private Element getClonePoint(Element element) {
        Comment comment;

        // Is it this element...
        comment = getCommentBefore(element);
        if(comment != null && comment.getTextContent().endsWith(SOAPUI_CLONE_COMMENT)) {
            comment.setTextContent(comment.getTextContent() + CLONED_POSTFIX);
            return element;
        }

        // Is it the first child element of this element...
        Element firstChildElement = getFirstChildElement(element);
        if(firstChildElement != null) {
            comment = getCommentBefore(firstChildElement);
            if(comment != null && comment.getTextContent().endsWith(SOAPUI_CLONE_COMMENT)) {
                comment.setTextContent(comment.getTextContent() + CLONED_POSTFIX);
                return firstChildElement;
            }
        }

        return null;
    }

    private void resetClonePoint(Element clonePoint) {
        Comment comment = getCommentBefore(clonePoint);

        if(comment == null) {
            throw new IllegalStateException("Call to reset a 'clonePoint' that doesn't have a comment before it.");
        }

        String commentText = comment.getTextContent();
        if(!commentText.endsWith(CLONED_POSTFIX)) {
            throw new IllegalStateException("Call to reset a 'clonePoint' that doesn't have a proper clone comment before it.");
        }

        comment.setTextContent(commentText.substring(0, commentText.length() - CLONED_POSTFIX.length()));
    }

    private Comment getCommentBefore(Element element) {
        Node sibling = element.getPreviousSibling();

        while(sibling != null) {
            if(sibling.getNodeType() == Node.COMMENT_NODE) {
                return (Comment) sibling;
            } else if(sibling.getNodeType() == Node.TEXT_NODE) {
                // continue...
                sibling = sibling.getPreviousSibling();
            } else {
                // It's an Element, CData, PI etc
                return null;
            }
        }

        return null;
    }

    private Element getFirstChildElement(Element element) {
        NodeList children = element.getChildNodes();
        int childCount = children.getLength();

        for(int i = 0; i < childCount; i++) {
            Node child = children.item(i);
            if(child.getNodeType() == Node.ELEMENT_NODE) {
                return (Element) child;
            }
        }

        return null;
    }

    /**
     * Clone a collection node.
     * <p/>
     * Note we have to frig with the OGNL expressions for collections/arrays because the
     * collection entry is represented by [0], [1] etc in the OGNL expression, not the actual
     * element name on the DOM e.g. collection node "order/items/item" (where "item" is the
     * actual collection entry) maps to the OGNL expression "order.items[0]" etc.
     *
     * @param element    The collection/array "entry" sub-branch.
     * @param cloneCount The number of times it needs to be cloned.
     * @param ognl       The OGNL expression for the collection/array. Not including the
     *                   indexing part.
     */
    private void cloneCollectionTemplateElement(Element element, int cloneCount, String ognl) {
        if (element == null) {
            return;
        }

        Node insertPoint = element.getNextSibling();
        Node parent = element.getParentNode();

        element.setAttributeNS(OGNLUtils.JBOSSESB_SOAP_NS, OGNLUtils.JBOSSESB_SOAP_NS_PREFIX + OGNLUtils.OGNL_ATTRIB, ognl + "[0]");
        for (int i = 0; i < cloneCount; i++) {
            Element clone = (Element) element.cloneNode(true);

            clone.setAttributeNS(OGNLUtils.JBOSSESB_SOAP_NS, OGNLUtils.JBOSSESB_SOAP_NS_PREFIX + IS_CLONE_ATTRIB, "true");
            clone.setAttributeNS(OGNLUtils.JBOSSESB_SOAP_NS, OGNLUtils.JBOSSESB_SOAP_NS_PREFIX + OGNLUtils.OGNL_ATTRIB, ognl + "[" + Integer.toString(i + 1) + "]");
            if (insertPoint == null) {
                parent.appendChild(clone);
            } else {
                parent.insertBefore(clone, insertPoint);
            }
        }
    }

    private void injectParameters(Element element, Map params, String soapNs) {
        NodeList children = element.getChildNodes();
        int childCount = children.getLength();

        for (int i = 0; i < childCount; i++) {
            Node node = children.item(i);

            if (childCount == 1 && node.getNodeType() == Node.TEXT_NODE) {
                if (isParameter(node)) {
                    String ognl = OGNLUtils.getOGNLExpression(element, soapNs);
                    Object param = OGNLUtils.getParameter(ognl, params);

                    element.removeChild(node);
                    element.appendChild(element.getOwnerDocument().createTextNode(param.toString()));
                }
            } else if (node.getNodeType() == Node.ELEMENT_NODE) {
                NamedNodeMap attributes = node.getAttributes();
                injectAttributeParameters(params, soapNs, node, attributes);
                injectParameters((Element) node, params, soapNs);
            }
        }

        element.removeAttributeNS(OGNLUtils.JBOSSESB_SOAP_NS, IS_CLONE_ATTRIB);
        element.removeAttributeNS(OGNLUtils.JBOSSESB_SOAP_NS, OGNLUtils.OGNL_ATTRIB);
    }
    
    private void injectAttributeParameters(Map params, String soapNs, Node node, NamedNodeMap attributes) {
        for (Node attribute : getNodes(attributes)) {
            if (isParameter(attribute)) {
                String localName = attribute.getLocalName();
                String ognl = OGNLUtils.getOGNLExpression((Element)node, soapNs) + "." + localName;
      
                Object param = OGNLUtils.getParameter(ognl, params);
                if (param != null && param.toString().length() > 0) {
                    attribute.setTextContent(param.toString());
                } else {
                    attributes.removeNamedItem(attribute.getNodeName());
                }
            }
        }
    }

    private boolean isParameter(Node node) {
        return node.getTextContent().equals("?");
    }

    private List<Node> getNodes(NamedNodeMap attributes) {
        int len = attributes.getLength();
        List<Node> list = new ArrayList<Node>(len);
        for ( int a = 0 ; a < len ; a++)
        {
            list.add(attributes.item(a));
        }
        return list;
    }

    private synchronized DocumentBuilder getDocBuilder() throws IOException {
        try {
            return docBuilderFactory.newDocumentBuilder();
        } catch (final ParserConfigurationException pce) {
            final IOException ioe = new IOException("Could not create document builder") ;
            ioe.initCause(pce) ;
            throw ioe ;
        }
    }
    
    public void setServerDataDir(final String datadir)
    {
        this.serverDataDir = datadir;
    }
    
    public String getServerDataDir()
    {
        return serverDataDir;
    }

    static final class SmooksCache
    {
        /**
         * The unused entries.
         */
        private final Map<String, Smooks> cachedResources = new HashMap<String, Smooks>();
        /**
         * The list of cached resource entries.
         */
        private final LinkedList<String> cachedResourceInsertion = new LinkedList<String>();
        /**
         * The active entries.
         */
        private final Map<String, SmooksEntry> active = new HashMap<String, SmooksEntry>();
        /**
         * The cache size.
         */
        private final int smooksLRUCacheSize;

        public SmooksCache(final int smooksLRUCacheSize)
        {
            this.smooksLRUCacheSize = smooksLRUCacheSize;
        }

        public synchronized void clean() {
            cachedResourceInsertion.clear() ;
            final Collection<Smooks> cachedSmooks = cachedResources.values() ;
            for(Smooks smooks: cachedSmooks) {
                closeSmooks(smooks) ;
            }
            cachedResources.clear() ;
            if (active.size() > 0) {
                logger.warn("Active smooks entries in cache") ;
            }
        }

        public SmooksEntry get(final String smooksResource)
            throws IOException, SAXException {
            final SmooksEntry smooks = internalGet(smooksResource);
            if (smooks != null) {
                return smooks;
            } else {
                return create(smooksResource);
            }
        }

        private synchronized SmooksEntry internalGet(final String smooksResource) {
            final SmooksEntry result;
            final SmooksEntry activeEntry = active.get(smooksResource);
            if (activeEntry != null) {
                activeEntry.incCount();
                result = activeEntry;
            } else {
                final Smooks smooks = (Smooks) cachedResources.remove(smooksResource);
                if (smooks != null) {
                    cachedResourceInsertion.remove(smooksResource);
                    final SmooksEntry entry = new SmooksEntry(smooks, smooksResource);
                    active.put(smooksResource, entry);
                    result = entry;
                } else {
                    result = null;
                }
            }
            return result;
        }

        private SmooksEntry create(final String smooksResource)
                throws IOException, SAXException {
            final SmooksEntry result;
            boolean close = false;
            final Smooks smooks = new Smooks();
            smooks.addConfigurations("smooks-resource", new ByteArrayInputStream(smooksResource.getBytes("UTF-8")));
            synchronized (this)
            {
                final SmooksEntry current = internalGet(smooksResource);
                if (current == null) {
                    final SmooksEntry newEntry = new SmooksEntry(smooks,
                            smooksResource);
                    active.put(smooksResource, newEntry);
                    result = newEntry;
                } else {
                    close = true;
                    result = current;
                }
            }
            if (close) {
                closeSmooks(smooks);
            } else {
                checkExpired();
            }
            return result;
        }

        public void release(final SmooksEntry smooksEntry) {
            boolean cached = false ;
            synchronized (this)
            {
                if (smooksEntry.decCount() == 0) {
                    final String smooksResource = smooksEntry.getSmooksResource();
                    cachedResources.put(smooksResource, smooksEntry.getSmooks());
                    cachedResourceInsertion.addLast(smooksResource);
                    cached = true ;
                }
            }
            if (cached) {
                checkExpired();
            }
        }

        private void checkExpired() {
            final List<Smooks> expiredSmooks;
            synchronized (this)
            {
                final int activeSize = active.size();
                final int cachedSize = cachedResourceInsertion.size();
                final int expiredCount = (activeSize >= smooksLRUCacheSize ? cachedSize
                        : activeSize + cachedSize - smooksLRUCacheSize);
                if (expiredCount > 0) {
                    expiredSmooks = new ArrayList<Smooks>(expiredCount);
                    for (int count = 0; count < expiredCount; count++) {
                        final String expiredSmooksResource = cachedResourceInsertion.removeFirst();
                        expiredSmooks.add((Smooks) cachedResources.remove(expiredSmooksResource));
                    }
                } else {
                    expiredSmooks = null;
                }
            }
            if (expiredSmooks != null) {
                for (Smooks smooks : expiredSmooks) {
                    closeSmooks(smooks);
                }
            }
        }

        private void closeSmooks(final Smooks smooks) {
            try {
                smooks.close();
            } catch (final Throwable th) {
                logger.warn("Unexpected exception cleaning smooks resource", th);
            }
        }
    }

    static final class SmooksEntry
    {
        private final Smooks smooks;
        private final String smooksResource;
        private int count = 1;

        SmooksEntry(final Smooks smooks, final String smooksResource)
        {
            this.smooks = smooks;
            this.smooksResource = smooksResource;
        }

        Smooks getSmooks() {
            return smooks;
        }

        String getSmooksResource() {
            return smooksResource;
        }

        int incCount() {
            return ++count;
        }

        int decCount() {
            return --count;
        }
    }
}
