/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.net.URI;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.xml.stream.XMLStreamException;

import org.apache.log4j.Logger;
import org.jboss.deployment.DeploymentException;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.publish.ContractReferencePublisher;
import org.jboss.internal.soa.esb.webservice.AddressingHandler;
import org.jboss.internal.soa.esb.webservice.ESBContractGenerator;
import org.jboss.internal.soa.esb.webservice.ESBResponseFilter;
import org.jboss.internal.soa.esb.webservice.ESBServiceEndpointInfo;
import org.jboss.metadata.web.jboss.JBossWebMetaData;
import org.jboss.mx.util.ObjectNameConverter;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.listeners.config.WebserviceInfo;
import org.jboss.soa.esb.listeners.config.model.ModelAdapter;
import org.jboss.soa.esb.listeners.config.model.ModelParser;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.soa.esb.util.FileUtil;
import org.jboss.soa.esb.util.JarUtil;

/**
 * Util class for JBoss AS Deployers.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 * @author <a href="mailto:mageshbk@jboss.com">Magesh Kumar B</a>
 * 
 */
public class JBossDeployerUtil
{
    private static final String PREFIX_CANONICAL_NAME = "jboss.esb:deployment=" ;
    private static final String ESB_ARTIFACT_NAME = "jbossesb.esb" ;
    
    private static final String HANDLER_CHAIN_PREFIX = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" + "<handler-chains xmlns=\"http://java.sun.com/xml/ns/javaee\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"" + " xsi:schemaLocation=\"http://java.sun.com/xml/ns/javaee javaee_web_services_1_2.xsd\">" + "<handler-chain><protocol-bindings>##SOAP11_HTTP</protocol-bindings>";
    private static final String HANDLER_CHAIN_POSTFIX = "</handler-chain></handler-chains>";
    private static final String HANDLER_PREFIX = "<handler><handler-name>";
    private static final String HANDLER_MIDDLE = "</handler-name><handler-class>";
    private static final String HANDLER_POSTFIX = "</handler-class></handler>";

    private static final String WSADDR_NAME = "WSAddressing Handler";
    private static final String WSADDR_HANDLER = "org.jboss.ws.extensions.addressing.jaxws.WSAddressingServerHandler";

    private static final String WSADDR_CONTEXT_NAME = "WSAddressing Context Handler";
    private static final String WSADDR_CONTEXT_HANDLER = AddressingHandler.class.getName();
    
    private static Logger log = Logger.getLogger(JBossDeployerUtil.class);

    public static final String WSIMPL_CXF = "CXF";
    public static final String WSIMPL_NATIVE = "Native";
    public static final String WSIMPL_UNKNOWN = "Unknown";
    private static String WSIMPL;
    private static Boolean WEB_META_DATA_PRESENT;

    private JBossDeployerUtil() { }

    public static synchronized String getWSImpl()
    {
        if (WSIMPL == null)
        {
            ClassLoader loader = Thread.currentThread().getContextClassLoader();
            try
            {
                // Try Native impl first
                loader.loadClass("org.jboss.ws.extensions.addressing.soap.SOAPAddressingBuilderImpl");
                WSIMPL = WSIMPL_NATIVE;
            }
            catch (Throwable t1)
            {
                try
                {
                    // Try CXF impl next
                    loader.loadClass("org.apache.cxf.ws.addressing.AddressingBuilderImpl");
                    WSIMPL = WSIMPL_CXF;
                }
                catch (Throwable t2) {}
            }
            if (WSIMPL == null)
            {
                WSIMPL = WSIMPL_UNKNOWN;
                log.warn("Could not determine WS implementation type") ;
            }
            else
            {
                log.info("Detected WS implementation, using " + WSIMPL) ;
            }
        }
        return WSIMPL;
    }
    
    public static synchronized boolean isWebMetaDataPresent()
    {
        if (WEB_META_DATA_PRESENT == null)
        {
            try
            {
                Class metaData = JBossWebMetaData.class;
                WEB_META_DATA_PRESENT = true ;
            }
            catch (final Throwable th)
            {
                WEB_META_DATA_PRESENT = false ;
            }
        }
        return WEB_META_DATA_PRESENT;
    }

    public static String getHandlers(final ESBServiceEndpointInfo serviceInfo)
    {
        final String result;

        if (!serviceInfo.isAddressing())
        {
            result = null;
        } 
        else
        {
            final StringBuilder sb = new StringBuilder(HANDLER_CHAIN_PREFIX);
            if (serviceInfo.isAddressing())
            {
                sb.append(HANDLER_PREFIX).append(WSADDR_CONTEXT_NAME).append(HANDLER_MIDDLE).append(WSADDR_CONTEXT_HANDLER).append(HANDLER_POSTFIX);
                if (getWSImpl().equals(WSIMPL_NATIVE))
                {
                    sb.append(HANDLER_PREFIX).append(WSADDR_NAME).append(HANDLER_MIDDLE).append(WSADDR_HANDLER).append(HANDLER_POSTFIX);
                }
            }
            sb.append(HANDLER_CHAIN_POSTFIX);
            result = sb.toString();
        }
        return result;
    }

    public static File getESBWarFile(final String deploymentName, final String esbName, final boolean createDir, final File dir)
    {
        final File esbDir = new File(dir, deploymentName);
        if (!esbDir.exists())
        {
            if (!createDir)
            {
                return null;
            }
            boolean result = esbDir.mkdir();
            if (!result && !esbDir.exists()) {
            	log.error("Could not create " + esbDir.getAbsolutePath());
            }
        }

        final int lastSeparator = esbName.lastIndexOf('.');
        final String warFileName = ((lastSeparator >= 0) ? esbName.substring(0, lastSeparator) : esbName) + ".war";

        return new File(esbDir, warFileName);
    }

    public static File createDir(final File parentDir, String dirName)
    {
        final File esbWarFiles = new File(parentDir, dirName);
        if (!esbWarFiles.exists()) {
            boolean result = esbWarFiles.mkdirs();
            if (!result && !esbWarFiles.exists()) {
            	log.error("Could not create directory " + esbWarFiles.getAbsolutePath());
            }
        }
        return esbWarFiles;
    }

    public static String readEsbConfig(final InputStream in) throws DeploymentException
    {
        try
        {
            final StringWriter sw = new StringWriter();
            XMLHelper.replaceSystemProperties(XMLHelper.getXMLStreamReader(in), XMLHelper.getXMLStreamWriter(sw));
            return sw.toString();
        } catch (XMLStreamException e)
        {
            throw new DeploymentException(e.getMessage(), e);
        } finally
        {
            try
            {
                in.close();
            } catch (IOException ignore)
            {
            }
        }
    }

    public static ModelAdapter getJbossEsbModel(final String configuration) throws DeploymentException
    {
        try
        {
            final StringReader sr = new StringReader(configuration);
            return ModelParser.getParser().parse(sr);
        } catch (final Exception ex)
        {
            throw new DeploymentException("Failed to parse jboss-esb.xml", ex);
        }
    }

    public static Properties getArtifactProperties(final String fileName) throws IOException
    {
        Properties properties = null;
        final InputStream in = ClassUtil.getResourceAsStream(fileName, JBossDeployerUtil.class);
        if (in != null)
        {
            log.debug("Loaded action artifact mapping from " + fileName);
            properties = new Properties();
            properties.load(in);
        }
        else
        {
            log.debug("No action artifact mapping") ;
        }
        return properties;
    }
    
    public static Set<ObjectName> getActionDependencies(final String deploymentName, final ModelAdapter model, final Properties actionArtifactProperties) throws DeploymentException
    {
        final Set<ObjectName> deps = new HashSet<ObjectName>();
        final Set<String> artifacts = new HashSet<String>() ;
        artifacts.add(ESB_ARTIFACT_NAME) ;
    
        final Set<String> actionClasses = model.getActions() ;
          
        final int numActionClasses = (actionClasses == null ? 0 : actionClasses.size()) ;
        if (numActionClasses > 0)
        {
            for(final String actionClass: actionClasses)
            {
                final String artifact = (String) actionArtifactProperties.get(actionClass) ;
                if (artifact != null)
                {
                    artifacts.add(artifact) ;
                }
            }
        }
          
        for(final String artifact: artifacts)
        {
            if (!deploymentName.equals(artifact))
            {
                final String canonicalName = PREFIX_CANONICAL_NAME + artifact ;
                try
                {
                    ObjectName on = ObjectNameConverter.convert(canonicalName);
                    deps.add(on) ;
                } 
                catch (MalformedObjectNameException e)
                {
                    throw new DeploymentException(e.getMessage(), e);
                }
            }
        }
        return deps;
    }

    public static Map<String, String> getSchemas(final File file) throws DeploymentException
    {
        if (file == null)
        {
            return Collections.emptyMap();
        }
        
        if (file.isDirectory()) 
        {
            return getSchemasFromDirectory(file);
        }
        
        final String fileName = file.getName();
        if (fileName.endsWith(".esb"))
            return getSchemasFromArchive(file);
            
        if (fileName.endsWith("-esb.xml"))
        {
            final File metaInf = file.getParentFile();
            final File deploymentRoot = metaInf.getParentFile();
            return getSchemasFromDirectory(deploymentRoot);
        }
        
        throw new DeploymentException("Unrecognized deployment : " + file + ".");
    }
    
    public static Map<String, String> getSchemasFromDirectory(final File directory) throws DeploymentException
    {
        final URI rootURI = directory.toURI() ;
        return getSchemasFromDir(rootURI, directory);
    }

    private static Map<String, String> getSchemasFromDir(final URI root, final File directory) throws DeploymentException
    {
        if (directory != null)
        {
            final Map<String, String> schemasMap = new HashMap<String, String>();
            
            if (directory.isDirectory())
            {
                final File[] files = directory.listFiles();
                for (File file : files)
                {
                    if (file.isDirectory())
                    {
                        schemasMap.putAll(getSchemasFromDir(root, file));
                    }
                    else if (isXsd(file.getName()))
                    {
                        try
                        {
                            final URI relativeURI = root.relativize(file.toURI()) ;
                            final String filename = relativeURI.getPath();
                            schemasMap.put(filename, FileUtil.readTextFile(file));
                        } 
                        catch (final IOException e)
                        {
                            throw new DeploymentException("IOException while trying to read xsd '" + file.getName() + "'", e);
                        }
                    }
                    else if (file.getName().endsWith(".jar"))
                    {
                        Map<String, String> schemasFromArchive = getSchemasFromArchive(file);
                        schemasMap.putAll(schemasFromArchive);
                    }
                }
            }
            return schemasMap;
        }
        
        return Collections.emptyMap();
    }
    
    private static boolean isXsd(final String fileName)
    {
        return fileName.endsWith(".xsd");
    }
    
    public static Map<String, String> getSchemasFromArchive(final File archive) throws DeploymentException
    {
        if (archive == null)
        {
            return Collections.emptyMap();
        }
        
        ZipInputStream zipInputStream = null;
        try
        {
            zipInputStream = new ZipInputStream(new FileInputStream(archive));
            return JarUtil.extractStringContents(".xsd", zipInputStream);
        }
        catch (FileNotFoundException e)
        {
            throw new DeploymentException(e.getMessage(), e);
        }
        catch (IOException e)
        {
            throw new DeploymentException(e.getMessage(), e);
        }
        finally
        {
            closeStream(zipInputStream);
        }
    }
    
    private static void closeStream (InputStream in)
    {
        try
        {
            in.close();
        }
        catch (final IOException e)
        {
            log.warn(e.getMessage(), e);
        }
        
    }

}
