/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;

/**
 * Aggregation Details.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class AggregationDetails {

    private String seriesUuid;
    private int messageNumber = -1;
    private int seriesSize = -1;
    private long seriesTimestamp = -1;
    private String splitId;

    public AggregationDetails(String seriesUuid, int messageNumber, int seriesSize, long seriesTimestamp) {
        AssertArgument.isNotNullAndNotEmpty(seriesUuid, "seriesUuid");
        if(messageNumber < 1) {
            throw new IllegalArgumentException("Invalid AggregatorDetails.  messageNumber < 1");
        }
        if(seriesSize < messageNumber) {
            throw new IllegalArgumentException("Invalid AggregatorDetails.  seriesSize < messageNumber");
        }
        if(seriesTimestamp < 1) {
            throw new IllegalArgumentException("Invalid AggregatorDetails.  timestamp < 1");
        }

        this.seriesUuid = seriesUuid;
        this.messageNumber = messageNumber;
        this.seriesSize = seriesSize;
        this.seriesTimestamp = seriesTimestamp;
    }

    public AggregationDetails(String aggregatorTag) {
        AssertArgument.isNotNullAndNotEmpty(aggregatorTag, "aggregatorTag");
        String[] tokens = aggregatorTag.split(":");

        if(tokens.length < 4) {
            throw new IllegalArgumentException("Invalid Aggregator Tag.  Must have 4 tokens (colon separated).");
        }

        this.seriesUuid = tokens[0];
        if(this.seriesUuid == null || this.seriesUuid.trim().equals("")) {
            throw new IllegalArgumentException("Invalid Aggregator Tag: seriesUuid is blank.");
        }
        try {
            this.messageNumber = Integer.parseInt(tokens[1]);
            if(messageNumber < 1) {
                throw new IllegalArgumentException("Invalid Aggregator Tag.  messageNumber < 1");
            }
            this.seriesSize = Integer.parseInt(tokens[2]);
            if(seriesSize < messageNumber) {
                throw new IllegalArgumentException("Invalid Aggregator Tag.  seriesSize < messageNumber");
            }
            this.seriesTimestamp = Long.parseLong(tokens[3]);
            if(seriesTimestamp < 1) {
                throw new IllegalArgumentException("Invalid Aggregator Tag.  timestamp < 1");
            }
            if(tokens.length == 5) {
                splitId = tokens[4];
            }
        } catch(NumberFormatException e) {
            if(this.messageNumber == -1) {
                throw new IllegalArgumentException("Invalid Aggregator Tag.  'messageNumber' must be an int.");
            } else if(this.seriesSize == -1) {
                throw new IllegalArgumentException("Invalid Aggregator Tag.  'seriesSize' must be an int.");
            } else if(this.seriesTimestamp == -1) {
                throw new IllegalArgumentException("Invalid Aggregator Tag.  'timestamp' must be an long.");
            }
        }
    }

    public String getSeriesUuid() {
        return seriesUuid;
    }

    public int getSeriesSize() {
        return seriesSize;
    }

    public int getMessageNumber() {
        return messageNumber;
    }

    public long getSeriesTimestamp() {
        return seriesTimestamp;
    }

    public String getSplitId() {
        return splitId;
    }

    public void setSplitId(String splitId) {
        this.splitId = splitId;
    }

    public boolean equals(Object obj) {
        if (!(obj instanceof AggregationDetails)) {
            // also covers null
            return false;
        } else if (obj == this) {
            return true;
        } else {
            return toString().equals(obj.toString());
        }
    }

    public int hashCode() {
        return toString().hashCode();
    }

    public String toString() {
        assertDetailsSet();
        if(splitId == null) {
            return seriesUuid + ":" + messageNumber + ":" + seriesSize + ":" + seriesTimestamp;
        } else {
            return seriesUuid + ":" + messageNumber + ":" + seriesSize + ":" + seriesTimestamp + ":" + splitId;
        }
    }

    private void assertDetailsSet() {
        if(seriesUuid == null || seriesUuid.trim().equals("")) {
            throw new IllegalStateException("AggregationDetails not set: 'seriesUuid' is null or empty.");
        }
        if(messageNumber == -1) {
            throw new IllegalStateException("AggregationDetails not set: 'messageNumber' is not set.");
        }
        if(seriesSize == -1) {
            throw new IllegalStateException("AggregationDetails not set: 'seriesSize' is not set.");
        }
        if(seriesTimestamp == -1) {
            throw new IllegalStateException("AggregationDetails not set: 'timestamp' is not set.");
        }
    }
}
