/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.gateway;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.remoting.HttpMarshaller;
import org.jboss.internal.soa.esb.remoting.HttpUnmarshaller;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.remoting.InvocationRequest;
import org.jboss.remoting.InvokerLocator;
import org.jboss.remoting.ServerInvocationHandler;
import org.jboss.remoting.ServerInvoker;
import org.jboss.remoting.callback.InvokerCallbackHandler;
import org.jboss.remoting.marshal.MarshalFactory;
import org.jboss.remoting.marshal.http.HTTPMarshaller;
import org.jboss.remoting.security.SSLSocketBuilder;
import org.jboss.remoting.transport.Connector;
import org.jboss.remoting.transport.http.HTTPMetadataConstants;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionUtils;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.KeyValuePair;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.lifecycle.AbstractManagedLifecycle;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.listeners.message.AbstractMessageComposer;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.listeners.message.UncomposedMessageDeliveryAdapter;
import org.jboss.soa.esb.message.*;
import org.jboss.soa.esb.message.MessagePayloadProxy.NullPayloadHandling;
import org.jboss.soa.esb.message.Properties;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.RegistryFactory;
import org.jboss.soa.esb.services.security.SecurityServiceException;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequest;
import org.jboss.soa.esb.services.security.auth.ExtractionException;
import org.jboss.soa.esb.services.security.auth.ExtractorUtil;
import org.jboss.soa.esb.services.security.auth.SecurityInfoExtractor;
import org.jboss.soa.esb.services.security.auth.http.JbrHttpSecurityInfoExtractor;
import org.jboss.soa.esb.services.security.auth.ws.BinarySecurityTokenExtractor;
import org.jboss.soa.esb.services.security.auth.ws.UsernameTokenExtractor;
import org.jboss.soa.esb.util.ClassUtil;

import javax.management.MBeanServer;
import javax.security.auth.login.LoginException;

import java.io.StringWriter;
import java.io.PrintWriter;
import java.net.*;
import java.util.*;

/**
 * JBoss Remoting listener implementation for receiving ESB unaware messages
 * over a JBoss Remoting channel.
 * <p/>
 * This class implements the JBoss Remoting interface {@link org.jboss.remoting.ServerInvocationHandler}.
 * Messages are pushed in through the {@link #invoke(org.jboss.remoting.InvocationRequest)} method,
 * which is an implementation of the {@link org.jboss.remoting.ServerInvocationHandler#invoke(org.jboss.remoting.InvocationRequest)}
 * method.
 * <p/>
 * The JBoss Remoting {@link org.jboss.remoting.transport.Connector}
 * configuration is populated by the {@link #initaliseJBRConnectorConfiguration(java.util.Map)}
 * method.  The remoting server locator URI is constructed by the
 * {@link #getJbrServerLocatorURI()} method.
 * <p/>
 * The default {@link org.jboss.soa.esb.listeners.message.MessageComposer} used by this listener is the
 * {@link org.jboss.soa.esb.listeners.gateway.JBossRemotingGatewayListener.JBossRemotingMessageComposer}.  All
 * message composer implementation supplied to this class will be supplied an instance of
 * {@link org.jboss.remoting.InvocationRequest}, from which they must compose an ESB aware
 * {@link Message} instance.
 * <p/>
 * See the <a href="http://labs.jboss.com/portal/jbossremoting">JBoss Remoting</a> docs
 * for details on configuring specific remoting server protocols.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class JBossRemotingGatewayListener extends AbstractManagedLifecycle implements ServerInvocationHandler {

    /**
     * JBoss Remoting connector config attribute name prefix.
     */
    public static final String JBR_PREFIX = "jbr-";
    /**
     * Server Protocol config attribute name.
     */
    public static final String JBR_SERVER_PROTOCOL = JBR_PREFIX + "serverProtocol";
    /**
     * Server Host config attribute name.
     */
    public static final String JBR_SERVER_HOST = JBR_PREFIX + ServerInvoker.SERVER_BIND_ADDRESS_KEY;
    /**
     * Server port config attribute name.
     */
    public static final String JBR_SERVER_PORT = JBR_PREFIX + ServerInvoker.SERVER_BIND_PORT_KEY;
    /**
     * JBoss Remoting request for the current thread.
     */
    private static ThreadLocal<InvocationRequest> currentRequest = new ThreadLocal<InvocationRequest>();
    /**
     * Class Logger instance.
     */
    private static Logger logger = Logger.getLogger(JBossRemotingGatewayListener.class);
    /**
     * JBoss Remoting connector.
     */
    private Connector connector;
    /**
     * Connector configuration.
     */
    private final Map<String, String> connectorConfig = new HashMap<String, String>();
    /**
     * Server URI.
     */
    private String jbrServerLocatorURI;
    /**
     * Delivery adapter.
     */
    private UncomposedMessageDeliveryAdapter messageDeliveryAdapter;
    /**
     * Simple flag marking the listener instance as being initialised or un-initialised.
     */
    private boolean initialised;
    /**
     * Service category to which this listener is associated.
     */
    private final String serviceCategory;
    /**
     * Service name to which this listener is associated.
     */
    private final String serviceName;
    /**
     * Listener endpoint EPR.
     */
    private EPR endpointReference;
    /**
     * Is the listener synchronous.
     */
    private boolean synchronous = true;
    /**
     * Asynchronous response.
     */
    private final String asyncResponse;
    /**
     * Flag indicating whether or not this gateway is servicing HTTP requests.
     */
    private boolean isHttp;
    /**
     * ServiceInvoker timeout if the ServiceInvoker is configured to be synchronous.
     */
    private long serviceInvokerTimeout;

    /**
     * Install our own marshaller/unmarshaller for HTTP.
     */
    static {
        MarshalFactory.addMarshaller(HTTPMarshaller.DATATYPE,
                new HttpMarshaller(),
                new HttpUnmarshaller());
    }

    /**
     * Construct the threaded managed lifecycle.
     *
     * @param config The configuration associated with this instance.
     * @throws org.jboss.soa.esb.ConfigurationException
     *          for configuration errors during initialisation.
     */
    public JBossRemotingGatewayListener(ConfigTree config) throws ConfigurationException {
        super(config);
        serviceCategory = config.getAttribute(ListenerTagNames.TARGET_SERVICE_CATEGORY_TAG);
        serviceName = config.getAttribute(ListenerTagNames.TARGET_SERVICE_NAME_TAG);
        synchronous = !config.getAttribute("synchronous", "true").equalsIgnoreCase("false");
        if (synchronous)
        {
            serviceInvokerTimeout = config.getLongAttribute(ListenerTagNames.SERVICE_INVOKER_TIMEOUT, 20000);
        }
        asyncResponse = config.getAttribute("asyncResponse");

        if(asyncResponse != null) {
            if(ClassUtil.getResourceAsStream(asyncResponse, getClass()) == null) {
                throw new ConfigurationException("Asynchronous response resource file '" + asyncResponse + "' not found on classpath.");
            }
        }
    }

    public static InvocationRequest getCurrentRequest() {
        return currentRequest.get();
    }

    /**
     * Is this listener instance initialised.
     *
     * @return True if this listener is initialised, otherwise false.
     */
    public boolean isInitialised() {
        return initialised;
    }

    /**
     * Is this listener instance started.
     * <p/>
     * Basically, Is this listener's JBoss Remoting Connector connected?
     *
     * @return True if this listener is started, otherwise false.
     */
    public boolean isStarted() {
        return (connector != null);
    }

    /*
     * ***************************************************************************
     *
     * AbstractManagedLifecycle methods...
     *
     * ****************************************************************************
     */

    @Override
    protected void doInitialise() throws ManagedLifecycleException {
        if (isInitialised()) {
            throw new ManagedLifecycleException("Unexpected request to initialise JBoss Remoting Gateway listener '" + getConfig().getName() + "'.  Gateway already initialised.");
        }

        try {
            endpointReference = new EPR(getJbrServerLocatorURI());
            messageDeliveryAdapter = createDeliveryAdapter();
            initaliseJBRConnectorConfiguration(connectorConfig);

            String protocol = getConfig().getAttribute(JBR_SERVER_PROTOCOL);
            isHttp = (protocol.equals("http") || protocol.equals("https"));
        } catch (ConfigurationException e) {
            throw new ManagedLifecycleException("Remoting Listener configuration failed.", e);
        }

        initialised = true;
    }

    @Override
    protected void doStart() throws ManagedLifecycleException {
        if (!isInitialised()) {
            throw new ManagedLifecycleException("Unexpected request to start JBoss Remoting Gateway listener '" + getConfig().getName() + "'.  Gateway not initialised.");
        }
        if (isStarted()) {
            throw new ManagedLifecycleException("Unexpected request to start JBoss Remoting Gateway listener '" + getConfig().getName() + "'.  Gateway already started.");
        }

        // Start the JBR Server...
        startJBRServer();

        // Regsiter the JBR Endpoint...
        try {
            registerEndpoint();
        } catch (Throwable t) {
            logger.error("Unable to register service endpoint '" + endpointReference.getAddr().getAddress()
                    + "' for service '" + serviceCategory + ":" + serviceName + "'.  Stopping JBossRemoting Server...", t);
            stopJBRServer();
        }
    }

    @Override
    protected void doStop() throws ManagedLifecycleException {
        if (!isStarted()) {
            throw new ManagedLifecycleException("Unexpected request to stop JBoss Remoting Gateway listener '" + getConfig().getName() + "'.  Gateway not running.");
        }

        unregisterEndpoint();
        stopJBRServer();
    }

    private void startJBRServer() throws ManagedLifecycleException {
        try {
            InvokerLocator locator = new InvokerLocator(jbrServerLocatorURI);

            connector = new Connector(locator, connectorConfig);
            connector.create();
            connector.addInvocationHandler(getConfig().getAttribute("name", this.toString()), this);
            connector.start();

            logger.info("JBoss Remoting Gateway listener '" + getConfig().getName() + "' started.");
        } catch (Throwable throwable) {
            if (connector != null)
            {
                connector.stop() ;
            }
            connector = null;
            throw new ManagedLifecycleException("Unable to start Remoting Listener instsance " + getClass().getName(), throwable);
        }
    }

    private void stopJBRServer() throws ManagedLifecycleException {
        try {
            connector.stop();
            logger.info("JBoss Remoting Gateway listener '" + getConfig().getName() + "' stopped.");
        } catch (Throwable throwable) {
            throw new ManagedLifecycleException("Unable to stop Remoting Listener instsance " + getClass().getName(), throwable);
        } finally {
            connector = null;
        }
    }

    private void registerEndpoint() throws ConfigurationException, RegistryException {
        String serviceDescription = getConfig().getAttribute(ListenerTagNames.SERVICE_DESCRIPTION_TAG);
        RegistryFactory.getRegistry().registerEPR(serviceCategory, serviceName, serviceDescription,
                endpointReference, endpointReference.getAddr().getAddress());
    }

    private void unregisterEndpoint() {
        try {
            RegistryFactory.getRegistry().unRegisterEPR(serviceCategory, serviceName, endpointReference);
        } catch (Throwable t) {
            logger.error("Unable to unregister service endpoint '" + endpointReference.getAddr().getAddress()
                    + "' for service '" + serviceCategory + ":" + serviceName + "'.", t);
        }
    }

    @Override
    protected void doDestroy() throws ManagedLifecycleException {
    }

    /*
     * ***************************************************************************
     *
     * JBoss Remoting ServerInvocationHandler methods...
     *
     * ****************************************************************************
     */

    /**
     * Process a remoting invocation message.
     * <p/>
     * This method uses an {@link org.jboss.soa.esb.listeners.message.UncomposedMessageDeliveryAdapter}
     * to carry out the delivery.  This delivery adpter is constructed with a
     * {@link org.jboss.soa.esb.listeners.message.MessageComposer} instance supplied through
     * configuration, otherwise it uses the
     * {@link org.jboss.soa.esb.listeners.gateway.JBossRemotingGatewayListener.JBossRemotingMessageComposer}.
     * <p/>
     * The message composer is responsible for mapping the remoting {@link org.jboss.remoting.InvocationRequest}
     * into an ESB aware {@link org.jboss.soa.esb.message.Message}, while the
     * {@link org.jboss.soa.esb.listeners.message.UncomposedMessageDeliveryAdapter} is responsible for its
     * delivery to the target service.
     *
     * @param invocationRequest JBoss Remoting request.
     * @return Message delivery acknowledgment response.
     * @throws Throwable Message processing failure.
     */
    public Object invoke(InvocationRequest invocationRequest) throws Throwable {
        Map responseMap = getJBRResponseMap(invocationRequest);

        // Set the request object on the thread so as to make it available
        // to the HttpMarshaller...
        currentRequest.set(invocationRequest);

        try {
            if (synchronous) {
                Object response = messageDeliveryAdapter.deliverSync(invocationRequest, serviceInvokerTimeout);
                if(logger.isDebugEnabled()) {
                    logger.debug("Returning response [" + response + "].");
                }

                if(isHttp) {
                	Integer responseCode;
                    Object code = responseMap.get(HTTPMetadataConstants.RESPONSE_CODE);
                    if (code != null) {
                    	responseCode = Integer.valueOf(code.toString());
                    } else {
                    	responseCode = Integer.valueOf(HttpURLConnection.HTTP_OK);
                    }
                    responseMap.put(HTTPMetadataConstants.RESPONSE_CODE, responseCode);
                    String responseCodeMessage = (String)responseMap.get(HTTPMetadataConstants.RESPONSE_CODE_MESSAGE);
                    if (responseCodeMessage == null) {
                    	responseCodeMessage = "OK";
                    }
                    responseMap.put(HTTPMetadataConstants.RESPONSE_CODE_MESSAGE, responseCodeMessage);
                }
                return response;
            } else {
                messageDeliveryAdapter.deliverAsync(invocationRequest);
                if(isHttp) {
                    responseMap.put(HTTPMetadataConstants.RESPONSE_CODE, HttpURLConnection.HTTP_ACCEPTED);
                    responseMap.put(HTTPMetadataConstants.RESPONSE_CODE_MESSAGE, "OK");
                }

                if(asyncResponse == null) {
                    return "<ack/>";
                } else {
                    return StreamUtils.readStreamString(getClass().getResourceAsStream(asyncResponse), "UTF-8");
                }
            }
        } catch (Throwable t) {
            logger.debug("JBoss Remoting Gateway failed to " + (synchronous ? "synchronously" : "asynchronously") + " deliver message to target service [" +
                    messageDeliveryAdapter.getDeliveryAdapter().getServiceCategory() + ":" +
                    messageDeliveryAdapter.getDeliveryAdapter().getServiceName() + "].", t);

            if(isHttp) {
                StringWriter stringWriter = new StringWriter();

                mapHTTPErrorDetails(responseMap, t);
                t.printStackTrace(new PrintWriter(stringWriter));
                responseMap.put(HTTPMetadataConstants.CONTENTTYPE, "text/plain");

                return stringWriter.toString();
            }

            throw t;
        }
    }

    private void mapHTTPErrorDetails(Map responseMap, Throwable t) {
        Throwable cause = t.getCause();

        if(cause instanceof SecurityServiceException && cause.getCause() instanceof LoginException) {
            responseMap.put(HTTPMetadataConstants.RESPONSE_CODE, HttpURLConnection.HTTP_UNAUTHORIZED);
            responseMap.put(HTTPMetadataConstants.RESPONSE_CODE_MESSAGE, "Invalid Login Credentials.");
        } else {
            responseMap.put(HTTPMetadataConstants.RESPONSE_CODE, HttpURLConnection.HTTP_INTERNAL_ERROR);
            responseMap.put(HTTPMetadataConstants.RESPONSE_CODE_MESSAGE, "Failed to service request.");
        }
    }

    public void setMBeanServer(MBeanServer mBeanServer) {
    }

    public void setInvoker(ServerInvoker serverInvoker) {
    }

    public void addListener(InvokerCallbackHandler invokerCallbackHandler) {
    }

    public void removeListener(InvokerCallbackHandler invokerCallbackHandler) {
    }

    /**
     * Initialise the JBossRemoting connector configuration.
     * <p/>
     * Constructs the JBR {@link org.jboss.remoting.InvokerLocator} URI
     * through a call to {@link #getJbrServerLocatorURI()}. Also
     * populates the server connector properties.
     * <p/>
     * Default behavior for population of the connector configuration is to load
     * all listener configuration properties whose name is prefixed with "jbr-",
     * stripping off the "jbr-" prefix from the name before adding.
     * So, to set the Server "timeout" configuration property
     * on the connector, you set the property name to "jbr-timeout".
     *
     * @param connectorConfig The configuration map instance to be populated.
     * @throws ConfigurationException Problem populating the configuration.
     */
    protected void initaliseJBRConnectorConfiguration(Map<String, String> connectorConfig) throws ConfigurationException {
        String protocol = getJbrServerLocatorURI().getScheme();
        boolean isHttps = protocol.equals("https");
        boolean isSecure = (isHttps || protocol.equals("sslsocket"));

        // Initialse the JBR connector URI...
        jbrServerLocatorURI = getJbrServerLocatorURI().toString();

        // Make sure the "SSLImplementation" attribute is set for https...
        if(isHttps) {
            connectorConfig.put("SSLImplementation", "org.jboss.remoting.transport.coyote.ssl.RemotingSSLImplementation");
        }

        // Populate the connector config...
        List<KeyValuePair> attributes = getConfig().attributesAsList();
        for (KeyValuePair attribute : attributes) {
            String attributeName = attribute.getKey();

            if (attributeName.startsWith(JBR_PREFIX)) {
                String jbrAttributeName = attributeName.substring(JBR_PREFIX.length());
                String postv220JbrSSLAttributeName = legacySSLConfigNameMappings.get(jbrAttributeName);

                connectorConfig.put(jbrAttributeName, attribute.getValue());
                if(postv220JbrSSLAttributeName != null) {
                    connectorConfig.put(postv220JbrSSLAttributeName, attribute.getValue());
                    connectorConfig.put("UseSSLServerSocketFactory", "false");
                } else if(legacySSLConfigNameMappings.containsValue(jbrAttributeName)) {
                    connectorConfig.put("UseSSLServerSocketFactory", "false");
                }
            }
        }

        if(isSecure && !connectorConfig.containsKey("serverSocketFactory")) {
            logger.info("Secure endpoint '" + jbrServerLocatorURI + "' doesn't define a 'serverSocketFactory'.  Using Server defaults.  See JBossRemoting documentation.");
        }
    }

    /**
     * Get the Service Locator URI for this remotng based listener.
     * <p/>
     * Uses the listener config to extract the {@link #JBR_SERVER_PROTOCOL protcol},
     * {@link #JBR_SERVER_HOST host} and {@link #JBR_SERVER_PORT port}
     * parameters for the server.  The host address defaults to
     * the value returned by {@link java.net.InetAddress#getLocalHost()}.
     *
     * @return The Server Locator URI.
     * @throws ConfigurationException One or more of the locator properties
     *                                are missing from the listener config.
     */
    protected URI getJbrServerLocatorURI() throws ConfigurationException {
        String protocol = getConfig().getAttribute(JBR_SERVER_PROTOCOL);
        String host = getConfig().getAttribute(JBR_SERVER_HOST);
        String port = getConfig().getAttribute(JBR_SERVER_PORT);

        if (protocol == null || protocol.trim().equals("")) {
            throw new ConfigurationException("Invalid JBoss Remoting Gateway configuration [" + getConfig().getName() + "]. 'protocol' configuration attribute not specified.");
        }
        if (host == null || host.trim().equals("")) {
            host = Configuration.getBindAddress();
        }
        if (port == null || port.trim().equals("")) {
            throw new ConfigurationException("Invalid JBoss Remoting Gateway configuration [" + getConfig().getName() + "]. 'port' configuration attribute not specified.");
        }

        String uriString = protocol + "://" + host + ":" + port;
        try {
            return new URI(uriString);
        } catch (URISyntaxException e) {
            throw new ConfigurationException("Invalid JBoss Remoting Gateway configuration [" + getConfig().getName() + "]. Invalid server locator URI '" + uriString + "'.");
        }
    }

    /**
     * Factory method for adapter creation.
     *
     * @return The adapter instance.
     * @throws ConfigurationException Configuration error.
     */
    protected UncomposedMessageDeliveryAdapter createDeliveryAdapter() throws ConfigurationException {
        return UncomposedMessageDeliveryAdapter.getGatewayDeliveryAdapter(getConfig(), new JBossRemotingMessageComposer<InvocationRequest>());
    }

    /**
     * Message composer for a JBoss Remoting {@link org.jboss.remoting.InvocationRequest}
     * instance.
     */
    public static class JBossRemotingMessageComposer<T extends InvocationRequest> extends AbstractMessageComposer<T> {

        private MessagePayloadProxy payloadProxy;
        
        private JbrHttpSecurityInfoExtractor jbrHttpSecurityExtractor = new JbrHttpSecurityInfoExtractor();
	    private final Set<SecurityInfoExtractor<String>> extractors = new LinkedHashSet<SecurityInfoExtractor<String>>();

        @Override
        public void setConfiguration(ConfigTree config) {
            super.setConfiguration(config);
            payloadProxy = new MessagePayloadProxy(config,
                    new String[] {ActionUtils.POST_ACTION_DATA, Body.DEFAULT_LOCATION, BytesBody.BYTES_LOCATION},
                    new String[] {ActionUtils.POST_ACTION_DATA});
            // Allow null to be set on as the message payload...
            payloadProxy.setNullSetPayloadHandling(NullPayloadHandling.NONE);
            
            final String securityNS = config.getAttribute("securityNS", "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd");
	        extractors.add(new UsernameTokenExtractor(securityNS));
	        extractors.add(new BinarySecurityTokenExtractor(securityNS));
        }

        @Override
        protected MessagePayloadProxy getPayloadProxy() {
            return payloadProxy;
        }

        @Override
        @SuppressWarnings("unchecked")
        protected void populateMessage(Message message, T invocationRequest) throws MessageDeliverException {

            // Set the payload from the JBR invocation...
            payloadProxy.setPayload(message, invocationRequest.getParameter());

            // Copy the request properties onto the message...
            Map properties = invocationRequest.getRequestPayload();


            if (properties != null) {
                //	Extract any jbr http security information that may exist
                AuthenticationRequest authRequest = jbrHttpSecurityExtractor.extractSecurityInfo(properties);

                // Extract any ws security information that may exist
                Object payload = invocationRequest.getParameter();
                if ( authRequest == null && payload instanceof String) {
                    
                    try
                    {
                        authRequest = ExtractorUtil.extract((String)payload, extractors);
                    }
                    catch (ExtractionException e)
                    {
	                    throw new MessageDeliverException(e.getMessage(), e);
                    }
                }
                
                try {
	                ExtractorUtil.addAuthRequestToMessage(authRequest, message);
                } catch (final SecurityServiceException e) {
                    throw new MessageDeliverException(e.getMessage(), e);
                }


                // Purposely not iterating over the Map.Entry Set because there's
                // a bug in the Map impl used by JBossRemoting.  Not all the
                // "values" are actually in the Map.Entry set.  Some of them are handled
                // from within an overridden impl of the Map.get(Object) method.
                Set names = properties.keySet();
                for (Object name : names) {
                    Object value = properties.get(name);
                    if(value != null) {
                        message.getProperties().setProperty(name.toString(), value);
                    }
                }
            }
        }

        @Override
        public Object decompose(Message message, T invocationRequest) throws MessageDeliverException {
            Properties properties = message.getProperties();
            String propertyNames[] = properties.getNames();
            Map responseMap = getJBRResponseMap(invocationRequest);

            for(String name : propertyNames) {
                Object value = properties.getProperty(name);

                if(value instanceof ResponseHeader) {
                	// JBESB-2511
                    ResponseHeader header = (ResponseHeader)value;
                    header.putNameValue(responseMap);
                }
                // JBESB-2761
                if(HTTPMetadataConstants.RESPONSE_CODE.equals(name) ||
                		HTTPMetadataConstants.RESPONSE_CODE_MESSAGE.equals(name)) {
                	responseMap.put(name, value);
                }
            }

            return super.decompose(message, invocationRequest);
        }
    }

    private static Map getJBRResponseMap(InvocationRequest invocationRequest) {
        Map responseMap = invocationRequest.getReturnPayload();

        if(responseMap == null) {
            responseMap = new LinkedHashMap();
            invocationRequest.setReturnPayload(responseMap);
        }
        return responseMap;
    }

    private static Map<String, String> legacySSLConfigNameMappings = new HashMap<String, String>();
    static {
        /*
         * The attribute names of some or all of the JBR SSLSocketBuilder config parameters have
         * changed (post v2.2.0) and don't match the JBR documentation.  This map is used to map
         * them to their new names, where the legacy names are used ala the documentation.
         *
         * See http://jira.jboss.com/jira/browse/JBREM-925
         */
        legacySSLConfigNameMappings.put("ClientAuthMode", SSLSocketBuilder.REMOTING_CLIENT_AUTH_MODE);
        legacySSLConfigNameMappings.put("KeyAlias", SSLSocketBuilder.REMOTING_KEY_ALIAS);
        legacySSLConfigNameMappings.put("KeyPassword", SSLSocketBuilder.REMOTING_KEY_PASSWORD);
        legacySSLConfigNameMappings.put("KeyStoreAlgorithm", SSLSocketBuilder.REMOTING_KEY_STORE_ALGORITHM);
        legacySSLConfigNameMappings.put("KeyStorePassword", SSLSocketBuilder.REMOTING_KEY_STORE_PASSWORD);
        legacySSLConfigNameMappings.put("KeyStoreType", SSLSocketBuilder.REMOTING_KEY_STORE_TYPE);
        legacySSLConfigNameMappings.put("KeyStoreURL", SSLSocketBuilder.REMOTING_KEY_STORE_FILE_PATH);
        legacySSLConfigNameMappings.put("ProviderName", SSLSocketBuilder.REMOTING_SSL_PROVIDER_NAME);
        legacySSLConfigNameMappings.put("SecureSocketProtocol", SSLSocketBuilder.REMOTING_SSL_PROTOCOL);
        legacySSLConfigNameMappings.put("ServerAuthMode", SSLSocketBuilder.REMOTING_SERVER_AUTH_MODE);
        legacySSLConfigNameMappings.put("ServerSocketUse-ClientMode", SSLSocketBuilder.REMOTING_SERVER_SOCKET_USE_CLIENT_MODE);
        legacySSLConfigNameMappings.put("SocketUseClient-Mode", SSLSocketBuilder.REMOTING_SOCKET_USE_CLIENT_MODE);
        legacySSLConfigNameMappings.put("TrustStoreAlgorithm", SSLSocketBuilder.REMOTING_TRUST_STORE_ALGORITHM);
        legacySSLConfigNameMappings.put("TrustStorePassword", SSLSocketBuilder.REMOTING_TRUST_STORE_PASSWORD);
        legacySSLConfigNameMappings.put("TrustStoreType", SSLSocketBuilder.REMOTING_TRUST_STORE_TYPE);
        legacySSLConfigNameMappings.put("TrustStoreURL", SSLSocketBuilder.REMOTING_TRUST_STORE_FILE_PATH);
    }
}
