/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.rules;

import java.util.Collection;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.log4j.Logger;
import org.drools.KnowledgeBase;
import org.drools.agent.KnowledgeAgent;
import org.drools.runtime.StatefulKnowledgeSession;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.lifecycle.LifecyclePriorities;
import org.jboss.soa.esb.lifecycle.LifecycleResource;
import org.jboss.soa.esb.lifecycle.LifecycleResourceException;
import org.jboss.soa.esb.lifecycle.LifecycleResourceFactory;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.rules.RuleInfo;
import org.jboss.soa.esb.services.rules.RuleService;
import org.jboss.soa.esb.services.rules.StatefulRuleInfo;

/**
 * JBossRules (aka Drools) Implementation of a rule engine interface for rules services. Here we use
 * <p/>
 *
 * @author jdelong@redhat.com
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class DroolsRuleService implements RuleService
{
    private static Logger log = Logger.getLogger(DroolsRuleService.class);

    /**
	 * The lifecycle resource state factory for RuleBases.
	 */
	private static final LifecycleResourceFactory<Map<String, DroolsRuleBaseState>> lifecycleRuleBaseStateFactory = new LifecycleRuleBaseStateFactory();

    /**
	 * The lifecycle resource factory for RuleAgents.
	 */
	private static final LifecycleResourceFactory<Map<String, KnowledgeAgent>> lifecycleRuleAgentFactory = new LifecycleRuleAgentFactory();

	/**
	 * The lifecycle resource factory rule sets.
	 */
	private static final LifecycleResourceFactory<ConcurrentHashMap<String, String>> lifecycleRuleSetFactory = new LifecycleRuleSetFactory();

	/**
	 * The lifecycle resource rule base states.
	 */
	private static final LifecycleResource<Map<String, DroolsRuleBaseState>> lifecycleRuleBaseStates = new LifecycleResource<Map<String, DroolsRuleBaseState>>(
			lifecycleRuleBaseStateFactory, LifecyclePriorities.RULE_BASE_PRIORITY);

	/**
	 * RuleAgents cache
	 */
	private static final LifecycleResource<Map<String, KnowledgeAgent>> lifecycleRuleAgents = new LifecycleResource<Map<String, KnowledgeAgent>>(
			lifecycleRuleAgentFactory, LifecyclePriorities.RULE_BASE_PRIORITY);

	/**
	 * The lifecycle resource for ruleset.
	 */
	private static final LifecycleResource<ConcurrentHashMap<String, String>> lifecycleRuleSets = new LifecycleResource<ConcurrentHashMap<String, String>>(
			lifecycleRuleSetFactory, LifecyclePriorities.RULE_BASE_PRIORITY);
	
	/**
	 * Execute rules using the Stateless rule engine API.
	 *
	 * @param ruleInfo The {@link RuleInfo} object contain processing instructions for the rule engine. Must not be null.
	 * @param message The ESB Message object.
	 *
	 */
	public Message executeStatelessRules(final RuleInfo ruleInfo, final Message message) throws RuleServiceException
	{
		AssertArgument.isNotNull(ruleInfo, "ruleInfo" );
		final DroolsRuleBaseState ruleBaseState = getRuleBaseStateForFileBasedRules(ruleInfo);
		return ruleBaseState.executeStatelessRules(ruleInfo, message);
	}

	public Message executeStatelessRulesFromDecisionTable(final RuleInfo ruleInfo, final Message message) throws RuleServiceException
	{
		AssertArgument.isNotNull(ruleInfo, "ruleInfo" );
		DroolsRuleBaseState ruleBaseState = getRuleBaseStateForDecisionTable(ruleInfo);
		return ruleBaseState.executeStatelessRules(ruleInfo, message);

	}

	public Message executeStatelessRulesFromRuleAgent(final RuleInfo ruleInfo, final Message message) throws RuleServiceException
	{
		AssertArgument.isNotNull(ruleInfo, "ruleInfo" );
		try
        {
            final DroolsRuleBaseState ruleBaseState = getRuleBaseStateForRuleAgent(ruleInfo);
            return ruleBaseState.executeStatelessRules(ruleInfo, message);
        }
        catch ( final Exception e)
        {
			final String err = "RuleAgent could not get the RuleBase: " + e.getMessage();
			log.error(err, e);
			throw new RuleServiceException(err, e);
        }

	}

	/**
	 * Execute rules using a certain ruleSet and domain specific language using the Stateful rule engine API
	 *
	 * @param info {@link StatefulRuleInfo} containing the execution information for the Rule engine.
	 * @param msg The ESB Message object.
	 * @return {@link Message} The ESB Message Object.
	 *
	 * @throws RuleServiceException If a problem occurs while trying to create the Drools RuleBase or if an exception
	 *                             occurs during execution
	 */
	public Message executeStatefulRules(final StatefulRuleInfo ruleInfo, final Message msg) throws RuleServiceException
    {
		AssertArgument.isNotNull(ruleInfo, "ruleInfo");
		final DroolsRuleBaseState ruleBaseState = getRuleBaseStateForFileBasedRules(ruleInfo);
		return ruleBaseState.executeStatefulRules(ruleInfo, msg);
    }

	/**
	 * Execute rules from a decision table using the Stateful rule engine API
	 *
	 * @param info {@link StatefulRuleInfo} containing the execution information for the Rule engine.
	 * @param msg The ESB Message object.
	 * @return {@link Message} The ESB Message Object.
	 *
	 * @throws RuleServiceException If a problem occurs while trying to create the Drools RuleBase or if an exception
	 */
    public Message executeStatefulRulesFromDecisionTable(final StatefulRuleInfo ruleInfo, final Message msg) throws RuleServiceException
    {
    	AssertArgument.isNotNull(ruleInfo, "ruleInfo");
		final DroolsRuleBaseState ruleBaseState = getRuleBaseStateForDecisionTable(ruleInfo);
		return ruleBaseState.executeStatefulRules(ruleInfo, msg);
    }

	public Message executeStatefulRulesFromRuleAgent(final StatefulRuleInfo ruleInfo, final Message msg) throws RuleServiceException
    {
		AssertArgument.isNotNull(ruleInfo, "ruleInfo");
	    try
        {
            final DroolsRuleBaseState ruleBaseState = getRuleBaseStateForRuleAgent(ruleInfo) ;
            return ruleBaseState.executeStatefulRules(ruleInfo, msg);
        }
        catch (RuleServiceException e)
        {
            throw e;
        }
        catch (Exception e)
        {
			final String err = "RuleAgent could not get the RuleBase: " + e.getMessage();
			log.error(err, e);
			throw new RuleServiceException(err, e);
        }
    }

    public Message continueStatefulRulesExecution(final StatefulRuleInfo ruleInfo, final Message msg) throws RuleServiceException
    {
    	AssertArgument.isNotNull(ruleInfo, "ruleInfo");
		try
		{
			final Map<String, DroolsRuleBaseState> ruleBaseStates = getCachedRuleBaseStates();
			final DroolsRuleBaseState ruleBaseState = ruleBaseStates.get(ruleInfo.getRuleSource());
			return ruleBaseState.executeStatefulRules(ruleInfo, msg);
		}
		catch (Exception e)
		{
			final String err = "Could not continue rule execution: " + e.getMessage();
			log.warn(err);
			throw new RuleServiceException(err, e);
		}
    }

	public void setConfigTree( final ConfigTree configTree )
	{
	}

	//	package protected methods

	/**
	 * Determine if file based rules need reloading and return the rulebase state
	 *
	 * @param ruleInfo
	 *
	 * @return Message with updated objects.
	 */
	DroolsRuleBaseState getRuleBaseStateForFileBasedRules(final RuleInfo ruleInfo) throws RuleServiceException
	{
		final String ruleSet = ruleInfo.getRuleSource();
		final String dsl = ruleInfo.getDslSource();
		final boolean ruleReload = ruleInfo.getReload();
		try
		{
			final DroolsRuleBaseHelper rbHelper = DroolsRuleBaseHelper.getInstance();


			final ConcurrentHashMap<String, String> ruleSets = lifecycleRuleSets.getLifecycleResource();

			synchronized(ruleSets)
			{
				String newRuleSet = null;
				if ( ruleReload )
				{
					newRuleSet = rbHelper.getRulesAsString( ruleSet, dsl );
					String oldRuleSet = ruleSets.put( ruleSet, newRuleSet );
					if ((oldRuleSet != null) && !oldRuleSet.equals(newRuleSet) )
					{
						final DroolsRuleBaseState ruleBaseState = getCachedRuleBaseStates().remove(ruleSet);
						if (ruleBaseState != null)
						{
							ruleBaseState.dispose() ;
						}
					}
				}

				
				final Map<String, DroolsRuleBaseState> ruleBaseStates = getCachedRuleBaseStates();
				DroolsRuleBaseState ruleBaseState = ruleBaseStates.get(ruleSet) ;
				if (ruleBaseState == null)
				{
					final KnowledgeBase ruleBase = rbHelper.createRuleBaseFromRuleFiles(ruleInfo);
					if (newRuleSet == null)
						newRuleSet = rbHelper.getRulesAsString(ruleSet, dsl);
					ruleSets.put(ruleSet, newRuleSet);
					ruleBaseState = new DroolsRuleBaseState(ruleBase) ;
					ruleBaseStates.put(ruleSet, ruleBaseState) ;
				}

				return ruleBaseState;
			}
		}
		catch (final LifecycleResourceException e)
		{
			throw new RuleServiceException("Could not load lifecycle data. " + e.getMessage(), e);
		}
	}

	/**
	 * Determine if decision table need reloading and return the rulebase state
	 *
	 * @param decisionTable -
	 *            String reference to a file which contains a decision table.
	 * @param ruleReload -
	 *            if set to true, a ruleSet update should result in reloading the
	 *            ruleSet.
	 *
	 * @return Message with updated objects.
	 */
	DroolsRuleBaseState getRuleBaseStateForDecisionTable(RuleInfo ruleInfo) throws RuleServiceException {

		try
		{
			final String decisionTable = ruleInfo.getRuleSource();
			final boolean ruleReload = ruleInfo.getReload();
			
            Map<String, DroolsRuleBaseState> ruleBaseStates = getCachedRuleBaseStates();

            DroolsRuleBaseState ruleBaseState = (!ruleReload ? ruleBaseStates.get( decisionTable ) : null) ;
            if (ruleBaseState == null)
            {
                synchronized (ruleBaseStates)
                {
                    if (!ruleReload)
                    {
                        ruleBaseState = ruleBaseStates.get( decisionTable );
                    }
                    
                    if (ruleBaseState == null)
                    {
                        final KnowledgeBase ruleBase = DroolsRuleBaseHelper.getInstance().createRuleBaseFromDecisionTable(ruleInfo);
                        ruleBaseState = new DroolsRuleBaseState(ruleBase) ;
                        final DroolsRuleBaseState origRuleBasedState = ruleBaseStates.put( decisionTable, ruleBaseState );
                        if (origRuleBasedState != null)
                        {
                            origRuleBasedState.dispose() ;
                        }
                    }
                }
            }
            return ruleBaseState;
        }
		catch (final LifecycleResourceException e)
		{
			throw new RuleServiceException("Caught a LifecycleResourceException :", e);
		}
	}
	
	DroolsRuleBaseState getRuleBaseStateForRuleAgent( final RuleInfo ruleInfo ) throws RuleServiceException
	{
		final String ruleAgentProperties = ruleInfo.getRuleSource();
		
		Map<String, KnowledgeAgent> ruleAgents;
		try
		{
			ruleAgents = getCachedRuleAgents();
		}
		catch (final LifecycleResourceException e)
		{
			throw new RuleServiceException("Caught a LifecycleResourceException :", e);
		}
		KnowledgeAgent ruleAgent = ruleAgents.get( ruleAgentProperties );
		if ( ruleAgent == null )
		{
			synchronized (ruleAgents)
			{
				ruleAgent = ruleAgents.get( ruleAgentProperties );
				if (ruleAgent == null)
				{
					ruleAgent = DroolsRuleBaseHelper.getInstance().createRuleAgent( ruleInfo );
					ruleAgents.put( ruleAgentProperties, ruleAgent );
				}
			}
		}

		KnowledgeBase currentRuleBase = ruleAgent.getKnowledgeBase();
		//	always update the cache as the rulebase might have been updated.
		final Map<String, DroolsRuleBaseState> ruleBaseStates;
		try
		{
			ruleBaseStates = getCachedRuleBaseStates() ;
		}
		catch (final LifecycleResourceException e)
		{
			throw new RuleServiceException("Caught a LifecycleResourceException :", e);
		}
		DroolsRuleBaseState ruleBaseState = ruleBaseStates.get(ruleAgentProperties) ;
		if ((ruleBaseState == null) || (ruleBaseState.getRuleBase() != currentRuleBase))
		{
			synchronized(ruleBaseStates)
			{
				ruleBaseState = ruleBaseStates.get(ruleAgentProperties) ;
				currentRuleBase = ruleAgent.getKnowledgeBase();
				if ((ruleBaseState == null) || (ruleBaseState.getRuleBase() != currentRuleBase))
				{
					if (ruleBaseState != null)
					{
						ruleBaseState.dispose() ;
					}
					ruleBaseState = new DroolsRuleBaseState(currentRuleBase) ;
					ruleBaseStates.put(ruleAgentProperties, ruleBaseState) ;
				}
			}
		}
		return ruleBaseState;
	}

	Map<String, KnowledgeAgent> getCachedRuleAgents() throws LifecycleResourceException
	{
		return lifecycleRuleAgents.getLifecycleResource();
	}

	Map<String, DroolsRuleBaseState> getCachedRuleBaseStates() throws LifecycleResourceException
	{
		return lifecycleRuleBaseStates.getLifecycleResource();
	}

	/**
	 * The lifecycle resource factory for rule base state.
	 *
	 * @author kevin
	 */
	public static class LifecycleRuleBaseStateFactory implements LifecycleResourceFactory<Map<String, DroolsRuleBaseState>>
	{
		/**
		 * Create a resource object which will be associated with the
		 * specified lifecycle identity.
		 *
		 * @param lifecycleIdentity
		 *            The associated lifecycle identity.
		 * @return The lifecycle resource
		 * @throws LifecycleResourceException
		 *             for errors during construction.
		 */
		public Map<String, DroolsRuleBaseState> createLifecycleResource( final String lifecycleIdentity) throws LifecycleResourceException
		{
			return new ConcurrentHashMap<String, DroolsRuleBaseState>();
		}

		/**
		 * Destroy a resource object which is associated with the specified lifecycle identity.
		 *
		 * @param resource
		 *            The lifecycle resource.
		 * @param lifecycleIdentity
		 *            The associated lifecycle identity.
		 * @return The lifecycle resource.
		 * @throws LifecycleResourceException
		 *             for errors during destroy.
		 */
		public void destroyLifecycleResource( final Map<String, DroolsRuleBaseState> resource, final String lifecycleIdentity) throws LifecycleResourceException
		{
		    if (resource != null)
		    {
		        final Collection<DroolsRuleBaseState> values = resource.values();
		        for (DroolsRuleBaseState ruleBaseState : values)
		        {
		            ruleBaseState.dispose() ;
		        }
		        resource.clear();
		    }
		}
	}

	public static class LifecycleRuleAgentFactory implements LifecycleResourceFactory<Map<String, KnowledgeAgent>>
	{
        /**
         * Create a resource object which will be associated with the
         * specified lifecycle identity.
         *
         * @param lifecycleIdentity
         *            The associated lifecycle identity.
         * @return The lifecycle resource
         * @throws LifecycleResourceException
         *             for errors during construction.
         */
        public Map<String, KnowledgeAgent> createLifecycleResource( final String lifecycleIdentity) throws LifecycleResourceException
        {
        	return new ConcurrentHashMap<String, KnowledgeAgent>();
        }

        /**
         * Destroy a resource object which is associated with the specified
         * lifecycle identity.
         *
         * @param resource
         *            The lifecycle resource.
         * @param lifecycleIdentity
         *            The associated lifecycle identity.
         * @return The lifecycle resource.
         * @throws LifecycleResourceException
         *             for errors during destroy.
         */
        public void destroyLifecycleResource( final Map<String, KnowledgeAgent> resource, final String lifecycleIdentity) throws LifecycleResourceException
        {
            if (resource != null)
            {
                final Collection<KnowledgeAgent> values = resource.values();
                for (KnowledgeAgent ruleAgent : values)
                {
                	String msg_suffix = "rule agent [" + ruleAgent.getName() + "] for " + lifecycleIdentity;
                    if (log.isInfoEnabled())
                    {
                        log.info("destroying " + msg_suffix);
                    }
                    KnowledgeBase kbase = ruleAgent.getKnowledgeBase();
                    if (kbase != null)
                    {
                    	Collection<StatefulKnowledgeSession> skss = kbase.getStatefulKnowledgeSessions();
                    	if (skss != null)
                    	{
                    		for (StatefulKnowledgeSession sks : skss)
                    		{
                    			if (sks != null)
                    			{
                    				sks.dispose();
                    			}
                    		}
                    	}
                    }
                    ruleAgent.dispose();
                }
                resource.clear();
            }
        }
    }

	/**
	 * The lifecycle resource factory for rule sets.
	 *
	 * @author kevin
	 */
	public static class LifecycleRuleSetFactory implements LifecycleResourceFactory<ConcurrentHashMap<String, String>>
	{
		/**
		 * Create a resource object which will be associated with the
		 * specified lifecycle identity.
		 *
		 * @param lifecycleIdentity
		 *            The associated lifecycle identity.
		 * @return The lifecycle resource
		 * @throws LifecycleResourceException
		 *             for errors during construction.
		 */
		public ConcurrentHashMap<String, String> createLifecycleResource( final String lifecycleIdentity) throws LifecycleResourceException
		{
			return new ConcurrentHashMap<String, String>();
		}

		/**
		 * Destroy a resource object which is associated with the specified
		 * lifecycle identity.
		 *
		 * @param resource
		 *            The lifecycle resource.
		 * @param lifecycleIdentity
		 *            The associated lifecycle identity.
		 * @return The lifecycle resource.
		 * @throws LifecycleResourceException
		 *             for errors during destroy.
		 */
		public void destroyLifecycleResource( final ConcurrentHashMap<String, String> resource, final String lifecycleIdentity) throws LifecycleResourceException
		{
		    if (resource != null)
		    {
		        resource.clear();
		    }
		}
	}
}
