package org.jbosson.plugins.jbossesb;

import java.io.File;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Set;

import org.apache.commons.lang.exception.ExceptionUtils;
import org.jboss.on.common.jbossas.JBPMWorkflowManager;
import org.jboss.on.common.jbossas.JBossASPaths;
import org.rhq.core.domain.configuration.Configuration;
import org.rhq.core.domain.configuration.PropertySimple;
import org.rhq.core.domain.content.PackageType;
import org.rhq.core.domain.content.transfer.DeployPackageStep;
import org.rhq.core.domain.content.transfer.RemovePackagesResponse;
import org.rhq.core.domain.content.transfer.ResourcePackageDetails;
import org.rhq.core.domain.resource.ResourceType;
import org.rhq.core.pluginapi.content.ContentContext;
import org.rhq.core.pluginapi.inventory.CreateResourceReport;
import org.rhq.core.pluginapi.inventory.InvalidPluginConfigurationException;
import org.rhq.core.pluginapi.inventory.ResourceContext;
import org.rhq.plugins.jbossas5.ApplicationServerComponent;
import org.rhq.plugins.jbossas5.connection.LocalProfileServiceConnectionProvider;
import org.rhq.plugins.jbossas5.connection.ProfileServiceConnection;
import org.rhq.plugins.jbossas5.connection.ProfileServiceConnectionProvider;
import org.rhq.plugins.jbossas5.connection.RemoteProfileServiceConnectionProvider;
import org.rhq.plugins.jbossas5.deploy.Deployer;
import org.rhq.plugins.jbossas5.deploy.LocalDeployer;
import org.rhq.plugins.jbossas5.deploy.RemoteDeployer;

import com.jboss.jbossnetwork.product.jbpm.handlers.ControlActionFacade;

/**
 * Component for ESB deployments.
 *
 * @author Tom Cunningham
 */
public class ESB5Component extends AbstractESBComponent
{
    static final String HOME_DIR = "homeDir";
    static final String SERVER_HOME_DIR = "serverHomeDir";

    static final String NAMING_URL = "namingURL";
    static final String CREDENTIALS = "credentials";
    static final String PRINCIPAL = "principal";

    private ProfileServiceConnection connection;
    
    private ESB5ContentFacetDelegate contentFacetDelegate;
	
    @Override
    public void start(ResourceContext context) {
 	   super.start(context);
       JBPMWorkflowManager workflowManager = createJbpmWorkflowManager(resourceContext);

 	   connectToProfileService();
 	   
       this.contentFacetDelegate = new ESB5ContentFacetDelegate(workflowManager, configPath,
    		   resourceContext.getContentContext());
    }

    public void stop() {
    	disconnectFromProfileService();
    }
    
    private JBPMWorkflowManager createJbpmWorkflowManager(ResourceContext resourceContext) {
        ContentContext contentContext = resourceContext.getContentContext();
        ControlActionFacade controlActionFacade = null;
        JBPMWorkflowManager workflowManager = new JBPMWorkflowManager(contentContext, controlActionFacade, this
            .getJBossASPaths());
        return workflowManager;
    }
    
    public boolean runningEmbedded() {
        ApplicationServerComponent jass = (ApplicationServerComponent)this.resourceContext.getParentResourceComponent();
        ResourceContext rc = jass.getResourceContext();

        Configuration pluginConfiguration = rc.getPluginConfiguration();
        String namingUrl = pluginConfiguration.getSimpleValue(NAMING_URL, null);
        return namingUrl == null;
    }
     
    private static void validateNamingURL(String namingURL) {
        URI namingURI;
        try {
            namingURI = new URI(namingURL);
        } catch (URISyntaxException e) {
            throw new RuntimeException("Naming URL '" + namingURL + "' is not valid: " + e.getLocalizedMessage());
        }
        if (!namingURI.isAbsolute())
            throw new RuntimeException("Naming URL '" + namingURL + "' is not absolute.");
        if (!namingURI.getScheme().equals("jnp"))
            throw new RuntimeException("Naming URL '" + namingURL
                + "' has an invalid protocol - the only valid protocol is 'jnp'.");
    }
    
    private void connectToProfileService() {
        ApplicationServerComponent jass = (ApplicationServerComponent)this.resourceContext.getParentResourceComponent();
        ResourceContext rc = jass.getResourceContext();

    	if (this.connection != null)
            return;
        // TODO: Check for a defunct connection and if found try to reconnect.
        ProfileServiceConnectionProvider connectionProvider;
        if (runningEmbedded()) {
            connectionProvider = new LocalProfileServiceConnectionProvider();
        } else {
            Configuration pluginConfig = rc.getPluginConfiguration();
            String namingURL = pluginConfig.getSimpleValue(NAMING_URL, null);
            validateNamingURL(namingURL);
            String principal = pluginConfig.getSimpleValue(PRINCIPAL, null);
            String credentials = pluginConfig.getSimpleValue(CREDENTIALS, null);
            connectionProvider = new RemoteProfileServiceConnectionProvider(namingURL, principal, credentials);
        }
        try {
            this.connection = connectionProvider.connect();
        } catch (RuntimeException e) {
            Throwable rootCause = ExceptionUtils.getRootCause(e);
            if (rootCause instanceof SecurityException) {
                if (log.isDebugEnabled()) {
                    log.debug("Failed to connect to Profile Service.", e);
                } else {
                    log.warn("Failed to connect to Profile Service - cause: " + rootCause);
                }
                throw new InvalidPluginConfigurationException(
                        "Values of 'principal' and/or 'credentials' connection properties are invalid.", rootCause);
            }
            log.debug("Failed to connect to Profile Service.", e);
        }
    }

    public void disconnectFromProfileService() {
        if (this.connection != null) {
            try {
                this.connection.getConnectionProvider().disconnect();
            } catch (RuntimeException e) {
                log.debug("Failed to disconnect from Profile Service.", e);
            } finally {
                this.connection = null;
            }
        }
  }
	
  private File resolvePathRelativeToHomeDir(String path) {
	  File configDir = new File(path);
	  if (!configDir.isAbsolute()) {
		  Configuration pluginConfig = this.resourceContext.getPluginConfiguration();
		  String homeDir = pluginConfig.getSimple(HOME_DIR).getStringValue();
		  configDir = new File(homeDir, path);
	  }
	  return configDir;
  }
    
  private JBossASPaths getJBossASPaths() {
      Configuration pluginConfiguration = this.resourceContext.getPluginConfiguration();

      String homeDir = pluginConfiguration.getSimpleValue(HOME_DIR, null);
      String serverHomeDir = pluginConfiguration.getSimpleValue(SERVER_HOME_DIR, null);

      return new JBossASPaths(homeDir, serverHomeDir);
  }
  
  public File getConfigurationPath() {
	  ApplicationServerComponent jass = (ApplicationServerComponent)this.resourceContext.getParentResourceComponent();
	  ResourceContext rc = jass.getResourceContext();
	  Configuration config = rc.getPluginConfiguration();
	  String configurationPath = config.getSimpleValue("serverHomeDir", null);
	  File configPath = new File(configurationPath);
	  if (!configPath.isDirectory()) {
		  throw new InvalidPluginConfigurationException("Configuration path '" + configPath + "' does not exist.");
	  }
    
	  return configPath;
  }
    
  public CreateResourceReport createResource(CreateResourceReport report) {
	  ResourceType resourceType = report.getResourceType();
	  String resourceTypeName = report.getResourceType().getName();

	  // Hack - AbstractDeployer is looking for a "deployExploded" value rather than a 
	  // deployZipped value - if we don't already have these properties, add them in
      ResourcePackageDetails details = report.getPackageDetails();
      Configuration deployTimeConfig = details.getDeploymentTimeConfiguration();

      if (deployTimeConfig.get("deployExploded") == null) {
		  PropertySimple ps = new PropertySimple();
		  ps.setName("deployExploded");
		  ps.setBooleanValue(!deployTimeConfig.getSimple("deployZipped").getBooleanValue());
		  deployTimeConfig.put(ps);
      }
		 
      if (deployTimeConfig.get("deployFarmed") == null) {
		  PropertySimple df = new PropertySimple();
		  df.setName("deployFarmed");
		  df.setBooleanValue(false);
		  deployTimeConfig.put(df);
      }
		  
      if (resourceTypeName.equals(RESOURCE_TYPE_ESB)) {
    	  getDeployer().deploy(report, resourceType);
      } else {
          throw new UnsupportedOperationException("Unknown Resource type: " + resourceTypeName);
      }
      
      disconnectFromProfileService();
      return report;
  }

  protected Deployer getDeployer() {
      ProfileServiceConnection profileServiceConnection = getConnection();
      if (runningEmbedded()) {
          return new LocalDeployer(profileServiceConnection);
      } else {
          return new RemoteDeployer(profileServiceConnection, this.resourceContext);
      }
  }

  public ProfileServiceConnection getConnection() {
      connectToProfileService();
      return this.connection;
  }

  public Set<ResourcePackageDetails> discoverDeployedPackages(PackageType type) {
      return contentFacetDelegate.discoverDeployedPackages(type);
  }

  public List<DeployPackageStep> generateInstallationSteps(ResourcePackageDetails packageDetails) {
      return contentFacetDelegate.generateInstallationSteps(packageDetails);
  }

  public RemovePackagesResponse removePackages(Set<ResourcePackageDetails> packages) {
      return contentFacetDelegate.removePackages(packages);
  }

  public InputStream retrievePackageBits(ResourcePackageDetails packageDetails) {
      return contentFacetDelegate.retrievePackageBits(packageDetails);
  }  
}