/*
 * JBoss, Home of Professional Open Source
 * Copyright 2010, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */

package org.jboss.soa.esb.server.routing;

import static org.junit.Assert.assertEquals;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import junit.framework.Test;

import org.jboss.test.JBossTestCase;
import org.jboss.internal.soa.esb.util.Encoding;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.routing.http.HttpRouter;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;


/**
 * Test for HttpRouter.
 *
 * @author <a href="mailto:mageshbk@jboss.com">Magesh Kumar B</a>
 */
public class HttpRouterUnitTest extends JBossTestCase
{

    public HttpRouterUnitTest(String name)
    {
        super(name);
    }

    public void testRouter_JavaSerialized() throws Exception
    {
        final String expected = "message: [ JAVA_SERIALIZED ]\n" +
                            "header: [  ]\n" +
                            "context: {}\n" +
                            "body: [ objects: {org.jboss.soa.esb.message.defaultEntry=bar} ]\n" +
                            "fault: [  ]\n" +
                            "attachments: [ Named:{}, Unnamed:[] ]\n" +
                            "properties: [ {} ]";

        String response = postSerializedMessage(MessageType.JAVA_SERIALIZED, expected, true);
        assertEquals(expected, response);
    }

    public void testRouter_XMLSerialized() throws Exception
    {
        final String expected = "<Envelope>" +
                                "<Header xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" />" +
                                "<Context />" +
                                "<Body>" +
                                "<Content><Key>b3JnLmpib3NzLnNvYS5lc2IubWVzc2FnZS5kZWZhdWx0RW50cnk=</Key>" +
                                "<Value>" +
                                "<marshalunmarshal><plugin-type>urn:xml/marshalunmarshal/plugin/serialization</plugin-type>rO0ABXQAA2Jhcg==</marshalunmarshal>" +
                                "</Value>" +
                                "</Content>" +
                                "</Body>" +
                                "<Attachment />" +
                                "<Properties />" +
                                "</Envelope>";

        String response = postSerializedMessage(MessageType.JBOSS_XML, expected, false);
        assertTrue("Serialized Message XML does not match",
                XMLHelper.compareXMLContent(new ByteArrayInputStream(expected.getBytes()), new ByteArrayInputStream(response.getBytes())));
    }

    private String postSerializedMessage(URI type, String expected, Boolean deserialize) throws Exception
    {
        final ConfigTree tree = new ConfigTree("WrappedMessage");
        tree.setAttribute("endpointUrl", "http://127.0.0.1:8080/esb-echo");
        tree.setAttribute("method", "post");
        tree.setAttribute("unwrap", "false");
        tree.setAttribute("MappedHeaderList", "SOAPAction, Content-Type, Accept, If-Modified-Since");
        
        HttpRouter router = new HttpRouter(tree);
        Message message = MessageFactory.getInstance().getMessage(type);
        message.getBody().add("bar");

        Message response = router.process(message);
        String responseBody = (String)response.getBody().get();
        String responseStr = null;
        if (deserialize)
            responseStr = Encoding.decodeToObject(responseBody).toString();
        else
            responseStr = responseBody;

        return responseStr;
    }

    public static Test suite() throws Exception
    {
        return getDeploySetup(HttpRouterUnitTest.class, "esb-echo.war");
    }

}
