/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.util;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;


/**
 * Context containing message flow priority.
 *
 * @author <a href="mailto:Kevin.Conner@jboss.com">Kevin Conner</a>
 *
 */
public final class MessageFlowContext
{
	private static ThreadLocal<Integer> messageFlowPriorityTL = new ThreadLocal<Integer>() ;
	
	private static final int MIN_PRIORITY = 0 ;
    private static final int MAX_PRIORITY = 9 ;

	/**
	 * Set the message flow priority
	 * @param messageFlowPriority The message flow priority
	 */
    public static void setMessageFlowPriority(final Integer messageFlowPriority)
    {
        messageFlowPriorityTL.set(messageFlowPriority) ;
    }

    /**
     * Get the message flow priority
     * @return The message flow priority or null if default
     */
    public static Integer getMessageFlowPriority()
    {
        return messageFlowPriorityTL.get();
    }
    
    /**
     * Retrieve message flow priority from configuration
     * @param configTree The current configuration
     * @return The message flow priority or null if not configured
     */
    public static Integer parseMessageFlowPriority(final ConfigTree config)
        throws ConfigurationException
    {
        final String priorityValue = config.getAttribute(ListenerTagNames.MESSAGE_FLOW_PRIORITY) ;
        if (priorityValue != null)
        {
            return parseMessageFlowPriority(priorityValue) ;
        }
        else
        {
            return null ;
        }
    }
    
    /**
     * Retrieve message flow priority from configuration
     * @param priorityValue The priority value.
     * @return The message flow priority or null if not configured
     */
    public static Integer parseMessageFlowPriority(final String priorityValue)
        throws ConfigurationException
    {
        final int priority ;
        try
        {
            priority = Integer.parseInt(priorityValue) ;
        }
        catch (final NumberFormatException nfe)
        {
            throw new ConfigurationException("Could not parse message flow priority " + priorityValue) ;
        }
        if ((priority < MIN_PRIORITY) || (priority > MAX_PRIORITY))
        {
            throw new ConfigurationException("Invalid gateway priority specified: " + priority) ;
        }
        else
        {
            return Integer.valueOf(priority) ;
        }
    }
}
