package org.jboss.soa.esb.addressing;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */


/**
 * This class represents the endpoint reference for services.
 */

import java.net.URI;
import java.net.URISyntaxException;

import org.apache.log4j.Logger;

/**
 * The Endpoint Reference class. All services (and clients) can be represented by
 * an EPR, which is effectively an address. If using SOA principles then the ultimate
 * recipient of the message should be addressed in a loosely-coupled manner: the service
 * should multiplex/demultiplex work across "objects" based on the message content and
 * the EPR should not address a specific "object".
 * 
 * @author marklittle
 *
 */

public class EPR
{
	/**
	 * Create a new Endpoint Reference with a null address.
	 */
	
	public EPR ()
	{
		_addr = new PortReference();
	}

	/**
	 * Create a new Endpoint Reference with a null address.
	 */
	protected EPR(final String type)
	{
		this() ;
		setType(type) ;
	}
	
	/**
	 * Create a new Endpoint Reference with the specified address.
	 * 
	 * @param addr the specified address.
	 */
	
	public EPR (PortReference addr)
	{
		_addr = addr;
	}

	/**
	 * Create a new Endpoint Reference with the specified address.
	 * 
	 * @param uri the specified address.
	 */
	
	public EPR (URI uri)
	{
		_addr = new PortReference(uri.toString());
	}
	
	/**
	 * Create a new Endpoint Reference with the specified address.
	 * 
	 * @param uri the specified address.
	 * @param type The EPR type
	 */
	protected EPR(final URI uri, final String type)
	{
		this(uri) ;
		setType(type) ;
	}
	
	/**
	 * Copy constructor.
	 * 
	 * @param from
	 */
	
	public EPR (EPR from)
	{
		/*
		 * Copy the underlying EPR values.
		 */
		
		_addr = (PortReference) from._addr.copy();
	}
	
	/**
	 * Override the address of this EPR.
	 * 
	 * @param uri the new address.
	 */
	
	public void setAddr (PortReference uri)
	{
		_addr = uri;
	}

        /**
         * Set the URI for this endpoint.
         * 
         * @param uri the address.
         */
        
        public void setURI(URI uri)
        {
                setAddr(new PortReference(uri.toASCIIString()));
        }
        
        /**
         * Get the URI address.
         * 
         * @return the address.
         * @throws URISyntaxException thrown if the address is invalid.
         */
        
        public URI getURI() throws URISyntaxException
        {
                return new URI(getAddr().getAddress());
        }
	
	/**
	 * Get the EPR address.
	 * 
	 * @return the address.
	 */
	
	public PortReference getAddr ()
	{
		return _addr;
	}

	/**
	 * Set the EPR type.
	 * @param type
	 */
	protected void setType(final String type)
	{
		getAddr().addExtension(EPR_TYPE, type) ;
	}
	
	/**
	 * Return a copy of this EPR.
	 */

	public EPR copy ()
	{
	    return new EPR(this);
	}

	public String toString ()
	{
		return "EPR: "+_addr.extendedToString();
	}

	public boolean equals (Object obj)
	{
		if (obj == this)
			return true;
		else
		{
			if (obj instanceof EPR)
				return ((EPR) obj)._addr.equals(_addr);
		}
		
		return false;
	}
	
	
	/**
	 * Return a hash code for this element.
	 * @return the element hash code.
	 */
	@Override
	public int hashCode()
	{
		return _addr.hashCode() ;
	}

        static protected final Logger _logger = Logger.getLogger(EPR.class);
        
	private PortReference _addr;
	public static final String EPR_TYPE = "type" ;

} 