/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */
/*
 * Copyright (C) 2002
 *
 * Arjuna Solutions Limited,
 * Newcastle upon Tyne,
 * Tyne and Wear,
 * UK.
 *
 * $Id: arjPropertyManager.java 2342 2006-03-30 13:06:17Z  $
 */

package org.jboss.soa.esb.common;

import com.arjuna.common.internal.util.propertyservice.PropertyManagerImpl;
import com.arjuna.common.internal.util.propertyservice.plugins.io.XMLFilePlugin;
import com.arjuna.common.util.exceptions.LoadPropertiesException;
import com.arjuna.common.util.propertyservice.PropertyManager;
import com.arjuna.common.util.propertyservice.PropertyManagerFactory;
import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.FatalError;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Hashtable;

/**
 * The ESB configuration file is split into modules, each concerned with a
 * specific capability. Each module may also depend upon another module. We use
 * a PropertyManager per module to manage properties and relationships.
 * 
 * All properties are grouped within <properties/> sections and individual
 * properties by <property/>. If you look at some you'll see that they may be
 * named and may have a depends value, which relates to another section. This
 * tells the PropertyManager to load the depends section first. A property value
 * may be overridden by a dependant section, but the PropertyManager remembers
 * the hierarchy, so if you want to you can inspect and change the lower level
 * value and the PropertyManager can be used to preserve these changes if the
 * values are written back out to the source location.
 * 
 * The way the PropertyManager works means that it builds up a list of dependant
 * modules in a directed acyclic graph. This means that there is a root, which
 * can see every property below it, but nodes below cannot see properties above
 * them: there is a level of isolation.
 * 
 * When creating PropertyManagers, you need to specify where in the graph it is
 * to "tie in". This obviously has an affect on what that instance can then see
 * or do: if it goes in at the wrong level, it won't be able to see properties
 * it may need (unless they were defined to be "system" properties, and in which
 * case they are globally accessible.)
 * 
 * This means that we now have isolated PropertyManagers for each module. It
 * helps the management side of things a lot, but does mean that we have to
 * consider where in the graph a property needs to be, or which PropertyManager
 * we want to use to try to access it.
 * 
 * @author marklittle
 * 
 */
public class ModulePropertyManager
{

	/*
	 * The names of the various modules in the property file.
	 */

	public static final String CORE_MODULE = "core";

	public static final String CONNECTION_MODULE = "connection";

	public static final String REGISTRY_MODULE = "registry";

	public static final String TRANSPORTS_MODULE = "transports";

	public static final String DBSTORE_MODULE = "dbstore";
	
	public static final String FILTER_MODULE = "filters";
	
	public static final String SECURITY_MODULE = "security";
	
	public static final String JCA_MODULE = "jca" ;

	public static final String RULES_MODULE = "rules";

	public static final String INTERCEPTOR_MODULE = "interceptors";
	
	/**
	 * Get the property manager responsible for the module component in the
	 * file.
	 * 
	 * @param name
	 *            the name of the module.
	 * @return the relevant property manager.
	 */

	public static PropertyManager getPropertyManager(String name)
	{
		synchronized (managers)
		{
			PropertyManager theManager = managers.get(name);

			if (theManager == null)
			{
				theManager = initialise(name);

				if (theManager != null)
				{
					managers.put(name, theManager);
				}
			}

			return theManager;
		}
	}

	private static final PropertyManager initialise(String name)
	{
		/**
		 * Retrieve the property manager from the factory and add the ESB
		 * properties file to it.
		 * 
		 * set com.arjuna.common.util.propertyservice.verbosePropertyManager=ON
		 * for verbose output from the PropertyManager.
		 */

		try
		{
			PropertyManager propertyManager = PropertyManagerFactory.getPropertyManager("org.jboss.soa.esb.propertymanager." + name, name);
	
			String propertiesFilename = System.getProperty(Environment.PROPERTIES_FILE, Environment.DEFAULT_PROPERTY_FILE);
	
			/*
			 * Does not cause reloading of the same file over and over. Once it is
			 * loaded, subsequent attempts to reload are ignored internally.
			 */
	
			try
			{
				propertyManager.load(XMLFilePlugin.class.getName(), propertiesFilename);
			}
			catch (LoadPropertiesException ex)
			{
				_logger.fatal("ModulePropertyManager failed to load property file "+propertiesFilename);
				
				throw new FatalError(ex);
			}
			catch (ClassNotFoundException e)
			{
				// something seriously wrong; better to terminate.
				
				_logger.fatal("ModulePropertyManager failed to load XML plugin", e);
				
				throw new FatalError(e);
			}
	
			return propertyManager;
		}
		catch (Exception e)
		{
			// something seriously wrong; better to terminate.
			
			_logger.fatal("ModulePropertyManager failed to load PropertyManager", e);
			
			throw new FatalError(e);
		}
	}

    public static void configure(InputStream esbConfig) throws IOException, SAXException {
        AssertArgument.isNotNull(esbConfig, "esbConfig");

        // Clear all existing PropertyManager instances...
        managers.clear();

        // Add the new PropertyManager instances...
        Document configDoc = YADOMUtil.parseStream(esbConfig, false, false);
        NodeList propertiesList = configDoc.getElementsByTagName("properties");
        for(int i = 0; i < propertiesList.getLength(); i++) {
            Element properties = (Element) propertiesList.item(i);
            String name = properties.getAttribute("name");
            PropertyManager propertyManager = new PropertyManagerImpl(name);
            NodeList propertyList = properties.getElementsByTagName("property");

            addProperties(propertyList, propertyManager);
            managers.put(name, propertyManager);
        }
    }

    private static void addProperties(NodeList propertyList, PropertyManager propertyManager) {
        for(int i = 0; i < propertyList.getLength(); i++) {
            Element property = (Element) propertyList.item(i);
            String name = property.getAttribute("name");
            String value = property.getAttribute("value");

            propertyManager.setProperty(name, value);
        }
    }

    public static Hashtable<String, PropertyManager> getManagers() {
        return managers;
    }

    public static void setManagers(Hashtable<String, PropertyManager> managers) {
        ModulePropertyManager.managers = managers;
    }

    private static Hashtable<String, PropertyManager> managers = new Hashtable<String, PropertyManager>();
	
	private static Logger _logger = Logger.getLogger(ModulePropertyManager.class);
}
