/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.util.EPRManager;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.services.registry.Registry;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.RegistryFactory;
import org.jboss.soa.esb.services.registry.ServiceNotFoundException;

import com.arjuna.common.util.propertyservice.PropertyManager;

/**
 * Extraction of common registry utility methods.
 * @author kevin
 */
public class RegistryUtil
{
    /**
     * The logger for this class.
     */
    private static final Logger logger = Logger.getLogger(RegistryUtil.class) ;

    /**
     * The lock for the registry.
     */
    private static final Lock registryLock = new ReentrantLock() ;
    
    /**
     * Get the EPR manager.
     * @return The EPR manager.
     */
    public static EPRManager getEprManager()
    {
        final PropertyManager manager = ModulePropertyManager.getPropertyManager(ModulePropertyManager.CORE_MODULE) ;
        final String dir = manager.getProperty(Environment.REGISTRY_FILE_HELPER_DIR, ".") ;
        return EPRManager.getInstance(dir) ;
    }

    /**
     * Register a service with the EPR manager.
     * @param name The service name
     * @param address The service address
     * @throws RegistryException for registration errors.
     */
    private static void register(final String name, final EPR address)
        throws RegistryException
    {
        try
        {
            getEprManager().saveEPR(name, address) ;
        }
        catch (final IOException ioe)
        {
            logger.error("Cannot register service", ioe) ;
            throw new RegistryException("Cannot register service", ioe) ;
        }
    }

    /**
     * Unregister a service with the EPR manager.
     * @param name The service name
     */
    private static void unregister(final String name)
    {
        try
        {
            getEprManager().removeEPR(name) ;
        }
        catch (final IOException ioe)
        {
            logger.warn("Failed to unregister service", ioe) ;
        }
    }

    /**
     * Register an EPR in the registry.
     * 
     * @param config a config tree containing the deployment-configuration of the service.
     * @param epr the epr (EndPoint Reference) of the service.
     * 
     * @throws RegistryException for registration errors.
     */
    public static void register(final ConfigTree config, final EPR epr)
        throws RegistryException
    {
        final String category = config.getAttribute(ListenerTagNames.SERVICE_CATEGORY_NAME_TAG) ;
        final String name = config.getAttribute(ListenerTagNames.SERVICE_NAME_TAG) ;
        final String removeOldService = config.getAttribute(ListenerTagNames.REMOVE_OLD_SERVICE);
        
        if ("eprManager".equalsIgnoreCase(category))
        {
            register(name, epr) ;
        }
        else
        {
            final String serviceDescription = config.getAttribute(ListenerTagNames.SERVICE_DESCRIPTION_TAG) ;
            final String eprDescription = config.getAttribute(ListenerTagNames.EPR_DESCRIPTION_TAG) ;
            final Registry registry = RegistryFactory.getRegistry() ;

            if (logger.isDebugEnabled())
            {
                logger.debug("Registering < " + name + ", " + epr + " >") ;
            }

            registryLock.lock();
            try
            {
            	if ("true".equalsIgnoreCase(removeOldService))            	
            		registry.unRegisterService(category, name);            	
            }
            catch (ServiceNotFoundException ex)
            {
            	// ignore as it's possible another client just did the removal for us.
            	logger.debug("removeOldService set : Could not unregister service < "+category+", "+name+" >.");
            }
            
            try {
                registry.registerEPR(category, name,
                        serviceDescription, epr, eprDescription) ;
            }
            finally
            {
                registryLock.unlock() ;
            }
            
        }
    }

    /**
     * Unregister the EPR from the registry.
     * 
     * @param category The category name of the service
     * @param name The name of the service
     * @param epr The service EPR.
     */
    public static void unregister(final String category, final String name, final EPR epr)
    {
        if ("eprManager".equalsIgnoreCase(category))
        {
            unregister(name) ;
        }
        else
        {
            registryLock.lock() ;
            try
            {
                final Registry registry = RegistryFactory.getRegistry() ;
                registry.unRegisterEPR(category, name, epr) ;
            }
            catch (final ServiceNotFoundException snfe) {
                logger.warn("Failed to unregister service", snfe);
            }
            catch (final RegistryException re)
            {
                logger.warn("Failed to unregister service", re) ;
            }
            finally
            {
                registryLock.unlock() ;
            }
        }
    }

    /**
     * Get the EPRs associated with the specified service.
     * @param category The category name of the service
     * @param name The name of the service
     * @return The collection of EPRs associated with the specified service.
     * 
     * @throws RegistryException for registration errors.
     */
    public static List<EPR> getEprs(final String category, final String name)
            throws RegistryException, ServiceNotFoundException
    {
        if ("eprManager".equalsIgnoreCase(category))
        {
            final EPR epr ;
            try
            {
                epr = getEprManager().loadEPR(name) ;
            }
            catch (final IOException ioe)
            {
                throw new RegistryException("No EPRs found for <" + category
                    + "><" + name + ">") ;
            }
            return Arrays.asList(new EPR[] {epr}) ;
        }

        final Registry reg = RegistryFactory.getRegistry() ;
        RegistryException eReg = null ;
        for (int count = 0; count < 5; count++)
        {
            try
            {
                return reg.findEPRs(category, name) ;
            }
            catch (final RegistryException re)
            {
                if (null == eReg)
                    eReg = re ;
                try
                {
                    Thread.sleep(500) ;  // TODO magic number
                }
                catch (final InterruptedException ie)
                {
                    break ;
                }
            }
        }
        throw eReg ;
    }
}
