/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.listeners.config.mappers120;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.jboss.internal.soa.esb.publish.ActionContractPublisher;
import org.jboss.internal.soa.esb.publish.ContractProvider;
import org.jboss.internal.soa.esb.publish.ContractPublisher;
import org.jboss.internal.soa.esb.publish.Publish;
import org.jboss.internal.soa.esb.util.MessageFlowContext;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.config.ServiceContract;
import org.jboss.soa.esb.listeners.config.ServicePublisher;
import org.jboss.soa.esb.listeners.config.WebserviceInfo;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.AbstractScheduledListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.Bus;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.BusProvider;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.DualListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.GatewayOnlyListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.Listener;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.ListenersDocument;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.MepType;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.Provider;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.Schedule;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.ActionDocument.Action;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.ActionsDocument.Actions;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.GlobalsDocument.Globals;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.GlobalsDocument.Globals.WarSecurity;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.JbossesbDocument.Jbossesb;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.PropertyDocument.Property;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.ProvidersDocument.Providers;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.ScheduleProviderDocument.ScheduleProvider;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.ServiceDocument.Service;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.ServicesDocument.Services;
import org.jboss.soa.esb.util.ClassUtil;
import org.w3c.dom.Document;

/**
 * XMLBeans based model implementation.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class XMLBeansModel {

    /**
     * XMLBeans config model instance.
     */
    private Jbossesb jbossesb;

    /**
     * Constructor.
     * @param xmlBeansDoc XMLBeans config model.
     */
    public XMLBeansModel(Jbossesb jbossesb) {
        this.jbossesb = jbossesb;
    }

    /**
     * Get the list of ESB Gateway Listeners from the configuration.
     * @return The list of ESB Gateway Listeners from the configuration.
     * @throws org.jboss.soa.esb.ConfigurationException Bad configuration.
     */
    public List<Listener> getGatewayListeners() throws ConfigurationException {
        return getListeners(true);
    }

    /**
     * Get the list of ESB Aware Listeners from the configuration.
     * @return The list of ESB Aware Listeners from the configuration.
     * @throws org.jboss.soa.esb.ConfigurationException Bad configuration.
     */
    public List<Listener> getESBAwareListeners() throws ConfigurationException {
        return getListeners(false);
    }

    /**
     * Get the Service to which the supplied {@link org.jboss.soa.esb.listeners.config.xbeanmodel110.Listener} configuration instance is bound.
     * @param listener The listener instance (Gateway or ESB Aware).
     * @return The Service to which the
     */
    public Service getService(Listener listener) {
        List<Service> services = getServices();

        for(Service service : services) {
            ListenersDocument.Listeners listeners = service.getListeners();

            if(listeners != null) {
                List<Listener> listenerList = listeners.getAbstractListenerList();

                if(listenerList != null) {
                    for(Listener serviceListener : listenerList) {
                        if(serviceListener == listener) {
                            return service;
                        }
                    }
                }
            }
        }

        throw new IllegalStateException("No Service instance found for the supplied Listener instance.  This should not be possible if the Listener instance was provided by this configuration.  Where has this Listener instance come from?");
    }

    /**
     * Get the Service list.
     * @return Service list.
     */
    public List<Service> getServices() {
        final Services services = jbossesb.getServices();
        if (services != null) {
            return jbossesb.getServices().getServiceList();
        } else {
            return Collections.emptyList();
        }
    }

    /**
     * Get the &lt;bus&gt; configuration matching the supplied busid reference value.
     * @param busid The required &lt;bus&gt; configuration reference value.
     * @return The Bus configuration instance.
     * @throws org.jboss.soa.esb.ConfigurationException Unknown busid reference value.
     */
    public Bus getBus(String busid) throws ConfigurationException {
        Bus bus = getOptionalBus(busid);

        if(bus == null) {
            throw new ConfigurationException("Invalid ESB Configuration: No <bus> configuration matching busid reference value [" + busid + "].");
        }

        return bus;
    }

    /**
     * Get the &lt;bus&gt; configuration matching the supplied busid reference value.
     * @param busid The required &lt;bus&gt; configuration reference value.
     * @return The Bus configuration instance.
     */
    public Bus getOptionalBus(String busid) {
        Providers providers = jbossesb.getProviders();

        if(providers != null) {
            List<Provider> providerList = providers.getProviderList();

            for(Provider provider : providerList) {
                if(provider instanceof BusProvider) {
                    List<Bus> buses = ((BusProvider)provider).getBusList();

                    for(Bus bus : buses) {
                        if(bus.getBusid().equals(busid)) {
                            return bus;
                        }
                    }
                }
            }
        }
        
        return null;
    }

    /**
     * Get the &lt;provider&gt; configuration containing the supplied Bus configuration instance.
     * @param bus The Bus config instance whose Provider is being sought.
     * @return The Provider configuration instance.
     */
    public Provider getProvider(Bus bus) {
        Providers providers = jbossesb.getProviders();

        if(providers != null) {
            List<Provider> providerList = providers.getProviderList();

            for(Provider provider : providerList) {
                if(provider instanceof BusProvider) {
                    List<Bus> buses = ((BusProvider)provider).getBusList();

                    for(Bus installedBus : buses) {
                        if(installedBus == bus) {
                            return provider;
                        }
                    }
                }
            }
        }
        
        throw new IllegalStateException("No Provider instance found for the supplied Bus config instance.  This should not be possible if the Bus instance was provided by this configuration.  Where has this Bus instance come from?");
    }

    /**
     * Get the &lt;provider&gt; configuration containing the supplied Bus configuration instance.
     * @param providerType The Bus Provider type is being sought.
     * @return The Provider configuration instance.
     */
    public Provider getProvider(Class<? extends BusProvider> providerType) {
        Providers providers = jbossesb.getProviders();

        if(providers != null) {
            List<Provider> providerList = providers.getProviderList();

            for(Provider provider : providerList) {
                if(providerType.isAssignableFrom(provider.getClass())) {
                    return provider;
                }
            }
        }
        
        return null;
    }

    public Schedule getSchedule(final String id) {
        if((id == null) || (jbossesb.getProviders() == null)) {
            return null;
        }

        List<Provider> providers = jbossesb.getProviders().getProviderList();

        for(Provider provider : providers) {
            if(provider instanceof ScheduleProvider) {
                final ScheduleProvider scheduleProvider = (ScheduleProvider)provider ;
                final List<Schedule> schedules = scheduleProvider.getScheduleList() ;
                for(Schedule schedule: schedules) {
                    if (id.equals(schedule.getScheduleid())) {
                        return schedule ;
                    }
                }
            }
        }

        return null;
    }

    public ScheduleProvider getScheduleProvider() {
        if(jbossesb.getProviders() == null) {
            return null;
        }

        List<Provider> providers = jbossesb.getProviders().getProviderList();

        for(Provider provider : providers) {
            if(provider instanceof ScheduleProvider) {
                return (ScheduleProvider) provider;
            }
        }

        return null;
    }

    public int getScheduledListenerCount() {
        int count = 0 ;
        if (jbossesb.getServices() != null) {
            final List<Service> services = getServices() ;
            for(Service service: services) {
                final ListenersDocument.Listeners listeners = service.getListeners() ;
                if (listeners != null) {
                    for (Listener listener: listeners.getAbstractListenerList()) {
                        if (listener instanceof AbstractScheduledListener) {
                            count++ ;
                        }
                    }
                }
            }
        }
        return count ;
    }

    public static boolean isGateway(Listener listener)
    {
        if(listener instanceof GatewayOnlyListener) {
            return true;
        } else if(listener instanceof DualListener) {
            return ((DualListener)listener).getIsGateway();
        }

        return false;
    }

    /**
     * Get the list of ESB Listeners based on their Gateway flag.
     * @return The list of ESB Aware or Gateway Listeners from the configuration.
     * @param isGateway Is the listener a gateway or ESB aware listener.
     * @throws org.jboss.soa.esb.ConfigurationException Bad configuration.
     */
    private List<Listener> getListeners(boolean isGateway) throws ConfigurationException {
        List<Listener> gateways = new ArrayList<Listener>();
        if (jbossesb.getServices() != null) {
            List<Service> services = getServices();

            for(Service service : services) {
                boolean listenerAdded = false;
                ListenersDocument.Listeners listeners = service.getListeners();

                if(listeners != null) {
                    for(Listener listener : listeners.getAbstractListenerList()) {
                        if(isGateway(listener) == isGateway) {
                            gateways.add(listener);
                            listenerAdded = true;
                        }
                    }
                }

                // Make sure each Service config has a message aware listener...
                // http://jira.jboss.com/jira/browse/JBESB-648
                if(!exposesInVMListener(service) && !isGateway && !listenerAdded) {
                    throw new ConfigurationException("Service configuration for Service '" + service.getCategory() + ":" + service.getName() + "' doesn't define a Message-Aware Listener (i.e. is-gateway='false').");
                }
            }
        }

        return gateways;
    }

    /**
     * Gets the setting for the number of seconds between reloads.
     *
     * @return The param reload seconds config value.
     */
    public String getParameterReloadSecs() {
        return jbossesb.getParameterReloadSecs().getStringValue();
    }

    public static String getProperty(List<Property> properties, String name, String defaultVal) {
        for (Property property : properties) {
            if(property.getName().equals(name)) {
                return property.getValue();
            }
        }

        return defaultVal;
    }

    public static boolean exposesInVMListener(Service service) {
        if(service.xgetInvmScope() != null && service.xgetInvmScope().getStringValue() != null) {
            return (service.xgetInvmScope().getStringValue().equals("GLOBAL"));
        }

        String systemDefaultScope = System.getProperty(Environment.DEFAULT_INVM_SCOPE, "GLOBAL");
        String defaultScope = ModulePropertyManager.getPropertyManager("core").getProperty(Environment.DEFAULT_INVM_SCOPE, systemDefaultScope);

        return defaultScope.equals("GLOBAL");
    }

    /**
     * Verify the schedule provider configuration.
     * @throws org.jboss.soa.esb.ConfigurationException
     */
    public void verifyScheduleProviderConfig()
        throws ConfigurationException {
        Providers providersConfig = jbossesb.getProviders();

        if(providersConfig == null) {
            return;
        }

        List<Provider> providers = providersConfig.getProviderList();
        int numScheduleProviders = 0;

        for(Provider provider : providers) {
            if(provider instanceof ScheduleProvider) {
                numScheduleProviders++;
            }
        }

        if(numScheduleProviders > 1) {
            throw new ConfigurationException("Configuration contains " + numScheduleProviders + " <schedule-provider> configurations.  Only one of this provider type can exist per configuration.");
        }
    }


    /**
     * Get a map of service publishers provided by each service.
     *
     * @return The map of service publishers, keyed by service.
     */
    public Map<org.jboss.soa.esb.Service, List<ServicePublisher>> getServicePublishers() {
        List<Service> serviceConfigs = getServices();
        final Map<org.jboss.soa.esb.Service, List<ServicePublisher>> servicePublishers = new LinkedHashMap<org.jboss.soa.esb.Service, List<ServicePublisher>>() ;

        for (Service docService : serviceConfigs) {
            ContractPublisher publisher = getContractPublisher(docService);
            final org.jboss.soa.esb.Service service = new org.jboss.soa.esb.Service(docService.getCategory(), docService.getName()) ;
            ServicePublisher servicePublisher = new ServicePublisher(service.getName(), service.getCategory(), publisher);

            servicePublisher.setDescription(docService.getDescription());
            addPublisher(servicePublishers, service, servicePublisher) ;
        }

        return servicePublishers ;
    }

    /**
     * Get the contract publisher for the service.
     * @param service The current service definition.
     * @return The contract publisher of null if none present.
     */
    private static ContractPublisher getContractPublisher(Service service) {
        if(service.getActions() == null || service.getActions().getActionList() == null) {
            return null;
        }

        for (Action action : service.getActions().getActionList()) {
            Class<Class> actionClass;

            try {
                actionClass = (Class<Class>) ClassUtil.forName(action.getClass1(), ServicePublisher.class);
            } catch (ClassNotFoundException e) {
                throw new RuntimeException("Failed to find action class '" + action.getClass1() + "'.", e);
            }

            Publish publishAnnotation = (Publish) actionClass.getAnnotation(Publish.class);
            if (publishAnnotation != null) {
                Class publisherClass;
                ActionContractPublisher publisher = null;

                publisherClass = publishAnnotation.value();
                try {
                    publisher = (ActionContractPublisher) publisherClass.newInstance();
                    final org.jboss.soa.esb.listeners.config.Action actionInfo = new org.jboss.soa.esb.listeners.config.Action(
                        action.getName(), action.getClass1(), action.getProcess(), toProperties(action.getPropertyList())) ;

                    if(actionInfo.getProperties().getProperty("publishContract", "true").equals("true")) {
                        publisher.setActionConfig(actionInfo);
                        // JBESB-3034
                        publisher = ActionContractPublisher.ProxyFactory.createContextClassLoaderProxy(publisher);
                        return publisher;
                    }
                } catch (ClassCastException e) {
                    throw new RuntimeException("Action Contract Publisher class '" + publisherClass.getName() + "' must implement " + ActionContractPublisher.class.getName());
                } catch (Exception e) {
                    throw new RuntimeException("Failed to instantiate Contract Publisher '" + publisherClass.getName() + "'. Class must implement a public default constructor.", e);
                }
            }
        }

        // No publisher configured on any of the actions in the processing chain...
        return null;
    }


    /**
     * Add the publisher into map for the specified service.
     * @param servicePublishers The service publishers
     * @param service The service name
     * @param publisher The publisher
     */
    private static void addPublisher(final Map<org.jboss.soa.esb.Service, List<ServicePublisher>> servicePublishers,
            final org.jboss.soa.esb.Service service, final ServicePublisher publisher)
    {
        final List<ServicePublisher> publishers = servicePublishers.get(service) ;
        if (publishers != null) {
            publishers.add(publisher);
        } else {
            final List<ServicePublisher> newPublishers = new ArrayList<ServicePublisher>() ;
            newPublishers.add(publisher) ;
            servicePublishers.put(service, newPublishers) ;
        }
    }

    /**
     * Generate the ESB Aware configuration document.
     * @return The ESB aware configuration.
     * @throws org.jboss.soa.esb.ConfigurationException Error creating configuration.
     */
    public Document generateESBAwareConfig()
        throws ConfigurationException
    {
        // Generate and serialise the configuration for the ESB Aware listeners...
        ESBAwareGenerator awareGenerator = new ESBAwareGenerator(this);
        return awareGenerator.generate();
    }

    /**
     * Generate the gateway configuration document.
     * @return The gateway configuration.
     * @throws org.jboss.soa.esb.ConfigurationException Error creating configuration.
     */
    public Document generateGatewayConfig()
        throws ConfigurationException
    {
        // Generate and serialise the configuration for the Gateway listeners...
        GatewayGenerator gatewayGenerator = new GatewayGenerator(this);
        return gatewayGenerator.generate();
    }

    /**
     * Get the properties from the document.
     * @param configProperties The document properties.
     * @return The properties.
     */
    public static Properties toProperties(List<Property> configProperties) {
        Properties properties = new Properties();

        for(Property property : configProperties) {
            String name = property.getName();
            String value = property.getValue();

            if(value != null) {
                properties.setProperty(name, value);
            }
            // TODO: https://jira.jboss.org/jira/browse/JBESB-2381
        }

        return properties;
    }

    /**
     * Get the list of actions in this deployment.
     * @return a list of actions or null if none present.
     */
    public Set<String> getActions()
    {
        final Services services = jbossesb.getServices() ;
        if (services != null)
        {
            final Set<String> actionClasses = new HashSet<String>() ;

            for(final Service service : services.getServiceList())
            {
                final Actions actions = service.getActions() ;
                if (actions != null)
                {
                    for (final Action action: actions.getActionList())
                    {
                        actionClasses.add(action.getClass1()) ;
                    }
                }
            }

            return actionClasses ;
        }
        return null ;
    }


    /**
     * Get the list of services which require a webservice endpoint.
     * @return The list of services.
     * @throws ConfigurationException 
     */
    public List<WebserviceInfo> getWebserviceServices()
        throws ConfigurationException
    {
        final List<WebserviceInfo> endpointServices = new ArrayList<WebserviceInfo>() ;
        final Services services = jbossesb.getServices() ;
        if (services != null)
        {
            for(final Service service : services.getServiceList())
            {
                final Actions actions = service.getActions() ;
                if (actions != null)
                {
                    if (!actions.isSetWebservice() || actions.getWebservice())
                    {
                        final String inXsd = actions.getInXsd() ;
                        if (inXsd != null)
                        {
                            final WebserviceInfo webserviceInfo = new WebserviceInfo(
                                new org.jboss.soa.esb.Service(service.getCategory(), service.getName()),
                                actions.getInXsd(), actions.getOutXsd(), actions.getFaultXsd(),
                                service.getDescription(), MepType.REQUEST_RESPONSE.equals(actions.getMep()),
                                actions.getRequestLocation(), actions.getResponseLocation(), actions.getAddressing(),
                                getMessageFlowPriority(service)) ;
                            endpointServices.add(webserviceInfo) ;
                        }
                    }
                }
            }
        }
        return endpointServices ;
    }
    
    public List<ServiceContract> getServiceContracts()
    {
        final List<ServiceContract> serviceContractList = new ArrayList<ServiceContract>() ;
        final Services services = jbossesb.getServices() ;
        if (services != null)
        {
            for(final Service service : services.getServiceList())
            {
                final Actions actions = service.getActions() ;
                if (actions != null)
                {
                	for (Action action : actions.getActionList())
                	{
                        Class<?> actionClass;
                        try {
                            actionClass = (Class<?>)ClassUtil.forName(action.getClass1(), ServiceContract.class);
                        } catch (ClassNotFoundException cnfe) {
                        	throw new RuntimeException(cnfe);
                        } catch (NoClassDefFoundError ncdfe) {
                        	throw new RuntimeException(ncdfe);
                        }
                        Publish publishAnnotation = (Publish)actionClass.getAnnotation(Publish.class);
                        if (publishAnnotation != null) {
                        	Class<?> publisherClass = publishAnnotation.value();
                        	if (ContractProvider.class.isAssignableFrom(publisherClass)) {
                        		ContractProvider provider;
                        		try {
	                        		provider = (ContractProvider)publisherClass.newInstance();
	                        		Properties properties = toProperties(action.getPropertyList());
	                        		provider.setContractProperties(properties);
                        		} catch (Exception e) {
                        			throw new RuntimeException("Failed to create ContractProvider", e);
                        		}
	                        	ServiceContract serviceContract = new ServiceContract(
	                        		new org.jboss.soa.esb.Service(service.getCategory(), service.getName()),
	                        		service.getDescription(),
	                        		provider
	                        	);
	                        	serviceContractList.add(serviceContract);
	                        	break; // first Action with a ContractProvider wins!
                        	}
                        }
                	}
                }
            }
        }
        return serviceContractList;
    }
    
    public String getAuthDomain()
    {
        WarSecurity warSecurity = getWarSecurity();
        if (warSecurity != null)
        {
	        return warSecurity.getDomain();
        }
        return null;
    }
    
    public String getAuthMethod()
    {
        WarSecurity warSecurity = getWarSecurity();
        if (warSecurity != null)
        {
            return warSecurity.getMethod().toString();
        }
        return null;
    }

    private WarSecurity getWarSecurity()
    {
        Globals globals = jbossesb.getGlobals();
        if (globals != null)
        {
            return globals.getWarSecurity();
        }
        return null;
    }
    
    private Integer getMessageFlowPriority(final Service service)
        throws ConfigurationException
    {
        final List<Property> propertyList = service.getPropertyList() ;
        for(Property property: propertyList)
        {
            if (ListenerTagNames.MESSAGE_FLOW_PRIORITY.equals(property.getName()))
            {
                return MessageFlowContext.parseMessageFlowPriority(property.getValue()) ;
            }
        }
        return null ;
    }
}