/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 */

package org.jboss.soa.esb.listeners.config.mappers130;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.eprs.JDBCEpr;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.config.xbeanmodel130.SqlBusDocument.SqlBus;
import org.jboss.soa.esb.listeners.config.xbeanmodel130.SqlListenerDocument.SqlListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel130.SqlMessageFilterDocument.SqlMessageFilter;
import org.jboss.soa.esb.listeners.config.xbeanmodel130.SqlProviderDocument.SqlProvider;
import org.jboss.soa.esb.listeners.gateway.SqlTableGatewayListener;
import org.w3c.dom.Element;

/**
 * Performs the mapping of a &lt;fs-listener&gt; XSD based configuration to the "ConfigTree"
 * style configuration, adding the "ConfigTree" listener config to the "root" node.
 *
 * @author <a href="mailto:kurt.stam@jboss.com">kurt.stam@jboss.com</a>
 */
public class SqlListenerMapper {

	/**
	 * Perform the mapping.
	 * @param root The "ConfigTree" configuration root node.
	 * @param listener The Jmslistener to be mapped into the ConfigTree.
	 * @param model The configuration model from which the mapping is being performed.
	 * @return The ConfigTree listener configuration node.
	 * @throws org.jboss.soa.esb.ConfigurationException Invalid listener configuration.
	 */
	public static Element map(Element root, SqlListener listener, XMLBeansModel model) throws ConfigurationException {
		Element listenerNode = YADOMUtil.addElement(root, "listener");
		SqlBus bus;
		SqlProvider provider;

        listenerNode.setAttribute("name", listener.getName());

		try {
			bus = (SqlBus) model.getBus(listener.getBusidref());
		} catch (ClassCastException e) {
			throw new ConfigurationException("Invalid busid reference [" + listener.getBusidref() + "] on listener [" + listener.getName() + "].  A <sql-listener> must reference a <sql-bus>.");
		}
		try {
			provider = (SqlProvider) model.getProvider(bus);
		} catch (ClassCastException e) {
			throw new ConfigurationException("Invalid bus config [" + listener.getBusidref() + "].  Should be contained within a <sql-provider> instance.  Unexpected exception - this should have caused a validation error!");
		}

		if (provider.getDatasource() != null) {
			if ((provider.getUsername() != null)
					|| (provider.getDriver() != null)) {
				throw new ConfigurationException ("Invalid sql-provider configuration : a datasource and a username/password/driver "
						+ "combination cannot both be defined.   Use only one (datasource or JDBC connection info)."
						+ "Datasource : [" + provider.getDatasource() + "] JDBC URL [" + provider.getUrl() + "]");
			}
		} else if (provider.getUrl() == null) {
			throw new ConfigurationException ("Invalid sql-provider configuration : a datasource or a URL/username/password/driver "
					+ "combination must be defined.   Use only one (datasource or JDBC connection info).  "
					+ "URL was null.");
		} else if (provider.getUsername() == null) {
			throw new ConfigurationException ("Invalid sql-provider configuration :  a datasource or a URL/username/password/driver "
					+ "combination must be defined.   Use only one (datasource or JDBC connection info).   "
					+ "Username was null.");
		} else if (provider.getDriver() == null) {
			throw new ConfigurationException ("Invalid sql-provider configuration : either a datasource or a URL/username/password/driver "
					+ "combination must be defined.   Use only one (datasource or JDBC connection info).   "
					+ "Driver was null.");
		}

		SqlMessageFilter messageFilter = listener.getSqlMessageFilter();
		if(messageFilter == null) {
			messageFilter = bus.getSqlMessageFilter();
			if(messageFilter == null) {
				throw new ConfigurationException("No <sql-destination> defined on either <sql-listener> [" + listener.getName() + "] or <sql-bus> [" + bus.getBusid() + "].");
			}
		}
        listenerNode.setAttribute("pollLatencySeconds", String.valueOf(listener.getPollFrequencySeconds()));
        if (provider.isSetTransacted()) {
            listenerNode.setAttribute(ListenerTagNames.TRANSACTED_TAG, Boolean.toString(provider.getTransacted())) ;
        }
		// Map the standard listener attributes - common across all listener types...
		MapperUtil.mapDefaultAttributes(listener, listenerNode, model);
		// Map the <property> elements targeted at the listener - from the listener itself.
		MapperUtil.mapProperties(listener.getPropertyList(), listenerNode);
		if(listener.getIsGateway()) {
			listenerNode.setAttribute("gatewayClass", SqlTableGatewayListener.class.getName());
			listenerNode.setAttribute(ListenerTagNames.IS_GATEWAY_TAG, Boolean.toString(listener.getIsGateway()));

			// Map EPR related attributes onto the listener - from the bus and provider and listener.
			// Note: This will change - the Gateways will also support the EPR element...
			mapSqlEprProperties(listenerNode, provider, messageFilter);
			MapperUtil.mapEPRProperties(listener, listenerNode, model);
		} else {
			Element eprNode = YADOMUtil.addElement(listenerNode, ListenerTagNames.EPR_TAG);

			// Map EPR related attributes onto the EPR - from the bus and provider and listener...
			mapSqlEprProperties(eprNode, provider, messageFilter);
			MapperUtil.mapEPRProperties(listener, eprNode, model);
			// Remove any empty attributes set on the EPR config...
			YADOMUtil.removeEmptyAttributes(eprNode);
		}

		// Remove any empty attributes set on the listener config...
		YADOMUtil.removeEmptyAttributes(listenerNode);

		return listenerNode;
	}

	private static void mapSqlEprProperties(Element toElement, SqlProvider provider, SqlMessageFilter messageFilter) {
		toElement.setAttribute(JDBCEpr.DATASOURCE_TAG, provider.getDatasource());
		toElement.setAttribute(JDBCEpr.URL_TAG, provider.getUrl());
		toElement.setAttribute(JDBCEpr.DRIVER_TAG, provider.getDriver());
		toElement.setAttribute(JDBCEpr.USERNAME_TAG, provider.getUsername());
		toElement.setAttribute(JDBCEpr.PASSWORD_TAG, provider.getPassword());
		toElement.setAttribute(JDBCEpr.TABLE_NAME_TAG, messageFilter.getTablename());
		toElement.setAttribute(JDBCEpr.WHERE_CONDITION_TAG, messageFilter.getWhereCondition());
		toElement.setAttribute(JDBCEpr.ORDER_BY_TAG, messageFilter.getOrderBy());
		toElement.setAttribute(JDBCEpr.MESSAGE_ID_COLUMN_TAG, messageFilter.getMessageIdColumn());
        toElement.setAttribute(JDBCEpr.RETRY_COUNT_COLUMN_TAG, messageFilter.getRetryCountColumn());
		toElement.setAttribute(JDBCEpr.DATA_COLUMN_TAG, messageFilter.getMessageColumn());
		toElement.setAttribute(JDBCEpr.STATUS_COLUMN_TAG, String.valueOf(messageFilter.getStatusColumn()));
		toElement.setAttribute(JDBCEpr.TIMESTAMP_COLUMN_TAG, messageFilter.getInsertTimestampColumn());
		toElement.setAttribute(JDBCEpr.POST_DEL_TAG, String.valueOf(messageFilter.getPostDelete()));
		toElement.setAttribute(JDBCEpr.ERROR_DEL_TAG, String.valueOf(messageFilter.getErrorDelete()));


	}
}
