/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.gateway;

import java.io.IOException;
import java.io.ObjectInputStream;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.couriers.FaultMessageException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.gateway.http.HttpRequestWrapper;
import org.jboss.soa.esb.listeners.message.UncomposedMessageDeliveryAdapter;
import org.jboss.soa.esb.message.Message;

/**
 * This class is for dispatching the http request to ESB action process pipe
 * line.By default, this servlet will be used to set to default servlet mapping
 * in the created StandardContext.
 * This class will also return the esb error message and error stack trace to client 
 * 
 * @author <a href="mailto:ema@redhat.com">Jim Ma</a>
 * @deprecated
 */
public class HttpDispatchServlet extends HttpServlet {
	private static final long serialVersionUID = 1L;

    private static final String METHOD_DELETE = "DELETE";
    private static final String METHOD_HEAD = "HEAD";
    private static final String METHOD_GET = "GET";
    private static final String METHOD_OPTIONS = "OPTIONS";
    private static final String METHOD_POST = "POST";
    private static final String METHOD_PUT = "PUT";
    private static final String METHOD_TRACE = "TRACE";
	
	/** If this a synchronous invocation * */
	private boolean synchronous = true;
	
	/** asyncResponse file location */
    private String asyncResponse = null;

    private int serviceTimeout = 20000;
	
	/** Default allow http method list */
    private String allowHttpMethods = "POST,GET";

    private ConfigTree config ;

	/** ESB message delivery adapter */
	public transient UncomposedMessageDeliveryAdapter messageDeliveryAdapter ;

	/**
	 * Initilize method
	 * 
	 * @throws ServletException
	 *             If error occured
	 */
	public void init() throws ServletException {
		config = (ConfigTree) getServletContext().getAttribute(
				"config");
		synchronous = !config.getAttribute("synchronous", "true").equalsIgnoreCase("false");

		serviceTimeout = Integer.parseInt(config.getAttribute("serviceTimeout", "20000"));

		asyncResponse = config.getAttribute("asyncResponse");
				
		
		if (config.getAttribute(HttpGatewayListener.ALLOW_HTTP_METHOD) != null) {
			allowHttpMethods = config.getAttribute(HttpGatewayListener.ALLOW_HTTP_METHOD).toUpperCase();
		}
		try {
			initTransient() ;
		} catch (ConfigurationException e) {
			throw new ServletException(e);
		}

	}

	/**
	 * Service method to process the inboud servlet request and reply the http
	 * servlet response
	 * 
	 * @param request
	 *            HttpServlet request
	 * @param response
	 *            HttpServlet response
	 * @param content
	 * @throws ServletException
	 *             For error occured during process
	 */
	protected void serveRequest(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        
		String servletPath = request.getServletPath();
		
		//block the invalid request 
		if (!"/".equals(servletPath)) {
			response.setStatus(404);
			return;
		}
		
		org.jboss.soa.esb.listeners.gateway.http.HttpRequestWrapper wrapper = new HttpRequestWrapper(request, response, null);
       
		try {
			if (synchronous) {
				messageDeliveryAdapter.deliverSync(wrapper, serviceTimeout);
			} else {
				messageDeliveryAdapter.deliverAsync(wrapper);
				if (asyncResponse != null) {			       
					response.getOutputStream().write(StreamUtils.readStream(getClass().getResourceAsStream(asyncResponse)));			        
				} 
			}
		} catch (FaultMessageException fme) {
			Message faultMessage = fme.getReturnedMessage();
			if (faultMessage != null) {
				Object obj = faultMessage.getProperties().getProperty(HttpMessageComposer.HTTP_RESPONSE_STATUS);
			    if (obj != null && obj instanceof Integer) {
			    	response.setStatus((Integer)obj);
			    } else {
			    	response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
			    }
			    
			    fme.printStackTrace(response.getWriter());			    
			}
			
		} catch (Exception ex) {
			response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
			ex.printStackTrace(response.getWriter());
		} 

	}
	
	@Override
	protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		if (isSupportMethod(METHOD_GET)) {
			serveRequest(request, response);
		} else {
			super.doGet(request, response);
		}
	}
	
	@Override
	protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		if (isSupportMethod(METHOD_POST)) {
			serveRequest(request, response);
		} else {
			super.doPost(request, response);
		}
	}
	
	@Override
	protected void doPut(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		if (isSupportMethod(METHOD_PUT)) {
			serveRequest(request, response);
		} else {
			super.doPut(request, response);
		}
	}
	@Override
	protected void doDelete(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		if (isSupportMethod(METHOD_DELETE)) {
			serveRequest(request, response);
		} else {
			super.doDelete(request, response);
		}
	}
	
	 protected void doOptions(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {

		boolean ALLOW_GET = isSupportMethod(METHOD_GET);
		boolean ALLOW_HEAD = ALLOW_GET;
		boolean ALLOW_POST = isSupportMethod(METHOD_POST);
		boolean ALLOW_PUT = isSupportMethod(METHOD_PUT);
		boolean ALLOW_DELETE = isSupportMethod(METHOD_DELETE);
		boolean ALLOW_TRACE = true;
		boolean ALLOW_OPTIONS = true;

		String allow = null;
		if (ALLOW_GET)
			if (allow == null) allow = METHOD_GET;
		if (ALLOW_HEAD)
			if (allow == null) allow = METHOD_HEAD;
			else
				allow += ", " + METHOD_HEAD;
		if (ALLOW_POST)
			if (allow == null) allow = METHOD_POST;
			else
				allow += ", " + METHOD_POST;
		if (ALLOW_PUT)
			if (allow == null) allow = METHOD_PUT;
			else
				allow += ", " + METHOD_PUT;
		if (ALLOW_DELETE)
			if (allow == null) allow = METHOD_DELETE;
			else
				allow += ", " + METHOD_DELETE;
		if (ALLOW_TRACE)
			if (allow == null) allow = METHOD_TRACE;
			else
				allow += ", " + METHOD_TRACE;
		if (ALLOW_OPTIONS)
			if (allow == null) allow = METHOD_OPTIONS;
			else
				allow += ", " + METHOD_OPTIONS;

		resp.setHeader("Allow", allow);
	}
	 
	 
	/**
	 * Check if support this http method
	 * @param http method name
	 * @return true if supported or false if not supported
	 */
	private boolean isSupportMethod(String name) {
		return allowHttpMethods.indexOf(name) >= 0;
	}		
	
	private void initTransient() throws ConfigurationException {
		messageDeliveryAdapter = UncomposedMessageDeliveryAdapter.getGatewayDeliveryAdapter(config,
			new HttpMessageComposer<HttpRequestWrapper>());
	}
	
	private void readObject(final ObjectInputStream ois) throws IOException, ClassNotFoundException {
		ois.defaultReadObject() ;
		try {
			initTransient();
		} catch (final ConfigurationException ce) {
			final IOException ioe = new IOException("Unexpected error deserialising servlet") ;
			ioe.initCause(ce) ;
			throw ioe ;
		}
	}
}
