/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2010
 */
package org.jboss.soa.esb.listeners.gateway.camel;

import static org.jboss.soa.esb.message.MessagePayloadProxy.NullPayloadHandling.NONE;

import java.io.Serializable;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashSet;
import java.util.Map.Entry;
import java.util.Set;

import org.apache.camel.Exchange;
import org.apache.camel.impl.DefaultMessage;
import org.apache.camel.impl.MessageSupport;
import org.apache.log4j.Logger;
import org.jboss.soa.esb.addressing.Call;
import org.jboss.soa.esb.addressing.eprs.InVMEpr;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.AbstractMessageComposer;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.Properties;

/**
 * CamelMessageComposer.
 * 
 * @author dward at jboss.org
 */
public class CamelMessageComposer<T extends org.apache.camel.Message> extends AbstractMessageComposer<T> {
	
	private static final Logger logger = Logger.getLogger(CamelMessageComposer.class);

	private ConfigTree config = null;
	private MessagePayloadProxy payloadProxy = null;
	private Set<String> mapHeaders;
	
	@Override
	public ConfigTree getConfiguration() {
		return config;
	}
	
	@Override
	public void setConfiguration(ConfigTree config) {
		this.config = config;
		payloadProxy = new MessagePayloadProxy(config);
		payloadProxy.setNullGetPayloadHandling(NONE);
		payloadProxy.setNullSetPayloadHandling(NONE);
		
		String mapHeaderConfig = config.getAttribute("mapHeaders");
		if(mapHeaderConfig != null) {
			String[] mapHeadersArray = mapHeaderConfig.split(",");
			
			mapHeaders = new HashSet<String>();
			for(int i = 0; i < mapHeadersArray.length; i++) {
				mapHeaders.add(mapHeadersArray[i].trim());
			}
		}
	}
	
	@Override
	protected MessagePayloadProxy getPayloadProxy() {
		return payloadProxy;
	}

	@Override
	protected void populateMessage(Message esbMessageIn, T camelMessageIn) throws MessageDeliverException {
		// maintain message id (both JBossESB and Camel use UUID.randomUUID(), so we can reuse)
		String camelMessageId = camelMessageIn.getMessageId();
		if (camelMessageId != null) {
			try {
				esbMessageIn.getHeader().getCall().setMessageID(new URI(camelMessageId));
			} catch (URISyntaxException e) {
				throw new MessageDeliverException("problem creating messageID", e);
			}
		}
		
		// maintain correlation
		setRelatesTo(camelMessageIn.getExchange(), esbMessageIn);
		
		// update esb properties from camel headers
		mapHeaders(esbMessageIn, camelMessageIn);
		
		// set esb body (payload) from camel body
		getPayloadProxy().setPayload(esbMessageIn, camelMessageIn.getBody(String.class));
	}

	/**
	 * Map the Camel Headers to the ESB Message Properties.
	 * <p/>
	 * Override to modify mapping behavior.
	 * @param esbMessageIn ESB message.
	 * @param camelMessageIn Camel message.
	 */
	protected void mapHeaders(Message esbMessageIn, T camelMessageIn) {
		Properties properties = esbMessageIn.getProperties();
		for (Entry<String, Object> entry : camelMessageIn.getHeaders().entrySet()) {
			String name = entry.getKey();
			
			if(mapHeaders != null && !mapHeaders.contains(name)) {
				// skip header mapping...
				continue;
			}
			
			Object value = entry.getValue();
			if (value instanceof Serializable) {
				properties.setProperty(name, value);
			} else {
				if(value != null && logger.isDebugEnabled()) {
					logger.debug("Camel header '" + name + "' is not Serializable (type " + value.getClass().getName() + ").  Cannot map to ESB Properties.  Ignoring.");
				}
			}
		}
	}
	
	@SuppressWarnings("unchecked")
	@Override
    public T decompose(Message esbMessageOut, T camelMessageIn) throws MessageDeliverException {
		org.apache.camel.Message camelMessageOut;
		if (camelMessageIn instanceof MessageSupport) {
			camelMessageOut = ((MessageSupport)camelMessageIn).newInstance();
		} else {
			camelMessageOut = new DefaultMessage();
		}
		
		// maintain message id (both JBossESB and Camel use UUID.randomUUID(), so we can reuse)
		URI esbMessageID = esbMessageOut.getHeader().getCall().getMessageID();
		if (esbMessageID != null) {
			camelMessageOut.setMessageId(esbMessageID.toString());
		}
		
		// maintain correlation
		if (camelMessageIn != null) {
			setRelatesTo(camelMessageIn.getExchange(), esbMessageOut);
		}
		
		// set camel headers from esb properties
		Properties properties = esbMessageOut.getProperties();
		for (String name : properties.getNames()) {
			Object value = properties.getProperty(name);
			if (value != null) {
				camelMessageOut.setHeader(name, value);
			}
		}
		
		// set camel body from esb body (payload)
		camelMessageOut.setBody(getPayloadProxy().getPayload(esbMessageOut));
		
		return (T)camelMessageOut;
    }
	
	private void setRelatesTo(Exchange exchange, Message esbMessage) throws MessageDeliverException {
		String exchangeId = (exchange != null ? exchange.getExchangeId() : null);
		if (exchangeId != null) {
			Call call = esbMessage.getHeader().getCall();
			URI relatesTo = call.getRelatesTo();
			if (relatesTo == null) {
				try {
					relatesTo = new URI(InVMEpr.INVM_PROTOCOL, "correlationID", exchangeId);
				} catch (URISyntaxException e) {
					throw new MessageDeliverException("problem creating relatesTo", e);
				}
				call.setRelatesTo(relatesTo);
			}
		}
	}

}
