package org.jboss.soa.esb.listeners.gateway.remotestrategies.cache;

import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.log4j.Logger;
import org.jboss.cache.CacheFactory;
import org.jboss.cache.DefaultCacheFactory;
import org.jboss.cache.CacheException;
import org.jboss.cache.Cache;
import org.jboss.cache.Fqn;
import org.jboss.cache.Node;
import org.jboss.cache.TreeCacheViewMBean;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * Implements a cache of file names which can be replicated in a cluster. 
 * 
 * See the <a href="http://labs.jboss.com/jbosscache">JBoss Cache</a> docs
 * for details on configuring the cache.
 * 
 * @author <a href="daniel.bevenius@redpill.se">Daniel Bevenius</a>				
 *
 */
public class FtpFileCache
{
	@SuppressWarnings("unused")
	private Logger log = Logger.getLogger( FtpFileCache.class );
	
	/**
	 * Default configuration file for jboss cache. 
	 * Used if no config is specified upon calling the constructor.
	 * Note that this file must then exist on the classpath
	 */
	private static final String DEFAULT_CONFIG_FILE = "/ftpfile-cache-config.xml";
	/**
	 * The fqn to use.
	 */
	private String fqn = "/ftp/cache/";
	/**
	 * The configuration file used to configure the tree cache
	 */
	private String config;
	/**
	 * The JBoss Cache instance
	 */
	private Cache cache;
	private Node rootNode;
	
	/**
	 * 
	 * @param config	either an absolute path of a relative path
	 * @throws Exception
	 */
	public FtpFileCache ( String config ) throws FtpFileCacheException
	{
		if ( config == null )
			config = DEFAULT_CONFIG_FILE;
		
		this.config = config;
		
		InputStream in = null;
		try
		{
		        in = getConfigInputStream( this.config );
		        CacheFactory factory = DefaultCacheFactory.getInstance();
			cache = factory.createCache(in);
			
			rootNode = cache.getRoot();
		}
		catch ( Exception e )
		{
			log.error( e );
			throw new FtpFileCacheException ( e );
		}
		finally
		{
			close( in );
		}
			
		
	}
	
	/**
	 * Gets an input stream from the specified path.
	 * 
	 * If the path denotes a file on the filesystem, that file
	 * will be used. 
	 * If a file is not found in the filesystem, this method will look 
	 * for the file on the classpath.
	 * 
	 * @param path			the path to the JBossCache configuration file
	 * @return InputStream	the inputstream for the passed-in path	
	 */
	protected InputStream getConfigInputStream( final String path ) 
	{
		InputStream in = null;
		File configFile = new File( path );
		if ( configFile.exists() )
		{
			log.debug("Reading jboss cache configuration from : " + path );
			
			try 
			{
				in =  new FileInputStream ( path );
			} 
			catch (FileNotFoundException e) 
			{
				log.error( e );
			}
		}
		else
		{
			log.debug("Reading jgroups configuration classpath : " + path );
			in = ClassUtil.getResourceAsStream( path, getClass() );
		}
		return in;
	}

	/**
	 * Start the cache
	 * @throws Exception
	 */
	public void start() throws FtpFileCacheException
	{
		try
		{
			cache.start();
		} 
		catch (Exception e)
		{
			log.error( e );
			throw new FtpFileCacheException( e );
		}
	}
	
	/**
	 * Stops the cache
	 *
	 */
	public void stop()
	{
		cache.stop();
	}

	/**
	 * Will add the filename to the cache.
	 * 
	 * @param fileName
	 * @throws CacheException 
	 */
	public void putFileName( final String fileName) throws CacheException
	{
	        Fqn tempFqn = new Fqn (fqn);
		cache.put(tempFqn, fileName, fileName );
	}

	/**
	 * Will get the filename if it exists in the cache.
	 * 
	 * @param fileName
	 * @throws CacheException
	 */
	public Object getFileName( final String fileName) throws CacheException
	{
	        Fqn tempFqn = new Fqn (fqn);
		return cache.get(tempFqn,fileName);
	}
	
	/**
	 * Removed the fileName from the cache if it exist there
	 * 
	 * @param fileName
	 * @return Object			the value of the removed object
	 * @throws CacheException
	 */
	public Object deleteFile( final String fileName ) throws CacheException
	{
	        Fqn tempFqn = new Fqn (fqn);
	    
		return cache.remove(tempFqn, fileName );
	}
	
	/**
	 * Checks to see if the filename exists in the cache
	 * 
	 * @param fileName	the filename to look for
	 * @return true	if the file exists in the cache
	 */
	public boolean containsFile( final String fileName )
	{
	        Fqn tempFqn = new Fqn (fqn);

	        String value = (String) cache.get(tempFqn, fileName);
		return (value != null);
	}
	
	/**
	 * Gets the a string of all files in the cache
	 * @return String
	 */
	public String getCache()
	{
	        StringBuffer buffer = new StringBuffer();
	        Set<String> fileNames = cache.getKeys(fqn);
	        for (Iterator it=fileNames.iterator(); it.hasNext();) {  
	            buffer.append((String) it.next());
	            buffer.append(" ");
	            // whatever  
	        }
		return buffer.toString();
	}

	public String getFqn()
	{
		return fqn;
	}

	public void setFqn(String fqn)
	{
		this.fqn = fqn;
	}

	public String getConfigFile( )
	{
		return config;
	}

	/**
	 * Removes all data under the fqn, but not the fqn 
	 * itself
	 * 
	 * @throws CacheException
	 */
	public void removeAll() throws CacheException
	{
	    Set<Node> nodes = cache.getRoot().getChildren();
	    Fqn tempFqn = new Fqn(fqn);
	    for (Iterator i = nodes.iterator(); i.hasNext();) {
	        Node n = (Node) i.next();
	        n.clearData();
	        cache.removeNode(n.getFqn());
	        cache.evict(n.getFqn(), true);
	    }
	}
	
	public void setDeleteOnEviction()
	{
		cache.addCacheListener( new DeleteOnEvictTreeCacheListener( cache) );
	}
	
	public void addCacheListener( Object listener )
	{
		if ( listener != null )
			cache.addCacheListener( listener );
	}
	
	public Map getCacheListeners()
	{      
	        Set listeners = cache.getCacheListeners();
	        Map resultMap = new Properties();
	        for (Iterator it=listeners.iterator(); it.hasNext();) {
	            resultMap.put(fqn, it.next());
	        }
		return resultMap;
	}
	
	public void setCacheListener( final Class cacheListenerClass ) throws FtpFileCacheException
	{
		if ( cacheListenerClass == null )
			return;
		try
		{
			Constructor constructor = cacheListenerClass.getConstructor(Cache.class );
			Object object = constructor.newInstance( cache );
			cache.addCacheListener(object);
		} 
		catch (SecurityException e)
		{
			throw createFtpFileCacheException( "SecurityException while trying set the CacheListener:",  e );
		} 
		catch (NoSuchMethodException e)
		{
			throw createFtpFileCacheException( "NoSuchMethodException while trying set the CacheListener:",  e );
		} 
		catch (IllegalArgumentException e)
		{
			throw createFtpFileCacheException( "IllegalArgumentException while trying set the CacheListener:",  e );
		} 
		catch (InstantiationException e)
		{
			throw createFtpFileCacheException( "InstantiationException while trying set the CacheListener:",  e );
		} 
		catch (IllegalAccessException e)
		{
			throw createFtpFileCacheException( "IllegalAccessException while trying set the CacheListener:",  e );
		} 
		catch (InvocationTargetException e)
		{
			throw createFtpFileCacheException( "InvocationTargetException while trying set the CacheListener:",  e );
		}
	}
	
	private FtpFileCacheException createFtpFileCacheException( final String msg,  Exception e )
	{
		log.error( msg, e );
		return new FtpFileCacheException( e );
	}
	
	private void close( Closeable c )
	{
		if ( c == null )
			return;
		
		try
		{
			c.close();
		}
		catch ( IOException e )
		{
			log.warn( "Error while trying to close Closable", e);
		}
	}

	public void destroy()
	{
		log.info( "destroy method of FtpFileCache called" );
		cache.destroy();
	}

}