/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security.auth.ws;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.security.cert.CertPath;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.codec.binary.Base64;

/**
 * Represents a WS-Security BinarySecurityToken.
 * <p/>
 *
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 * @author <a href="mailto:tcunning@redhat.com">Tom Cunningham</a>
 */
public class BinarySecurityTokenImpl implements BinarySecurityToken
{
	public static final String X509V3 = "X509v3";
	public static final String X509PKIPATHV1 = "X509PKIPathv1";
	public static final String PKCS7 = "PKCS7";
	
	private String encodingType;
	private String valueType;
	private List<Certificate> cert;
	private enum EncodingType { Base64Binary, HexBinary }

	public String getEncodingType()
	{
		return encodingType;
	}

	public void setEncodingType(String encodingType)
	{
		this.encodingType = stripNS(encodingType);
	}

	public String getValueType()
	{
		return valueType;
	}

	public void setValueType(String valueType)
	{
		this.valueType = stripNS(valueType);
	}

	public List<Certificate> getKeys()
	{
		return cert;
	}

	public void setKey(final String key)
	{
		cert = new ArrayList<Certificate>();
		try
		{
			byte[] keyBytes = null;
			if ( encodingType.equalsIgnoreCase( EncodingType.Base64Binary.toString() ) )
			{
			    Base64 decoder = new Base64();
    			keyBytes = decoder.decode(key.getBytes());
			}
			else
			{
				keyBytes = key.getBytes();
			}

		 	CertificateFactory factory = CertificateFactory.getInstance( certificateMatch( valueType ) );
		 	InputStream in = new ByteArrayInputStream(keyBytes);
		 			 	
			if (X509PKIPATHV1.equals(getValueType())) {
			 	CertPath path = null;
			 	try {
			 		path = factory.generateCertPath(in);
			 	} catch (CertificateException ce) {
					throw new IllegalStateException("Could not create certificate: ", ce);		 		
			 	}
			 	
				List certs = path.getCertificates();
				cert = certs;
			} else if (X509V3.equals(getValueType())) {
				Certificate certificate = factory.generateCertificate(in);
				cert.add(certificate);
			} else if (PKCS7.equals(getValueType())) {
				throw new IllegalStateException(getValueType() + " not implemented.");
			} else {
				throw new IllegalStateException(getValueType() + " not implemented.");				
			}
		}
		catch (CertificateException e)
		{
			throw new IllegalStateException("Could not create certificate: ", e);
		}
	}

	public String certificateMatch(final String valueType)
	{
		if ( valueType.startsWith("X509") )
			return "X.509";
		return valueType;
	}

	public String pathMatch(final String valueType)
	{
		if (valueType.startsWith(""))
			return "PkiPath";
		return "PKCS7";
	}
	
	
	private String stripNS(String value)
	{
		if ( value != null )
		{
			if ( value.startsWith("http"))
			{
				final int idx = value.indexOf('#');
    			if ( idx > 0 )
    				value = value.substring( idx + 1 );
			}
			else
			{
    			final int idx = value.indexOf(':');
    			if ( idx > 0 )
    				value = value.substring( idx + 1 );
			}
		}
		return value;
	}
}
