/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.internal.soa.esb.services.registry;

import java.net.URI;
import java.util.List;
import java.util.concurrent.CyclicBarrier;

import junit.framework.TestCase;

import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.services.registry.Registry;
import org.jboss.soa.esb.services.registry.ServiceNotFoundException;


/**
 * Unit tests for the local registry interceptor
 * 
 * @author <a href="mailto:Kevin.Conner@jboss.com">Kevin Conner</a>
 */
public class LocalRegistryInterceptorUnitTest extends TestCase
{
    /**
     * Tests of caching, intended to run within the current cache validity period
     */
    public void testLocalRegistration()
        throws Exception
    {
        final String category = "category" ;
        final String name = "name" ;
        final String missingCategory = "missingCategory" ;
        final String missingName = "missingName" ;
        final EPR epr1 = new EPR() ;
        final EPR epr2 = new EPR() ;
        final EPR epr3 = new EPR(URI.create("urn:test")) ;
        
        final LocalRegistryInterceptor interceptor = new LocalRegistryInterceptor() ;
        final RegistryStatsInterceptor stats = new RegistryStatsInterceptor() ;
        final MockRegistry registry = new MockRegistry() ;
        stats.setRegistry(registry) ;
        interceptor.setRegistry(stats) ;

        interceptor.registerEPR(category, name, "Description", epr1, "EPR description") ;
        
        // findAllServices passes through to registry
        assertEquals("findAllServices count", 0, stats.getFindAllServicesCount()) ;
        interceptor.findAllServices() ;
        assertEquals("findAllServices count", 1, stats.getFindAllServicesCount()) ;
        interceptor.findAllServices() ;
        assertEquals("findAllServices count", 2, stats.getFindAllServicesCount()) ;

        // findServices passes through to registry
        assertEquals("findServices count", 0, stats.getFindServicesCount()) ;
        interceptor.findServices(category) ;
        assertEquals("findServices count", 1, stats.getFindServicesCount()) ;
        interceptor.findServices(category) ;
        assertEquals("findServices count", 2, stats.getFindServicesCount()) ;

        // findEPR/findEPRs go to registry if not present in the local registry interceptor
        assertEquals("findEPRs count", 0, stats.getFindEPRsCount()) ;
        interceptor.findEPRs(category, name) ;
        assertEquals("findEPRs count", 0, stats.getFindEPRsCount()) ;
        interceptor.findEPRs(missingCategory, missingName) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        interceptor.findEPR(category, name) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertEquals("findEPR count", 0, stats.getFindEPRCount()) ;
        try
        {
            interceptor.findEPR(missingCategory, missingName) ;
            // The JAXR registry would return null here, the behaviour is not well defined
            fail("Expected ServiceNotFoundException") ;
        }
        catch (final ServiceNotFoundException snfe) {} // ignore
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertEquals("findEPR count", 1, stats.getFindEPRCount()) ;

        // registerEPR adds to local registry interceptor and then to registry
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertEquals("registerEPR count", 1, stats.getRegisterEPRCount()) ;
        final List<EPR> registerOrigEPRs = interceptor.findEPRs(category, name) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertNotNull("Original EPRs", registerOrigEPRs) ;
        assertEquals("Original EPR count", 1, registerOrigEPRs.size()) ;
        interceptor.registerEPR(category, name, "Description", epr2, "EPR description") ;
        assertEquals("registerEPR count", 2, stats.getRegisterEPRCount()) ;
        final List<EPR> duplicateEPRs = interceptor.findEPRs(category, name) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertNotNull("Duplicate EPRs", duplicateEPRs) ;
        assertEquals("Duplicate EPR count", 1, duplicateEPRs.size()) ;
        interceptor.registerEPR(category, name, "Description", epr3, "EPR description") ;
        assertEquals("registerEPR count", 3, stats.getRegisterEPRCount()) ;
        final List<EPR> registerNewEPRs = interceptor.findEPRs(category, name) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertNotNull("New EPRs", registerNewEPRs) ;
        assertEquals("New EPR count", 2, registerNewEPRs.size()) ;

        // unRegister removes from local registry interceptor and then registry
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertEquals("unRegisterEPR count", 0, stats.getUnRegisterEPRCount()) ;
        final List<EPR> unRegisterOrigEPRs = interceptor.findEPRs(category, name) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertNotNull("Original EPRs", unRegisterOrigEPRs) ;
        assertEquals("Original EPR count", 2, unRegisterOrigEPRs.size()) ;
        interceptor.unRegisterEPR(category, name, epr3) ;
        assertEquals("unRegisterEPR count", 1, stats.getUnRegisterEPRCount()) ;
        final List<EPR> unRegisterNewEPRs = interceptor.findEPRs(category, name) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertNotNull("New EPRs", unRegisterNewEPRs) ;
        assertEquals("New EPR count", 1, unRegisterNewEPRs.size()) ;
        interceptor.unRegisterEPR(category, name, epr2) ;
        assertEquals("unRegisterEPR count", 2, stats.getUnRegisterEPRCount()) ;
        final List<EPR> unRegisterFinalEPRs = interceptor.findEPRs(category, name) ;
        assertEquals("findEPRs count", 1, stats.getFindEPRsCount()) ;
        assertNotNull("Duplicate EPRs", unRegisterFinalEPRs) ;
        assertEquals("Duplicate EPR count", 1, unRegisterFinalEPRs.size()) ;
        interceptor.unRegisterEPR(category, name, epr1) ;
        assertEquals("unRegisterEPR count", 3, stats.getUnRegisterEPRCount()) ;
        final List<EPR> finalEPRs = interceptor.findEPRs(category, name) ;
        assertEquals("findEPRs count", 2, stats.getFindEPRsCount()) ;
        assertNotNull("Final EPRs", finalEPRs) ;
        assertEquals("Final EPR count", 0, finalEPRs.size()) ;
        EPR finalEPR = null ;
        try
        {
            finalEPR = interceptor.findEPR(category, name) ;
            // The JAXR registry would return null here, the behaviour is not well defined
            fail("Expected ServiceNotFoundException") ;
        }
        catch (final ServiceNotFoundException snfe) {} // ignore
        assertNull("Final EPR", finalEPR) ;
        assertEquals("findEPR count", 2, stats.getFindEPRCount()) ;
        assertEquals("findEPRs count", 2, stats.getFindEPRsCount()) ;

        // Test unregistering of service
        registry.registerEPR(category, name, "Description", epr1, "EPR description") ;
        assertEquals("unRegisterService count", 0, stats.getUnRegisterServiceCount()) ;
        interceptor.unRegisterService(category, name) ;
        assertEquals("unRegisterService count", 1, stats.getUnRegisterServiceCount()) ;
        try
        {
            interceptor.findEPR(category, name) ;
            fail("Expected ServiceNotFoundException") ;
        }
        catch (final ServiceNotFoundException snfe) {} // ignore
    }

    /**
     * Test concurrent access to next interceptor in chain.
     * @throws Exception
     */
    public void testConcurrency()
        throws Exception
    {
        final int numServices = 5 ;
        final String category = "category" ;
        final String name = "name" ;
        
        final LocalRegistryInterceptor interceptor = new LocalRegistryInterceptor() ;
        final RegistryConcurrencyInterceptor concurrency = new RegistryConcurrencyInterceptor(new CyclicBarrier(numServices)) ;
        final MockRegistry registry = new MockRegistry() ;
        concurrency.setRegistry(registry) ;
        interceptor.setRegistry(concurrency) ;
        
        final String[] names = new String[numServices] ;
        for (int count = 0 ; count < numServices ; count++)
        {
            names[count] = name + count ;
            interceptor.registerEPR(category, names[count], "description", new EPR(), "EPR description") ;
        }
        final int numThreads = numServices*2 ;
        
        final CyclicBarrier barrier = new CyclicBarrier(numThreads) ;
        final ConcurrencyTest[] tests = new ConcurrencyTest[numThreads] ;
        for(int count = 0 ; count < numThreads ; count++)
        {
            tests[count] = new ConcurrencyTest(barrier, interceptor, category, names[count%numServices]) ;
            tests[count].start() ;
        }
        
        for(int count = 0 ; count < numThreads ; count++)
        {
            tests[count].join();
        }
        
        for(int count = 0 ; count < numThreads ; count++)
        {
            assertNull("Throwable occurred", tests[count].getThrowable()) ;
        }
        
        assertEquals("Registry findEPRs invocation", 0, concurrency.getFindEPRsCount()) ;
        assertFalse("Barrier timeout", concurrency.isTimeout()) ;
    }

    private static final class ConcurrencyTest extends Thread
    {
        private final CyclicBarrier barrier ;
        private final Registry registry ;
        private final String category ;
        private final String name ;
        
        private Throwable throwable ;

        ConcurrencyTest(final CyclicBarrier barrier, final Registry registry, 
            final String category, final String name)
        {
            this.barrier = barrier ;
            this.registry = registry ;
            this.category = category ;
            this.name = name ;
        }

        public void run()
        {
            try
            {
                barrier.await() ;
                registry.findEPRs(category, name) ;
            }
            catch (final Throwable th)
            {
                throwable = th ;
            }
        }

        Throwable getThrowable()
        {
            return throwable ;
        }
    }
}
