/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.config.mappers131;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.List;

import junit.framework.JUnit4TestAdapter;

import org.apache.xmlbeans.XmlException;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.config.xbeanmodel131.JmsBusDocument;
import org.jboss.soa.esb.listeners.config.xbeanmodel131.JmsListenerDocument;
import org.jboss.soa.esb.listeners.config.xbeanmodel131.Listener;
import org.jboss.soa.esb.listeners.config.xbeanmodel131.Provider;
import org.jboss.soa.esb.listeners.config.xbeanmodel131.ActivationConfigDocument.ActivationConfig;
import org.jboss.soa.esb.listeners.config.xbeanmodel131.JbossesbDocument.Factory;
import org.jboss.soa.esb.listeners.config.xbeanmodel131.JmsJcaProviderDocument.JmsJcaProvider;
import org.jboss.soa.esb.listeners.config.xbeanmodel131.JmsListenerDocument.JmsListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel131.PropertyDocument.Property;
import org.junit.Before;
import org.junit.Test;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * Unit test for JmsListenerMapper
 * 
 * @author Daniel Bevenius
 *
 */
public class JmsListenerMapperUnitTest
{
	private static final String ESB_CONFIG = "jbossesb_config_jms_jca.xml";
	
	private Element root;
	private XMLBeansModel model;
	private JmsListenerDocument.JmsListener jmsListener;
	private JmsJcaProvider jmsJcaProvider;
	
	@Test ( expected = ConfigurationException.class )
	public void tryToOverrideDestinationActivationConfigProperty() throws ConfigurationException, IOException
	{
		callJmsListenerMap( "destination", "/queue/dummyQueue" );
	}
	
	@Test ( expected = ConfigurationException.class )
	public void tryToOverride_DestinationType_ActivationConfigProperty() throws ConfigurationException, IOException
	{
		callJmsListenerMap( "destinationType", "Topic" );
	}
	
	@Test ( expected = ConfigurationException.class )
	public void tryToOverrideMessageSelectorActivationConfigProperty() throws ConfigurationException, IOException
	{
		callJmsListenerMap( "messageSelector", "dummyselector" );
	}
	
	@Test ( expected = ConfigurationException.class )
	public void tryToOverrideMaxMessagesActivationConfigProperty() throws ConfigurationException, IOException
	{
		callJmsListenerMap( "maxMessages", "300" );
	}
	
	//	setup methods
	
	@Before
	public void setup() throws ConfigurationException, IOException, XmlException
	{
		root = createRootElement();
		model = getXmlBeanModel( ESB_CONFIG );
		jmsListener = getListener( model );
		jmsJcaProvider = getJmsJcaProvider( model );
	}
	
	// 	helper methods
	
	private void callJmsListenerMap(
			String activationConfigPropertyName,
			String activationConfigPropertyValue
			) throws ConfigurationException
	{
		ActivationConfig activationConfig = jmsJcaProvider.getActivationConfig();
		createActivationProperty( activationConfig, activationConfigPropertyName, activationConfigPropertyValue );
		
		JmsListenerMapper.map( root , jmsListener, model );
	}
			
	private Property createActivationProperty( 
			final ActivationConfig activationConfig, 
			final String name, 
			final String value )
	{
		Property property = activationConfig.addNewProperty();
		property.setName( name );
		property.setValue( value ); 
		return property;
	}
	
	private JmsListener getListener( final XMLBeansModel model ) throws ConfigurationException
	{
		List<Listener> listeners = model.getESBAwareListeners();
		return (JmsListener) listeners.get(0);
	}
	
	private JmsJcaProvider getJmsJcaProvider( final XMLBeansModel model ) throws ConfigurationException, IOException
	{
		List<Listener> listeners = model.getESBAwareListeners();
		JmsListenerDocument.JmsListener jmsListener = (JmsListener) listeners.get(0);
		assertEquals ( false, jmsListener.getIsGateway() );
		
		JmsBusDocument.JmsBus jmsBus = (JmsBusDocument.JmsBus) model.getBus( jmsListener.getBusidref() );
		assertNotNull( "JmsBus element should exist", jmsBus );
		Provider provider = model.getProvider( jmsBus );
		assertTrue( provider instanceof JmsJcaProvider );
		return (JmsJcaProvider) provider;
	}
	
	private XMLBeansModel getXmlBeanModel( String fileName ) throws ConfigurationException, IOException, XmlException
	{
		InputStream inputStream = getClass().getResourceAsStream( ESB_CONFIG );
		final Reader reader = new InputStreamReader(inputStream);
		XMLBeansModel model = new XMLBeansModel(Factory.parse(reader).getJbossesb());
		return model;
	}
	
	private Element createRootElement() throws ConfigurationException
	{
		Document doc = YADOMUtil.createDocument();
		Element root = YADOMUtil.addElement(doc, "jbossesb-gateways");
		root.setAttribute("parameterReloadSecs", "1000" );
		return root;
	}
	
	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter( JmsListenerMapperUnitTest.class );
	}
}
