/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.services.rules;

import org.jboss.internal.soa.esb.services.rules.RuleServiceException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;

/**
 * RulesEngine Interface. Separates implementation from the runtime, allowing
 * the ESB to support different implementations of rule engines.
 * 
 * @author jdelong@redhat.com
 * 
 */
public interface RuleService {
	
	public void setConfigTree(ConfigTree configTree);

	/**
	 * Execute rules using a certain ruleSet and domain specific language using the Stateless rule engine API
	 * 
	 * @param ruleInfo -
	 * @param message -
	 *            Message that is updated with the results.
	 * 
	 * @return Message with updated objects.
	 */
	public Message executeStatelessRules(final RuleInfo ruleInfo, final Message message) throws RuleServiceException;
	
	/**
	 * Execute rules from a decision table using the Stateless rule engine API
	 * 
	 * @param ruleInfo
	 * @param message -
	 *            Message that is updated with the results.
	 * 
	 * @return Message with updated objects.
	 */
	public Message executeStatelessRulesFromDecisionTable(final RuleInfo ruleInfo, final Message message) throws RuleServiceException;
	
	/**
	 * Execute rules using a rule package retrieved via the Rule Agent from a URL of local file system
	 *            using the Stateless rule engine API
	 * 
	 * @param ruleInfo
	 * @param message -
	 *            Message that is updated with the results.
	 * 
	 * @return Message with updated objects.
	 */
	public Message executeStatelessRulesFromRuleAgent(final RuleInfo ruleInfo, final Message message) throws RuleServiceException;

	/**
	 * Execute rules using a certain ruleSet and domain specific language using the Stateful rule engine API
	 * 
	 * @param statefulRuleInfo
	 * @param message -
	 *            Message that is updated with the results.ry 
	 * 
	 * @return Message with updated objects.
	 */
	public Message executeStatefulRules(final StatefulRuleInfo statefulRuleInfo, final Message msg) throws RuleServiceException;
	
	

	/**
	 * Execute rules from a decision table using the Stateful rule engine API
	 * 
	 * @param statefulRuleInfo
	 * @param message -
	 *            Message that is updated with the results.
	 * 
	 * @return Message with updated objects.
	 */
	public Message executeStatefulRulesFromDecisionTable(final StatefulRuleInfo statefulRuleInfo, final Message msg) throws RuleServiceException;

	/**
	 * Execute rules using a rule package retrieved via the Rule Agent from a URL of local file system
	 *            using the Stateful rule engine API
	 * 
	 * @param statefulRuleInfo
	 * @param message -
	 *            Message that is updated with the results.mory 
	 * 
	 * @return Message with updated objects.
	 */
	public Message executeStatefulRulesFromRuleAgent(final StatefulRuleInfo statefulRuleInfo, final Message msg) throws RuleServiceException;

	/**
	 * Continue executing rules where the working memory already exists.
	 * 
	 * @param statefulRuleInfo
	 * @param message -
	 *            Message that is updated with the results.
	 * 
	 * @return Message with updated objects.
	 */
	public Message continueStatefulRulesExecution(final StatefulRuleInfo statefulRuleInfo, final Message msg) throws RuleServiceException;
	
}
