/*
 * JBoss, Home of Professional Open Source
 * Copyright 2010, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */

package org.jboss.soa.esb.actions.soap.adapter.cxf;

import java.util.Map;

import javax.servlet.ServletContext;

import org.apache.cxf.Bus;
import org.apache.cxf.transport.DestinationFactory;
import org.apache.cxf.transport.DestinationFactoryManager;
import org.apache.cxf.transport.servlet.ServletTransportFactory;
import org.apache.log4j.Logger;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.wsf.spi.deployment.Endpoint;
import org.jboss.wsf.stack.cxf.ServletControllerExt;
import org.jboss.wsf.stack.cxf.configuration.BusHolder;

/**
 * This provides alternative implementations for obtaining the ServletControllerExt for the CXF integration.
 *  
 * @author <a href="mailto:kevin.conner@jboss.com">Kevin Conner</a>
 */
abstract class ServletControllerExtProviderFactory
{
    private static final ServletControllerExtProviderFactory FACTORY ;
    private static final Logger LOGGER = Logger.getLogger(ServletControllerExtProviderFactory.class) ;
    
    public static ServletControllerExtProvider getFactory(final Endpoint endpoint, final ServletContext context)
    {
        return FACTORY.getProvider(endpoint, context) ;
    }
    
    abstract ServletControllerExtProvider getProvider(final Endpoint endpoint, final ServletContext context) ;
    
    static
    {
        boolean busHolderExists = false ;
        try
        {
            ClassUtil.forName("org.jboss.wsf.stack.cxf.configuration.BusHolder", ServletControllerExtProviderFactory.class) ;
            busHolderExists = true ;
        }
        catch (final Throwable th) {} // ignore
        
        if (busHolderExists)
        {
            FACTORY = new BusHolderFactory() ;
        }
        else
        {
            FACTORY = new ServletControllerExtFactory() ;
        }
    }
    
    private static final class BusHolderFactory extends ServletControllerExtProviderFactory
    {
        ServletControllerExtProvider getProvider(final Endpoint endpoint, final ServletContext context)
        {
            return new BusHolderImpl(endpoint, context) ;
        }
    }
    
    private static final class BusHolderImpl implements ServletControllerExtProvider
    {
        private final ServletControllerExt servletControllerExt ;
        
        BusHolderImpl(final Endpoint endpoint, final ServletContext context)
        {
            final BusHolder busHolder = endpoint.getService().getDeployment().getAttachment(BusHolder.class) ;
            final Bus bus = busHolder.getBus() ;

            final ServletTransportFactory servletTransportFactory = new SOAPProcessorServletTransportFactory(bus, getTransport(bus)) ;
            servletControllerExt = new ServletControllerExt(servletTransportFactory, null, context, bus) ;
        }

        private static ServletTransportFactory getTransport(final Bus bus)
        {
            final DestinationFactoryManager dfm = bus.getExtension(DestinationFactoryManager.class) ;
            try
            {
                final DestinationFactory df = dfm.getDestinationFactory("http://cxf.apache.org/transports/http/configuration") ;
                if (df instanceof ServletTransportFactory)
                {
                    return (ServletTransportFactory)df ;
                }
            }
            catch (final Throwable th)
            {
                throw new IllegalStateException("Could not locate CXF transport", th) ;
            }
            throw new IllegalStateException("Could not locate CXF transport") ;
        }
        
        public ServletControllerExt getServletControllerExt()
        {
            return servletControllerExt;
        }
    }

    private static final class ServletControllerExtFactory extends ServletControllerExtProviderFactory
    {
        ServletControllerExtProvider getProvider(final Endpoint endpoint, final ServletContext context)
        {
            return new ServletControllerExtImpl(endpoint, context) ;
        }
    }
    
    private static final class ServletControllerExtImpl implements ServletControllerExtProvider
    {
        private final ServletControllerExt servletControllerExt ;
        
        ServletControllerExtImpl(final Endpoint endpoint, final ServletContext context)
        {
            final Map<Class<?>, Object> extensions = endpoint.getAttachment(Map.class) ;
            servletControllerExt = (extensions == null ? null : (ServletControllerExt) extensions.get(ServletControllerExt.class)) ;
            if (servletControllerExt == null)
            {
                LOGGER.error("ServletControllerExt not present in endpoint attachments.  Has the CXF integration bean configured?  See ESB documentation for details.") ;
                throw new IllegalStateException("Could not locate CXF ServletControllerExt attachment") ;
            }
        }

        public ServletControllerExt getServletControllerExt()
        {
            return servletControllerExt;
        }
    }
}
