/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.util.ClassUtil;

import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.context.support.AbstractApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

/**
 * Abstract class for Actions that makes calls to Spring beans.
 * <p/>
 * <p>
 * <pre>{@code
 * Usage :
 * <action name="sayHello" class="xyx.com.SimpleSpringAction" process="process">
 *     <property name="springContextXml" value="spring-context1.xml, spring-context2.xml"/>
 * </action>
 * }</pre><br>
 * 
 * Description of configuration properties:
 * <ul>
 * <li><i>springContextXml</i>: A single Spring bean definition xml file or a comma separated list of xml files.
 * </ul>
 * 
 * @author <a href="mailto:james.williams@redhat.com">James Williams</a>.
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 * 
 */
public abstract class AbstractSpringAction extends AbstractActionLifecycle implements ActionLifecycle
{
	private Logger logger = Logger.getLogger( AbstractSpringAction.class );
	
	/** property name used in config file */
	private static final String SPRING_CONTEXT_XML_ATTR = "springContextXml";
	
	private static final String SNOWDROP_VFS_APP_CONTEXT_IMPL = "org.jboss.spring.vfs.context.VFSClassPathXmlApplicationContext";

	/** Spring bean definition xml fil */
	private String springContextXml;

	/** configuration object */
	protected ConfigTree configTree;

	/** Spring Application Context */
	private AbstractApplicationContext appContext;
	
	/**
	 * Constructs and instance with storing the passed in ConfigTree instance.
	 * <p/>
	 * 
	 * @param configTree the ConfigTree instance that will be stored
	 */
	public AbstractSpringAction(final ConfigTree configTree)
	{
		this.configTree = configTree;
	}

	/**
	 * Please do not call this no-args constructor and use
	 * {@link #AbstractSpringAction(ConfigTree)} instead. This is because
	 * the field configTree will be using in the {@link #initialise()} method
	 * and will be null this config tree is not saved.
	 * 
	 * @deprecated Use {@link #AbstractSpringAction(ConfigTree)} instead
	 */
	public AbstractSpringAction()
	{
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.actions.AbstractActionLifecycle#initialise()
	 */
	public void initialise() throws ActionLifecycleException
	{
		if (configTree == null)
		{
			throw new ActionLifecycleException( "Please make sure that the AbstractSpringAction(ConfigTree configTree) constructor was called and not the no-args constructor." );
		}

		springContextXml = configTree.getAttribute( SPRING_CONTEXT_XML_ATTR );
		if (springContextXml == null || springContextXml.equals(""))
		{
			throw new ActionLifecycleException( "No Spring context specified on action config: " + SPRING_CONTEXT_XML_ATTR + "." );
		} 
		
		initializeSpring();
	}

	/**
	 * Check to see if Spring Bean factory is null. Mostly used for unit tests,
	 * but could provide use in other situations.
	 * 
	 * @return true if the bean factory is null
	 */
	public boolean isBeanFactoryNull()
	{
		return appContext == null;
	}
	
	/**
	 * Generic Exception handler for Spring Actions. Displays the root cause
	 * message and full stack trace.
	 * 
	 * @param message
	 * @param exception
	 */
	public void exceptionHandler( Message message, Throwable exception )
	{
		Throwable rootCause = exception.getCause();
		StackTraceElement[] traceElms = rootCause.getStackTrace();

		StringBuffer stackTrace = new StringBuffer( "Exception Root Cause is: \n" );
		stackTrace.append( rootCause.getMessage() );
		stackTrace.append( "\n Full Stack Trace is: \n" );
		for (StackTraceElement elm : traceElms)
		{
			stackTrace.append( elm );
			stackTrace.append( "\n" );
		}

		logger.error( stackTrace.toString() );
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.actions.AbstractActionLifecycle#destroy()
	 */
	public void destroy() throws ActionLifecycleException
	{
	    if (appContext != null)
	    {
	        appContext.close();
	        appContext = null;
	    }
	}

	/**
	 * Initialize Spring IoC
	 * 
	 * @throws ActionLifecycleException
	 */
	protected void initializeSpring() throws ActionLifecycleException
	{
		if (isBeanFactoryNull())
		{
			loadSpringIoc();
		}
	}

	/**
	 * Request a BeanFactory instance from the action. If no Spring IoC
	 * container exists, create one.
	 * 
	 * @return Spring Bean Factory
	 * @throws ActionLifecycleException
	 */
	protected BeanFactory getBeanFactory() throws ActionLifecycleException
	{
		return appContext;
	}
	
	/**
	 * Conviencence method for printing a console log header
	 * @deprecated
	 */
	protected void logHeader() { }

	/**
	 * Conviencence method for printing a console log footer
	 * @deprecated
	 */
	protected void logFooter() { }


	/**
	 * Create a spring IoC container.
	 * 
	 * @throws ActionLifecycleException
	 */
	private void loadSpringIoc() throws ActionLifecycleException
	{
		try
		{
			String[] contextXmlTokens = springContextXml.split(",");
			
			try {
				// Suppressing warnings on this unused classforname because appContext is given
				// a different value for AS4 and AS5
				@SuppressWarnings("unused")
				Class<?> snowdropVfsAppContextClass = ClassUtil.forName(SNOWDROP_VFS_APP_CONTEXT_IMPL, AbstractSpringAction.class);
				appContext = new VFSClassPathXmlApplicationContextFactory().getInstance(contextXmlTokens);
			} catch(ClassNotFoundException e) {
				// OK... not running on an AS5 variant... just use the core app context impl...
				appContext = new ClassPathXmlApplicationContext(contextXmlTokens);
			}
		} 
		catch (final BeansException e)
		{
			throw new ActionLifecycleException( "BeansException caught in loadSpringToc : " , e );
		}
	}
		
	private static class VFSClassPathXmlApplicationContextFactory {
		private AbstractApplicationContext getInstance(String[] contextXmlTokens) {
			return new org.jboss.spring.vfs.context.VFSClassPathXmlApplicationContext(contextXmlTokens);
		}
	}	
}
