/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2006, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.bpel.samples.quickstart.purchase;

import java.util.logging.Level;
import java.util.logging.Logger;

import javax.annotation.Resource;
import javax.ejb.MessageDriven;
import javax.ejb.MessageDrivenContext;
import javax.ejb.ActivationConfigProperty;
import javax.jms.JMSException;
import javax.jms.MapMessage;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.xml.ws.WebServiceRef;

/**
 * Asynchronous invoice callback bean.
 * 
 * @author Jeff DeLong
 * @author Alejandro Guizar
 */
@MessageDriven(activationConfig = {
		@ActivationConfigProperty(propertyName = "destinationType", propertyValue = "javax.jms.Queue"),
		@ActivationConfigProperty(propertyName = "destination", propertyValue = "queue/InvoiceQueue") })
public class InvoiceCallbackBean implements MessageListener {

	@Resource
	private MessageDrivenContext messageContext;
	@WebServiceRef
	private PurchaseSVC purchaseService;

	private final Logger logger =
			Logger.getLogger(InvoiceCallbackBean.class.getName());

	/**
	 * Process the invoice callback message.
	 */
	public void onMessage(Message message) {
		if (!(message instanceof MapMessage)) {
			logger.severe("received non-map message");
			messageContext.setRollbackOnly();
			return;
		}
		try {
			// extract contents
			MapMessage invoiceMessage = (MapMessage) message;
			int orderId = invoiceMessage.getInt("orderId");
			float amount = invoiceMessage.getFloat("amount");

			// populate invoice with contents
			Invoice inv = new Invoice();
			inv.setOrderId(orderId);
			inv.setAmount(amount);

			// send invoice back to requester
			purchaseService.getInvoiceCallbackSP().sendInvoice(inv);

			logger.log(
					Level.FINE,
					"Requested invoice for order #{0,number,integer} with amount {1,number,currency}",
					new Object[] { orderId, amount });
		}
		catch (JMSException e) {
			messageContext.setRollbackOnly();
			logger.log(Level.SEVERE, "could not read invoice message", e);
		}
	}
}
