package org.jboss.internal.soa.esb.command;

import java.util.Hashtable;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import org.jboss.soa.esb.helpers.ConfigTree;

/**
 * In Memory Blocking Command Queue. <p/> Suitable for testing or any other
 * purpose. <p/> The command queue's configuration needs to specify the queue
 * name via a "command-queue-name" attribute supplied in the configuration to
 * the {@link #open(ConfigTree)} method. The queues are stored statically and
 * can be accessed via the {@link #getQueue(String)} method using the queue
 * name.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 * @since Version 4.0
 */
public class InMemoryCommandQueue implements CommandQueue
{

	/**
	 * Command queue name attribute name.
	 */
	public static final String COMMAND_QUEUE_NAME = "command-queue-name";

	private static Hashtable<String, InMemoryCommandQueue> commandQueues = new Hashtable<String, InMemoryCommandQueue>();

	private String name;

	private BlockingQueue<String> queue = new LinkedBlockingQueue<String>();

	public void open(ConfigTree config) throws CommandQueueException
	{
		if (config == null)
		{
			throw new IllegalArgumentException(
					"null 'config' arg in method call.");
		}

		name = config.getAttribute(COMMAND_QUEUE_NAME);
		if (name == null)
		{
			throw new CommandQueueException(
					"Attribute 'command-queue-name' must be specified on the command queue configuration.");
		}
		commandQueues.put(name, this);
	}

	/**
	 * Add a command to the in-memory command queue. <p/> Blocks until the
	 * command has been consumed.
	 * 
	 * @param command
	 *            The command string.
	 */
	public void addCommand(String command)
	{
		queue.add(command);
		while (!queue.isEmpty())
		{
			try
			{
				Thread.sleep(100);
			}
			catch (InterruptedException e)
			{
				e.printStackTrace();
			}
		}
	}

	public String receiveCommand(long timeout) throws CommandQueueException
	{
		if (name == null || !commandQueues.containsKey(name))
		{
			throw new CommandQueueException(
					"Sorry.  Invalid call to 'receiveCommand' method.  Queue is not open!");
		}

		try
		{
			return queue.take();
		}
		catch (InterruptedException e)
		{
			throw new CommandQueueException(
					"Error taking command message from command queue.", e);
		}
	}

	public void close() throws CommandQueueException
	{
		commandQueues.remove(name);
	}

	/**
	 * Get the command queue based on the name supplied in the configuration
	 * ("command-queue-name").
	 * 
	 * @param name
	 *            The name of the queue ala the "command-queue-name" attribute
	 *            on the queue configuration.
	 * @return The MockCommandQueue instance, or null if no such queue exists.
	 */
	public static InMemoryCommandQueue getQueue(String name)
	{
		return commandQueues.get(name);
	}
}
