/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/

package org.jboss.internal.soa.esb.dependencies;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;

import org.apache.juddi.v3.client.config.UDDIClerkManager;
import org.apache.juddi.v3.client.config.UDDIClientContainer;
import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.system.ServiceMBeanSupport;

public class JuddiClientService extends ServiceMBeanSupport implements JuddiClientServiceMBean
{
    private static final Logger logger = Logger.getLogger(JuddiClientService.class);
    
    private static final String DEFAULT_UDDI_CLIENT_FILE = "esb.juddi.client.xml";
    private static final String UDDI_CLIENT_RESOURCE_FILE = "uddi.xml";
    
    private String propertiesResourceFile ;
    private String propertiesFileDir;
    private String nodeName;
    private String managerName;
    private UDDIClerkManager manager;

    public String getPropertiesResourceFile()
    {
       return propertiesResourceFile;
    }

    public void setPropertiesResourceFile(String propertiesResourceFile)
    {
       this.propertiesResourceFile = propertiesResourceFile;
    }

    public String getNodeName() {
       return this.nodeName;
    }

    public void setNodeName(String nodeName) {
      this.nodeName = nodeName;
    }

    public String getManagerName() {
      return this.managerName;
    }

    public void setManagerName(String managerName) {
      this.managerName = managerName;
    }

    protected void startService() throws Exception
    {
         logger.info("starting juddi client service");
         final String propertiesResourceFileVal ;
         if (propertiesResourceFile == null)
         {
             propertiesResourceFileVal = DEFAULT_UDDI_CLIENT_FILE ;
         }
         else
         {
             propertiesResourceFileVal = propertiesResourceFile ;
         }
         final File baseFile = new File(propertiesResourceFileVal) ;
         
         if (propertiesFileDir == null)
         {
             final String errorMsg = String.format("No property named '%s' was configured in jbossesb.sar/META-INF/jboss-service.xml for %s", "propertiesFileDir", getClass().getName());
             throw new IllegalArgumentException(errorMsg);
         }
         
         File dataDir = new File(propertiesFileDir);
         if (!dataDir.exists())
         {
             final String errorMsg = String.format("The directory configured for %s='%s' does not exist.", "warFilesDir", dataDir);
             throw new FileNotFoundException(errorMsg);
         }
         final File uddiClientResourceFile = new File(dataDir, UDDI_CLIENT_RESOURCE_FILE) ;

         final InputStream xmlPropertyIS ;
         if (baseFile.isAbsolute())
         {
             xmlPropertyIS = new FileInputStream(baseFile);
         }
         else
         {
             final URL url = Thread.currentThread().getContextClassLoader().getResource(propertiesResourceFileVal);
             xmlPropertyIS = url.openStream() ;
         }
         try
         {
             final OutputStream os = new FileOutputStream(uddiClientResourceFile) ;
             try
             {
                 XMLHelper.replaceSystemProperties(XMLHelper.getXMLStreamReader(xmlPropertyIS),
                     XMLHelper.getXMLStreamWriter(os)) ;
             }
             finally
             {
                 os.close() ;
             }
         } finally {
             xmlPropertyIS.close() ;
         }

         System.setProperty("javax.xml.registry.ConnectionFactoryClass","org.apache.ws.scout.registry.ConnectionFactoryImpl");
         manager = new UDDIClerkManager(uddiClientResourceFile.getAbsoluteFile().toURI().toString());
         // addClerkManager must precede manager.start() to avoid a race in jUDDI, do not change this order.
         UDDIClientContainer.addClerkManager(manager) ;
         manager.start();
	 this.managerName = this.manager.getName();
         System.setProperty("org.apache.juddi.v3.client.manager.name", this.manager.getName());
         System.setProperty("org.apache.juddi.v3.client.node.name", this.nodeName);
         super.startService();
     }


     protected void stopService() throws Exception
     {
         if (manager != null)
         {
             manager.stop() ;
         }
         super.stopService();
     }

     public void setPropertiesFileDir(final String directory)
     {
         this.propertiesFileDir = directory;
     }
}
