/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.registry.client;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Properties;
import java.util.concurrent.atomic.AtomicReference;

import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.configuration.ConfigurationException;
import org.apache.juddi.v3.client.config.UDDIClerkManager;
import org.apache.juddi.v3.client.config.UDDIClientContainer;
import org.apache.juddi.v3.client.config.UDDINode;
import org.apache.juddi.v3.client.transport.Transport;
import org.apache.juddi.v3.client.transport.TransportException;
import org.apache.juddi.v3_service.JUDDIApiPortType;
import org.apache.log4j.Logger;
import org.uddi.v3_service.UDDICustodyTransferPortType;
import org.uddi.v3_service.UDDIInquiryPortType;
import org.uddi.v3_service.UDDIPublicationPortType;
import org.uddi.v3_service.UDDISecurityPortType;
import org.uddi.v3_service.UDDISubscriptionListenerPortType;
import org.uddi.v3_service.UDDISubscriptionPortType;

public class JuddiRMITransport extends Transport
{
    /**
     * The logger for this transport.
     */
    private static final Logger logger = Logger.getLogger(JuddiRMITransport.class) ;
    /**
     * The manager name to use for this transport.
     */
    private final String managerName ;
    /**
     * The node name to use for this transport.
     */
    private final String nodeName ;
    /**
     * The JNDI environment properties used for the initial context.
     */
    private final Properties jndiEnv ;
    
    /**
     * The api service instance.
     */
    private final AtomicReference<JUDDIApiPortType> apiService = new AtomicReference<JUDDIApiPortType>() ;
    /**
     * The custody transfer service instance.
     */
    private final AtomicReference<UDDICustodyTransferPortType> custodyTransferService = new AtomicReference<UDDICustodyTransferPortType>() ;
    /**
     * The inquiry service instance.
     */
    private final AtomicReference<UDDIInquiryPortType> inquiryService = new AtomicReference<UDDIInquiryPortType>() ;
    /**
     * The publish service instance.
     */
    private final AtomicReference<UDDIPublicationPortType> publishService = new AtomicReference<UDDIPublicationPortType>() ;
    /**
     * The security service instance.
     */
    private final AtomicReference<UDDISecurityPortType> securityService = new AtomicReference<UDDISecurityPortType>() ;
    /**
     * The subscription listener service.
     */
    private final AtomicReference<UDDISubscriptionListenerPortType> subscriptionListenerService = new AtomicReference<UDDISubscriptionListenerPortType>() ;
    /**
     * The subscription service.
     */
    private final AtomicReference<UDDISubscriptionPortType> subscriptionService = new AtomicReference<UDDISubscriptionPortType>() ;

    /**
     * Default constructor.
     */
    public JuddiRMITransport()
        throws ConfigurationException
    {
        this(null) ;
    }

    /**
     * Construct a transport for the specified node.
     * @param nodeName The node name.
     */
    public JuddiRMITransport(final String nodeName)
        throws ConfigurationException
    {
        this(null, nodeName) ;
    }

    /**
     * Construct a transport for the specified node.
     * @param managerName The manager name.
     * @param nodeName The node name.
     */
    public JuddiRMITransport(final String managerName, final String nodeName)
        throws ConfigurationException
    {
        this.managerName = managerName ;
        this.nodeName = (nodeName == null ? Transport.DEFAULT_NODE_NAME : nodeName) ;
        jndiEnv = createEnvironment(managerName, nodeName) ;
    }

    @Override
    public JUDDIApiPortType getJUDDIApiService(final String endpointURL)
        throws TransportException
    {
        final JUDDIApiPortType currentApiService = apiService.get() ;
        if (currentApiService != null)
        {
            return currentApiService ;
        }
        final String apiURL = getUDDINode(managerName, nodeName).getJuddiApiUrl() ;
        final JUDDIApiPortType newApiService = lookupService(jndiEnv, apiURL) ;
        if (apiService.compareAndSet(null, newApiService))
        {
            return newApiService ;
        }
        else
        {
            return apiService.get() ;
        }
    }

    @Override
    public UDDICustodyTransferPortType getUDDICustodyTransferService(final String endpointURL)
        throws TransportException
    {
        final UDDICustodyTransferPortType currentCustodyTransferService = custodyTransferService.get() ;
        if (currentCustodyTransferService != null)
        {
            return currentCustodyTransferService ;
        }
        final String custodyTransferURL = getUDDINode(managerName, nodeName).getCustodyTransferUrl() ;
        final UDDICustodyTransferPortType newCustodyTransferService = lookupService(jndiEnv, custodyTransferURL) ;
        if (custodyTransferService.compareAndSet(null, newCustodyTransferService))
        {
            return newCustodyTransferService ;
        }
        else
        {
            return custodyTransferService.get() ;
        }
    }

    @Override
    public UDDIInquiryPortType getUDDIInquiryService(final String endpointURL)
        throws TransportException
    {
        final UDDIInquiryPortType currentInquiryService = inquiryService.get() ;
        if (currentInquiryService != null)
        {
            return currentInquiryService ;
        }
        final String inquiryURL = getUDDINode(managerName, nodeName).getInquiryUrl() ;
        final UDDIInquiryPortType newInquiryService = lookupService(jndiEnv, inquiryURL) ;
        if (inquiryService.compareAndSet(null, newInquiryService))
        {
            return newInquiryService ;
        }
        else
        {
            return inquiryService.get() ;
        }
    }

    @Override
    public UDDIPublicationPortType getUDDIPublishService(final String endpointURL)
            throws TransportException
    {
        final UDDIPublicationPortType currentPublishService = publishService.get() ;
        if (currentPublishService != null)
        {
            return currentPublishService ;
        }
        final String publishURL = getUDDINode(managerName, nodeName).getPublishUrl() ;
        final UDDIPublicationPortType newPublishService = lookupService(jndiEnv, publishURL) ;
        if (publishService.compareAndSet(null, newPublishService))
        {
            return newPublishService ;
        }
        else
        {
            return publishService.get() ;
        }
    }

    @Override
    public UDDISecurityPortType getUDDISecurityService(final String endpointURL)
            throws TransportException
    {
        final UDDISecurityPortType currentSecurityService = securityService.get() ;
        if (currentSecurityService != null)
        {
            return currentSecurityService ;
        }
        final String securityURL = getUDDINode(managerName, nodeName).getSecurityUrl() ;
        final UDDISecurityPortType newSecurityService = lookupService(jndiEnv, securityURL) ;
        if (securityService.compareAndSet(null, newSecurityService))
        {
            return newSecurityService ;
        }
        else
        {
            return securityService.get() ;
        }
    }

    @Override
    public UDDISubscriptionListenerPortType getUDDISubscriptionListenerService(final String endpointURL)
        throws TransportException
    {
        final UDDISubscriptionListenerPortType currentSubscriptionListenerService = subscriptionListenerService.get() ;
        if (currentSubscriptionListenerService != null)
        {
            return currentSubscriptionListenerService ;
        }
        final String subscriptionListenerURL = getUDDINode(managerName, nodeName).getSubscriptionListenerUrl() ;
        final UDDISubscriptionListenerPortType newSubscriptionListenerService = lookupService(jndiEnv, subscriptionListenerURL) ;
        if (subscriptionListenerService.compareAndSet(null, newSubscriptionListenerService))
        {
            return newSubscriptionListenerService ;
        }
        else
        {
            return subscriptionListenerService.get() ;
        }
    }

    @Override
    public UDDISubscriptionPortType getUDDISubscriptionService(final String endpointURL)
        throws TransportException
    {
        final UDDISubscriptionPortType currentSubscriptionService = subscriptionService.get() ;
        if (currentSubscriptionService != null)
        {
            return currentSubscriptionService ;
        }
        final String subscriptionURL = getUDDINode(managerName, nodeName).getSubscriptionUrl() ;
        final UDDISubscriptionPortType newSubscriptionService = lookupService(jndiEnv, subscriptionURL) ;
        if (subscriptionService.compareAndSet(null, newSubscriptionService))
        {
            return newSubscriptionService ;
        }
        else
        {
            return subscriptionService.get() ;
        }
    }

    private static Properties createEnvironment(final String managerName, final String nodeName)
        throws ConfigurationException
    {
        final Properties jndiEnv = new Properties() ;
        final UDDIClerkManager manager = UDDIClientContainer.getUDDIClerkManager(managerName) ;
        final UDDINode uddiNode = manager.getClientConfig().getUDDINode(nodeName) ;
        final String initialContextFactory = uddiNode.getFactoryInitial() ;
        final String providerURL = uddiNode.getFactoryNamingProvider() ;
        final String urlPkgPrefixes = uddiNode.getFactoryURLPkgs() ;
        
        if (initialContextFactory != null)
        {
            jndiEnv.setProperty(Context.INITIAL_CONTEXT_FACTORY, initialContextFactory) ;
        }
        if (providerURL != null)
        {
            jndiEnv.setProperty(Context.PROVIDER_URL, providerURL) ;
        }
        if (urlPkgPrefixes != null)
        {
            jndiEnv.setProperty(Context.URL_PKG_PREFIXES, urlPkgPrefixes) ;
        }
        return jndiEnv ;
    }

    @SuppressWarnings("unchecked")
    private static <T> T lookupService(final Properties env, final String url)
        throws TransportException
    {
        final URI uri ;
        try
        {
            uri = new URI(url) ;
        }
        catch (final URISyntaxException urise)
        {
            throw new TransportException("Failed to parse service URL", urise) ;
        }
        final String serviceLocation = uri.getPath() ;
        if (logger.isDebugEnabled())
        {
            logger.debug("Looking up service=" + serviceLocation) ;
        }
        final Object service ;
        try
        {
            final InitialContext context = new InitialContext(env) ;
            service = context.lookup(serviceLocation) ;
        }
        catch (final NamingException ne)
        {
            throw new TransportException("Failure during JNDI lookup of service: " + serviceLocation, ne) ;
        }
        return (T)service ;
    }
    
    private static UDDINode getUDDINode(final String managerName, final String nodeName)
        throws TransportException
    {
        try
        {
            final UDDIClerkManager manager = UDDIClientContainer.getUDDIClerkManager(managerName) ;
            return manager.getClientConfig().getUDDINode(nodeName) ;
        }
        catch (final ConfigurationException ce)
        {
            throw new TransportException("Failed to retrieve UDDI node: " + nodeName, ce) ;
        }
    }
}
